/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.service.writer.mail;

import java.util.*;
import java.io.*;
import javax.activation.*;
import javax.mail.*;
import javax.mail.internet.*;
import javax.naming.NamingException;

import jp.ossc.nimbus.core.*;
import jp.ossc.nimbus.io.CSVReader;
import jp.ossc.nimbus.service.distribute.KeepAliveCheckerSelector;
import jp.ossc.nimbus.service.distribute.SmtpKeepAliveChecker;
import jp.ossc.nimbus.service.jndi.JndiFinder;
import jp.ossc.nimbus.service.writer.*;

/**
 * [MT[rXB<p>
 * Java Mailł́ASession̎擾yѐݒAMessage̐yѐݒASMTPT[oւ̑M̋@\B<br>
 * ̃T[rXł́A̋@\mvO~OōsłB<br>
 * <p>
 * Java MailłSession̎擾ɂ́AQނ̕@B<br>
 * {@link Session#getInstance(Properties, Authenticator)}ŁACӂ̃ZbVw肵Session擾@B<br>
 * AvP[VT[oŐݒ肵SessionJNDIoRŎ擾@B<br>
 * ̃T[rXł́ȂoT|[g܂B<br>
 * ܂AO҂̕@̏ꍇ́A{@link #setSessionProperties(Properties)}ŁAݒ肵ZbVꗥɐݒ肷B<br>
 * AAȉ̃ZbV́Aꗥw肪Ȃꍇ́Aꂼȉ̕@Őݒ肳B<br>
 * <table border="1">
 *   <tr bgcolor="#CCCCFF"><th>ZbV</th><th>ftHg̐ݒ@</th></tr>
 *   <tr><td>mail.smtp.host</td><td>{@link #setSmtpKeepAliveCheckerSelectorServiceName(ServiceName)}w肳Ăꍇ́Aw肳ꂽ{@link KeepAliveCheckerSelector}ɂđIꂽ{@link SmtpKeepAliveChecker}{@link SmtpKeepAliveChecker#getHostIp()}Ŏ擾lB<br>łȂꍇ́A{@link #setSmtpHostName(String)}Ŏw肳ꂽlB</td></tr>
 *   <tr><td>mail.smtp.port</td><td>{@link #setSmtpKeepAliveCheckerSelectorServiceName(ServiceName)}w肳Ăꍇ́Aw肳ꂽ{@link KeepAliveCheckerSelector}ɂđIꂽ{@link SmtpKeepAliveChecker}{@link SmtpKeepAliveChecker#getHostPort()}Ŏ擾lB<br>łȂꍇ́A{@link #setSmtpPort(int)}Ŏw肳ꂽlB</td></tr>
 *   <tr><td>mail.transport.protocol</td><td>smtp</td></tr>
 *   <tr><td>mail.smtp.from</td><td>{@link #setEnvelopeFromAddressKey(String)}Ŏw肳ꂽL[ŁA{@link #write(WritableRecord)}̈{@link WritableRecord}擾lB<br>w肳ĂȂꍇ́A{@link #setEnvelopeFromAddress(String)}Ŏw肳ꂽlB</td></tr>
 * </table>
 * <p>
 * Message̐́Ajavax.mail.MimeMessage𐶐܂B<br>
 * MimeMessageւ̊eݒ́ÃT[rX̊eŐݒ\łB<br>
 * eɁAQނ̐ݒ@pӂĂB<br>
 * {@link #write(WritableRecord)}̈{@link WritableRecord}擾邽߂̃L[w肷@ƁAl̂̂ݒ肷@łB<br>
 * O҂̏ꍇ́AMessageWriterĂԃNCAgɂāACӂɎw肷鎖\ŁA҂̕@́ÃT[rX̐ݒňꗥlݒ肷鎖\łB<br>
 * 鑮ɑ΂āAL̂Qނ̐ݒ@Ƃݒ肳Ăꍇɂ́AO҂̕LƂȂB<br>
 * ܂A{@link Message#setSentDate(Date)}́AIɑM̎ݒ肳B<br>
 * <p>
 * SMTPT[oւ̑M@\ƂẮASMTPT[oɑ΂ă[hoXȂ瑗M@\ƁAMsɁÅԊuāAgC@\B<br>
 *
 * @author M.Takata
 */
public class MailMessageWriterService extends ServiceBase
 implements MessageWriter, MailMessageWriterServiceMBean{
    
    private static final long serialVersionUID = 8479884337523286206L;
    
    private static final String SESSION_PROPERTY_NAME_HOST = "mail.smtp.host";
    private static final String SESSION_PROPERTY_NAME_PORT = "mail.smtp.port";
    private static final String SESSION_PROPERTY_NAME_TRANSPORT_PROTOCOL = "mail.transport.protocol";
    private static final String SESSION_PROPERTY_VALUE_TRANSPORT_PROTOCOL = "smtp";
    private static final String SESSION_PROPERTY_NAME_FROM = "mail.smtp.from";
    
    private Properties sessionProperties;
    
    private ServiceName authenticatorServiceName;
    private Authenticator authenticator;
    
    private Properties headers;
    private String[] headerKeys;
    
    private String envelopeFromAddressKey;
    private String envelopeFromAddress;
    private boolean isEnvelopeFromAddressValidate;
    
    private String fromAddressKey;
    private String fromAddress;
    private String fromPersonalKey;
    private String fromPersonal;
    private String fromPersonalEncodingKey;
    private String fromPersonalEncoding;
    private boolean isFromAddressValidate;
    
    private String senderAddressKey;
    private String senderAddress;
    private String senderPersonalKey;
    private String senderPersonal;
    private String senderPersonalEncodingKey;
    private String senderPersonalEncoding;
    private boolean isSenderAddressValidate;
    
    private String toAddressKey;
    private String[] toAddress;
    private String toPersonalKey;
    private String[] toPersonals;
    private String toPersonalEncodingKey;
    private String[] toPersonalEncodings;
    private String toPersonalEncoding;
    private boolean isToAddressValidate;
    
    private String ccAddressKey;
    private String[] ccAddress;
    private String ccPersonalKey;
    private String[] ccPersonals;
    private String ccPersonalEncodingKey;
    private String[] ccPersonalEncodings;
    private String ccPersonalEncoding;
    private boolean isCcAddressValidate;
    
    private String bccAddressKey;
    private String[] bccAddress;
    private String bccPersonalKey;
    private String[] bccPersonals;
    private String bccPersonalEncodingKey;
    private String[] bccPersonalEncodings;
    private String bccPersonalEncoding;
    private boolean isBccAddressValidate;
    
    private String replyToAddressKey;
    private String[] replyToAddress;
    private String replyToPersonalKey;
    private String[] replyToPersonals;
    private String replyToPersonalEncodingKey;
    private String[] replyToPersonalEncodings;
    private String replyToPersonalEncoding;
    private boolean isReplyToAddressValidate;
    
    private String subjectKey;
    private String subject;
    private String subjectEncodingKey;
    private String subjectEncoding;
    
    private String contentIDKey;
    private String contentID;
    
    private String contentLanguageKey;
    private String[] contentLanguage;
    
    private String contentMD5Key;
    private String contentMD5;
    
    private String descriptionKey;
    private String description;
    
    private String descriptionEncodingKey;
    private String descriptionEncoding;
    
    private String dispositionKey;
    private String disposition;
    
    private String filePartKey;
    private String fileCharset = MimeUtility.getDefaultJavaCharset();
    private String fileCharsetKey;
    private String fileLanguage;
    private String fileLanguageKey;
    
    private String bodyText;
    private String bodyIndexKey;
    private int bodyIndex = -1;
    private String bodyEncoding;
    
    private String smtpHostName;
    private int smtpPort = 25;
    private ServiceName smtpKeepAliveCheckerSelectorServiceName;
    private KeepAliveCheckerSelector smtpKeepAliveCheckerSelector;
    
    private int retryCount = -1;
    private long retryInterval = -1;
    
    private ServiceName jndiFinderServiceName;
    private JndiFinder jndiFinder;
    private String mailSessionJndiName = DEFAULT_MAIL_SESSION_JNDI_NAME;
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setSessionProperties(Properties prop){
        sessionProperties = prop;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public Properties getSessionProperties(){
        return sessionProperties;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setAuthenticatorServiceName(ServiceName name){
        authenticatorServiceName = name;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public ServiceName getAuthenticatorServiceName(){
        return authenticatorServiceName;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setHeaders(Properties prop){
        headers = prop;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public Properties getHeaders(){
        return headers;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setHeaderKeys(String[] keys){
        headerKeys = keys;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String[] getHeaderKeys(){
        return headerKeys;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setEnvelopeFromAddressKey(String key){
        envelopeFromAddressKey = key;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String getEnvelopeFromAddressKey(){
        return envelopeFromAddressKey;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setEnvelopeFromAddress(String address){
        envelopeFromAddress = address;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String getEnvelopeFromAddress(){
        return envelopeFromAddress;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setEnvelopeFromAddressValidate(boolean isValidate){
        isEnvelopeFromAddressValidate = isValidate;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public boolean isEnvelopeFromAddressValidate(){
        return isEnvelopeFromAddressValidate;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setFromAddressKey(String key){
        fromAddressKey = key;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String getFromAddressKey(){
        return fromAddressKey;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setFromAddress(String address){
        fromAddress = address;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String getFromAddress(){
        return fromAddress;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setFromPersonalKey(String key){
        fromPersonalKey = key;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String getFromPersonalKey(){
        return fromPersonalKey;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setFromPersonal(String personal){
        fromPersonal = personal;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String getFromPersonal(){
        return fromPersonal;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setFromPersonalEncodingKey(String key){
        fromPersonalEncodingKey = key;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String getFromPersonalEncodingKey(){
        return fromPersonalEncodingKey;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setFromPersonalEncoding(String encoding){
        fromPersonalEncoding = encoding;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String getFromPersonalEncoding(){
        return fromPersonalEncoding;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setFromAddressValidate(boolean isValidate){
        isFromAddressValidate = isValidate;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public boolean isFromAddressValidate(){
        return isFromAddressValidate;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setSenderAddressKey(String key){
        senderAddressKey = key;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String getSenderAddressKey(){
        return senderAddressKey;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setSenderAddress(String address){
        senderAddress = address;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String getSenderAddress(){
        return senderAddress;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setSenderPersonalKey(String key){
        senderPersonalKey = key;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String getSenderPersonalKey(){
        return senderPersonalKey;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setSenderPersonal(String personal){
        senderPersonal = personal;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String getSenderPersonal(){
        return senderPersonal;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setSenderPersonalEncodingKey(String key){
        senderPersonalEncodingKey = key;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String getSenderPersonalEncodingKey(){
        return senderPersonalEncodingKey;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setSenderPersonalEncoding(String encoding){
        senderPersonalEncoding = encoding;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String getSenderPersonalEncoding(){
        return senderPersonalEncoding;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setSenderAddressValidate(boolean isValidate){
        isSenderAddressValidate = isValidate;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public boolean isSenderAddressValidate(){
        return isSenderAddressValidate;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setToAddressKey(String key){
        toAddressKey = key;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String getToAddressKey(){
        return toAddressKey;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setToAddress(String[] address){
        toAddress = address;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String[] getToAddress(){
        return toAddress;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setToPersonalKey(String key){
        toPersonalKey = key;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String getToPersonalKey(){
        return toPersonalKey;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setToPersonals(String[] personal){
        toPersonals = personal;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String[] getToPersonals(){
        return toPersonals;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setToPersonalEncodingKey(String key){
        toPersonalEncodingKey = key;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String getToPersonalEncodingKey(){
        return toPersonalEncodingKey;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setToPersonalEncodings(String[] encoding){
        toPersonalEncodings = encoding;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String[] getToPersonalEncodings(){
        return toPersonalEncodings;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setToPersonalEncoding(String encoding){
        toPersonalEncoding = encoding;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String getToPersonalEncoding(){
        return toPersonalEncoding;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setToAddressValidate(boolean isValidate){
        isToAddressValidate = isValidate;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public boolean isToAddressValidate(){
        return isToAddressValidate;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setCcAddressKey(String key){
        ccAddressKey = key;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String getCcAddressKey(){
        return ccAddressKey;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setCcAddress(String[] address){
        ccAddress = address;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String[] getCcAddress(){
        return ccAddress;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setCcPersonalKey(String key){
        ccPersonalKey = key;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String getCcPersonalKey(){
        return ccPersonalKey;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setCcPersonals(String[] personal){
        ccPersonals = personal;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String[] getCcPersonals(){
        return ccPersonals;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setCcPersonalEncodingKey(String key){
        ccPersonalEncodingKey = key;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String getCcPersonalEncodingKey(){
        return ccPersonalEncodingKey;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setCcPersonalEncodings(String[] encoding){
        ccPersonalEncodings = encoding;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String[] getCcPersonalEncodings(){
        return ccPersonalEncodings;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setCcPersonalEncoding(String encoding){
        ccPersonalEncoding = encoding;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String getCcPersonalEncoding(){
        return ccPersonalEncoding;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setCcAddressValidate(boolean isValidate){
        isCcAddressValidate = isValidate;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public boolean isCcAddressValidate(){
        return isCcAddressValidate;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setBccAddressKey(String key){
        bccAddressKey = key;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String getBccAddressKey(){
        return bccAddressKey;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setBccAddress(String[] address){
        bccAddress = address;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String[] getBccAddress(){
        return bccAddress;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setBccPersonalKey(String key){
        bccPersonalKey = key;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String getBccPersonalKey(){
        return bccPersonalKey;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setBccPersonals(String[] personal){
        bccPersonals = personal;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String[] getBccPersonals(){
        return bccPersonals;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setBccPersonalEncodingKey(String key){
        bccPersonalEncodingKey = key;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String getBccPersonalEncodingKey(){
        return bccPersonalEncodingKey;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setBccPersonalEncodings(String[] encoding){
        bccPersonalEncodings = encoding;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String[] getBccPersonalEncodings(){
        return bccPersonalEncodings;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setBccPersonalEncoding(String encoding){
        bccPersonalEncoding = encoding;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String getBccPersonalEncoding(){
        return bccPersonalEncoding;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setBccAddressValidate(boolean isValidate){
        isBccAddressValidate = isValidate;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public boolean isBccAddressValidate(){
        return isBccAddressValidate;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setReplyToAddressKey(String key){
        replyToAddressKey = key;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String getReplyToAddressKey(){
        return replyToAddressKey;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setReplyToAddress(String[] address){
        replyToAddress = address;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String[] getReplyToAddress(){
        return replyToAddress;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setReplyToPersonalKey(String key){
        replyToPersonalKey = key;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String getReplyToPersonalKey(){
        return replyToPersonalKey;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setReplyToPersonals(String[] personal){
        replyToPersonals = personal;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String[] getReplyToPersonals(){
        return replyToPersonals;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setReplyToPersonalEncodingKey(String key){
        replyToPersonalEncodingKey = key;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String getReplyToPersonalEncodingKey(){
        return replyToPersonalEncodingKey;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setReplyToPersonalEncodings(String[] encoding){
        replyToPersonalEncodings = encoding;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String[] getReplyToPersonalEncodings(){
        return replyToPersonalEncodings;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setReplyToPersonalEncoding(String encoding){
        replyToPersonalEncoding = encoding;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String getReplyToPersonalEncoding(){
        return replyToPersonalEncoding;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setReplyToAddressValidate(boolean isValidate){
        isReplyToAddressValidate = isValidate;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public boolean isReplyToAddressValidate(){
        return isReplyToAddressValidate;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setSubjectKey(String key){
        subjectKey = key;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String getSubjectKey(){
        return subjectKey;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setSubject(String subject){
        this.subject = subject;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String getSubject(){
        return subject;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setSubjectEncodingKey(String key){
        subjectEncodingKey = key;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String getSubjectEncodingKey(){
        return subjectEncodingKey;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setSubjectEncoding(String encoding){
        subjectEncoding = encoding;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String getSubjectEncoding(){
        return subjectEncoding;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setContentIDKey(String key){
        contentIDKey = key;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String getContentIDKey(){
        return contentIDKey;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setContentID(String id){
        this.contentID = id;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String getContentID(){
        return contentID;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setContentLanguageKey(String key){
        contentLanguageKey = key;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String getContentLanguageKey(){
        return contentLanguageKey;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setContentLanguage(String[] lang){
        contentLanguage = lang;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String[] getContentLanguage(){
        return contentLanguage;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setContentMD5Key(String key){
        contentMD5Key = key;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String getContentMD5Key(){
        return contentMD5Key;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setContentMD5(String val){
        contentMD5 = val;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String getContentMD5(){
        return contentMD5;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setDescriptionKey(String key){
        descriptionKey = key;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String getDescriptionKey(){
        return descriptionKey;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setDescription(String val){
        description = val;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String getDescription(){
        return description;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setDescriptionEncodingKey(String key){
        descriptionEncodingKey = key;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String getDescriptionEncodingKey(){
        return descriptionEncodingKey;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setDescriptionEncoding(String encoding){
        descriptionEncoding = encoding;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String getDescriptionEncoding(){
        return descriptionEncoding;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setDispositionKey(String key){
        dispositionKey = key;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String getDispositionKey(){
        return dispositionKey;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setDisposition(String val){
        disposition = val;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String getDisposition(){
        return disposition;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setFilePartKey(String key){
        filePartKey = key;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String getFilePartKey(){
        return filePartKey;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setFileCharsetKey(String key){
        fileCharsetKey = key;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String getFileCharsetKey(){
        return fileCharsetKey;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setFileCharset(String charset){
        fileCharset = charset;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String getFileCharset(){
        return fileCharset;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setFileLanguageKey(String key){
        fileLanguageKey = key;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String getFileLanguageKey(){
        return fileLanguageKey;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setFileLanguage(String lang){
        fileLanguage = lang;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String getFileLanguage(){
        return fileLanguage;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setBodyIndex(int index){
        bodyIndex = index;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public int getBodyIndex(){
        return bodyIndex;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setBodyIndexKey(String key){
        bodyIndexKey = key;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String getBodyIndexKey(){
        return bodyIndexKey;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setBodyText(String text){
        bodyText = text;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String getBodyText(){
        return bodyText;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setBodyEncoding(String encoding){
        bodyEncoding = encoding;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String getBodyEncoding(){
        return bodyEncoding;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setSmtpHostName(String name){
        smtpHostName = name;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String getSmtpHostName(){
        return smtpHostName;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setSmtpPort(int port){
        smtpPort = port;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public int getSmtpPort(){
        return smtpPort;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setRetryCount(int count){
        retryCount = count;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public int getRetryCount(){
        return retryCount;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setRetryInterval(long millis){
        retryInterval = millis;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public long getRetryInterval(){
        return retryInterval;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setSmtpKeepAliveCheckerSelectorServiceName(ServiceName name){
        smtpKeepAliveCheckerSelectorServiceName = name;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public ServiceName getSmtpKeepAliveCheckerSelectorServiceName(){
        return smtpKeepAliveCheckerSelectorServiceName;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setJndiFinderServiceName(ServiceName name){
        jndiFinderServiceName = name;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public ServiceName getJndiFinderServiceName(){
        return jndiFinderServiceName;
    }
    
    // MailMessageWriterServiceMBeanJavaDoc
    public void setMailSessionJndiName(String name){
        mailSessionJndiName = name;
    }
    // MailMessageWriterServiceMBeanJavaDoc
    public String getMailSessionJndiName(){
        return mailSessionJndiName;
    }
    
    /**
     * T[rX̊JnsB<p>
     *
     * @exception Exception T[rX̊JnɎsꍇ
     */
    public void startService() throws Exception{
        if(authenticatorServiceName != null){
            authenticator = (Authenticator)ServiceManagerFactory
                .getServiceObject(authenticatorServiceName);
        }
        
        if(fromAddressKey == null && fromAddress == null){
            throw new IllegalArgumentException("It is necessary to set any of FromAddressKey and FromAddress.");
        }
        if(toAddress != null && toAddress.length != 0){
            if(toPersonals != null && toPersonals.length != toAddress.length){
                throw new IllegalArgumentException("It is necessary to set toAddress and toPersonals to the same length.");
            }
            if(toPersonalEncodings != null && toPersonalEncodings.length != toAddress.length){
                throw new IllegalArgumentException("It is necessary to set toAddress and toPersonalEncodings to the same length.");
            }
        }
        if(ccAddress != null && ccAddress.length != 0){
            if(ccPersonals != null && ccPersonals.length != ccAddress.length){
                throw new IllegalArgumentException("It is necessary to set ccAddress and ccPersonals to the same length.");
            }
            if(ccPersonalEncodings != null && ccPersonalEncodings.length != ccAddress.length){
                throw new IllegalArgumentException("It is necessary to set ccAddress and ccPersonalEncodings to the same length.");
            }
        }
        if(bccAddress != null && bccAddress.length != 0){
            if(bccPersonals != null && bccPersonals.length != bccAddress.length){
                throw new IllegalArgumentException("It is necessary to set bccAddress and bccPersonals to the same length.");
            }
            if(bccPersonalEncodings != null && bccPersonalEncodings.length != bccAddress.length){
                throw new IllegalArgumentException("It is necessary to set bccAddress and bccPersonalEncodings to the same length.");
            }
        }
        
        if(smtpKeepAliveCheckerSelectorServiceName != null){
            smtpKeepAliveCheckerSelector
                 = (KeepAliveCheckerSelector)ServiceManagerFactory
                    .getServiceObject(smtpKeepAliveCheckerSelectorServiceName);
        }
        if(smtpHostName == null && smtpKeepAliveCheckerSelector == null){
            throw new IllegalArgumentException("It is necessary to set any of SmtpHostName and SmtpKeepAliveCheckerSelectorServiceName.");
        }
        
        if(jndiFinderServiceName != null){
            jndiFinder = (JndiFinder)ServiceManagerFactory
                .getServiceObject(jndiFinderServiceName);
        }
    }
    
    /**
     * w肳ꂽR[h[MB<p>
     *
     * @param rec [M郌R[h
     * @exception MessageWriteException [MɎsꍇ
     */
    public void write(WritableRecord rec) throws MessageWriteException{
        
        int maxCount = retryCount > 0 ? retryCount : 0;
        Exception exception = null;
        for(int count = 0; count <= maxCount; count++){
            try{
                sendMail(rec);
                return;
            }catch(NamingException e){
                exception = e;
            }catch(IllegalWriteException e){
                exception = e;
            }catch(MessagingException e){
                exception = e;
            }
            if(retryInterval > 0){
                try{
                    Thread.sleep(retryInterval);
                }catch(InterruptedException e){
                }
            }
        }
        throw new MessageWriteException(exception);
    }
    
    /**
     * w肳ꂽR[h[MB<p>
     *
     * @param rec [M郌R[h
     * @exception MessageWriteException javax.mail.Messageɐݒ肷lsȏꍇA܂SMTPT[ołꍇ
     * @exception IllegalWriteException javax.mail.Message̓ǂݎpɏ݂sꂽꍇ
     * @exception MessagingException [MɎsꍇ
     * @exception NamingException javax.mail.SessionJNDIlookup̂Ɏsꍇ
     */
    public void sendMail(WritableRecord rec)
     throws MessageWriteException, IllegalWriteException, MessagingException,
            NamingException{
        
        final Map<Object, WritableElement> elementMap = rec.getElementMap();
        
        // FromAhX쐬
        final InternetAddress fromInternetAddress = createAddress(
            elementMap,
            fromAddress,
            fromPersonal,
            fromPersonalEncoding,
            fromAddressKey,
            fromPersonalKey,
            fromPersonalEncodingKey,
            isFromAddressValidate
        );
        if(fromInternetAddress == null){
            throw new MessageWriteException("The From address is null.");
        }
        
        // SenderAhX쐬
        final InternetAddress senderInternetAddress = createAddress(
            elementMap,
            senderAddress,
            senderPersonal,
            senderPersonalEncoding,
            senderAddressKey,
            senderPersonalKey,
            senderPersonalEncodingKey,
            isSenderAddressValidate
        );
        
        // ToAhX쐬
        final InternetAddress[] toInternetAddress = createAddressArray(
            elementMap,
            toAddress,
            toPersonals,
            toPersonalEncodings,
            toPersonalEncoding,
            toAddressKey,
            toPersonalKey,
            toPersonalEncodingKey,
            isToAddressValidate
        );
        
        // CcAhX쐬
        final InternetAddress[] ccInternetAddress = createAddressArray(
            elementMap,
            ccAddress,
            ccPersonals,
            ccPersonalEncodings,
            ccPersonalEncoding,
            ccAddressKey,
            ccPersonalKey,
            ccPersonalEncodingKey,
            isCcAddressValidate
        );
        
        // BccAhX쐬
        final InternetAddress[] bccInternetAddress = createAddressArray(
            elementMap,
            bccAddress,
            bccPersonals,
            bccPersonalEncodings,
            bccPersonalEncoding,
            bccAddressKey,
            bccPersonalKey,
            bccPersonalEncodingKey,
            isBccAddressValidate
        );
        if((toInternetAddress == null || toInternetAddress.length == 0)
            && (ccInternetAddress == null || ccInternetAddress.length == 0)
            && (bccInternetAddress == null || bccInternetAddress.length == 0)
        ){
            throw new MessageWriteException("The destination address is null.");
        }
        
        // ReplyToAhX쐬
        final InternetAddress[] replyToInternetAddress = createAddressArray(
            elementMap,
            replyToAddress,
            replyToPersonals,
            replyToPersonalEncodings,
            replyToPersonalEncoding,
            replyToAddressKey,
            replyToPersonalKey,
            replyToPersonalEncodingKey,
            isReplyToAddressValidate
        );
        
        // {쐬
        String body = bodyText;
        final int bIndex = getIntValue(elementMap, bodyIndex, bodyIndexKey);
        if(bIndex >= 0){
            List<WritableElement> elements = rec.getElements();
            if(elements.size() > bIndex){
                final StringBuilder buf = new StringBuilder();
                for(int i = bIndex, imax = elements.size(); i < imax; i++){
                    WritableElement element = elements.get(i);
                    if(element != null){
                        buf.append(element);
                    }
                }
                body = buf.toString();
            }
        }
        Session session = null;
        if(jndiFinder != null){
            session = (Session)jndiFinder.lookup(mailSessionJndiName);
        }else{
            
            final Properties props = new Properties(System.getProperties());
            
            if(smtpKeepAliveCheckerSelector != null){
                final SmtpKeepAliveChecker checker
                     = (SmtpKeepAliveChecker)smtpKeepAliveCheckerSelector
                        .selectChecker(rec);
                if(checker == null){
                    throw new MessageWriteException("All smtp server is dead.");
                }
                // SMTPT[õzXgݒ肷
                props.setProperty(
                    SESSION_PROPERTY_NAME_HOST,
                    checker.getHostIp()
                );
                // SMTPT[õ|[gԍݒ肷
                props.setProperty(
                    SESSION_PROPERTY_NAME_PORT,
                    String.valueOf(checker.getHostPort())
                );
            }else{
                // SMTPT[õzXgݒ肷
                props.setProperty(SESSION_PROPERTY_NAME_HOST, smtpHostName);
                // SMTPT[õ|[gԍݒ肷
                props.setProperty(
                    SESSION_PROPERTY_NAME_PORT,
                    String.valueOf(smtpPort)
                );
            }
            
            // MvgRݒ肷
            props.setProperty(
                SESSION_PROPERTY_NAME_TRANSPORT_PROTOCOL,
                SESSION_PROPERTY_VALUE_TRANSPORT_PROTOCOL
            );
            
            // envelope-fromAhX쐬
            final InternetAddress envelopeFromInternetAddress = createAddress(
                elementMap,
                envelopeFromAddress,
                null,
                null,
                envelopeFromAddressKey,
                null,
                null,
                isEnvelopeFromAddressValidate
            );
            if(envelopeFromInternetAddress != null){
                // envelope-fromAhXݒ肷
                props.setProperty(
                    SESSION_PROPERTY_NAME_FROM,
                    envelopeFromInternetAddress.getAddress()
                );
            }
            
            if(sessionProperties != null){
                props.putAll(sessionProperties);
            }
            
            session = Session.getInstance(props, authenticator);
        }
        final MimeMessage message = new MimeMessage(session);
        
        // Content-IDݒ肷
        String val = getStringValue(elementMap, contentID, contentIDKey);
        if(val != null){
            message.setContentID(val);
        }
        
        // Content-Languageݒ肷
        String[] vals = getStringArrayValue(
            elementMap,
            contentLanguage,
            contentLanguageKey
        );
        if(vals != null){
            message.setContentLanguage(vals);
        }
        
        // Content-MD5ݒ肷
        val = getStringValue(elementMap, contentMD5, contentMD5Key);
        if(val != null){
            message.setContentMD5(val);
        }
        
        // Content-Descriptionݒ肷
        val = getStringValue(elementMap, description, descriptionKey);
        if(val != null){
            final String enc = getStringValue(
                elementMap,
                descriptionEncoding,
                descriptionEncodingKey
            );
            if(enc == null){
                message.setDescription(val);
            }else{
                message.setDescription(val, enc);
            }
        }
        
        // Content-Dispositionݒ肷
        val = getStringValue(elementMap, disposition, dispositionKey);
        if(val != null){
            message.setDisposition(val);
        }

        // FromAhXݒ肷
        message.setFrom(fromInternetAddress);
        
        // SenderAhXݒ肷
        if(senderInternetAddress != null){
            message.setSender(senderInternetAddress);
        }
        
        // ToAhXݒ肷
        if(toInternetAddress != null){
            for(int i = 0; i < toInternetAddress.length; i++){
                message.addRecipient(
                    Message.RecipientType.TO,
                    toInternetAddress[i]
                );
            }
        }
        
        // CcAhXݒ肷
        if(ccInternetAddress != null){
            for(int i = 0; i < ccInternetAddress.length; i++){
                message.addRecipient(
                    Message.RecipientType.CC,
                    ccInternetAddress[i]
                );
            }
        }
        
        // BccAhXݒ肷
        if(bccInternetAddress != null){
            for(int i = 0; i < bccInternetAddress.length; i++){
                message.addRecipient(
                    Message.RecipientType.BCC,
                    bccInternetAddress[i]
                );
            }
        }
        
        // ReplyToAhXݒ肷
        if(replyToInternetAddress != null){
            message.setReplyTo(replyToInternetAddress);
        }
        
        // 薼ݒ肷
        String enc =
            getStringValue(elementMap, subjectEncoding, subjectEncodingKey);
        if(enc != null){
            message.setSubject(
                getStringValue(elementMap, subject, subjectKey),
                enc
            );
        }else{
            message.setSubject(
                getStringValue(elementMap, subject, subjectKey)
            );
        }
        
        File[] files = null;
        if(filePartKey != null){
            files = getFileArrayValue(elementMap, filePartKey);
        }
        if(files != null && files.length != 0){
            
            final MimeMultipart mp = new MimeMultipart();
            
            // {ݒ肷
            final MimeBodyPart textPart = new MimeBodyPart();
            if(bodyEncoding == null){
                textPart.setText(body);
            }else{
                textPart.setText(body, bodyEncoding);
            }
            mp.addBodyPart(textPart);
            
            // Ytt@CYt
            for(int i = 0; i < files.length; i++){
                final MimeBodyPart filePart = new MimeBodyPart();
                filePart.setDataHandler(
                    new DataHandler(new FileDataSource(files[i]))
                );
                setFileName(
                    filePart,
                    files[i].getName(),
                    getStringValue(elementMap, fileCharset, fileCharsetKey),
                    getStringValue(elementMap, fileLanguage, fileLanguageKey)
                );
                mp.addBodyPart(filePart);
            }
            
            message.setContent(mp);
        }else{
            // {ݒ肷
            if(bodyEncoding == null){
                message.setText(body);
            }else{
                message.setText(body, bodyEncoding);
            }
        }
        
        // wb_ݒ肷
        if(headers != null){
            for(Map.Entry<Object, Object> entry : headers.entrySet()){
                message.setHeader((String)entry.getKey(), (String)entry.getValue());
            }
        }
        if(headerKeys != null){
            for(int i = 0; i < headerKeys.length; i++){
                final String header = getStringValue(
                    elementMap,
                    null,
                    headerKeys[i]
                );
                if(header != null){
                    message.setHeader(headerKeys[i], header);
                }
            }
        }

        // Mݒ肷
        message.setSentDate(new Date());
        
        // ݒLɂ
        message.saveChanges();
        
        // [𑗐M
        Transport.send(message);
    }
    
    private void setFileName(
        Part part,
        String filename,
        String charset,
        String lang
    ) throws MessagingException{
        
        ContentDisposition disposition;
        String[] strings = part.getHeader("Content-Disposition");
        if(strings == null || strings.length < 1){
            disposition = new ContentDisposition(Part.ATTACHMENT);
        }else{
            disposition = new ContentDisposition(strings[0]);
            disposition.getParameterList().remove("filename");
        }
        
        part.setHeader(
            "Content-Disposition",
            disposition.toString()
                + encodeParameter("filename", filename, charset, lang)
        );
        
        ContentType cType;
        strings = part.getHeader("Content-Type");
        if(strings == null || strings.length < 1){
            cType = new ContentType(part.getDataHandler().getContentType());
        }else{
            cType = new ContentType(strings[0]);
        }
        
        try{
            // I want to public the MimeUtility#doEncode()!!!
            String mimeString = MimeUtility.encodeWord(filename, charset, "B");
            // cut <CRLF>...
            StringBuilder sb = new StringBuilder();
            int i;
            while((i = mimeString.indexOf('\r')) != -1){
                sb.append(mimeString.substring(0, i));
                mimeString = mimeString.substring(i + 2);
            }
            sb.append(mimeString);
            
            cType.setParameter("name", new String(sb));
        }catch(UnsupportedEncodingException e){
            throw new MessagingException("Encoding error", e);
        }
        part.setHeader("Content-Type", cType.toString());
    }
    
    private String encodeParameter(
        String name,
        String value,
        String encoding,
        String lang
    ){
        StringBuilder result = new StringBuilder();
        StringBuilder encodedPart = new StringBuilder();
        
        boolean needWriteCES = !isAllAscii(value);
        boolean CESWasWritten = false;
        boolean encoded;
        boolean needFolding = false;
        int sequenceNo = 0;
        int column;
        
        while(value.length() > 0){
            // index of boundary of ascii/non ascii
            int lastIndex;
            boolean isAscii = value.charAt(0) < 0x80;
            for(lastIndex = 1; lastIndex < value.length(); lastIndex++){
                if(value.charAt(lastIndex) < 0x80){
                    if(!isAscii) break;
                }else{
                    if(isAscii) break;
                }
            }
            if(lastIndex != value.length()) needFolding = true;
            
            RETRY:
            while(true){
                encodedPart.delete(0, encodedPart.length());
                String target = value.substring(0, lastIndex);
                
                byte[] bytes;
                try{
                    if(isAscii){
                        bytes = target.getBytes("us-ascii");
                    }else{
                        bytes = target.getBytes(encoding);
                    }
                }catch(UnsupportedEncodingException e){
                    bytes = target.getBytes(); // use default encoding
                    encoding = MimeUtility.mimeCharset(
                        MimeUtility.getDefaultJavaCharset()
                    );
                }
                
                encoded = false;
                // It is not strict.
                column = name.length() + 7; // size of " " and "*nn*=" and ";"
                
                for(int i = 0; i < bytes.length; i++){
                    if(bytes[i] > ' ' && bytes[i] < 'z'
                         && HeaderTokenizer.MIME.indexOf((char)bytes[i]) < 0){
                        encodedPart.append((char)bytes[i]);
                        column++;
                    }else{
                        encoded = true;
                        encodedPart.append('%');
                        String hex  = Integer.toString(bytes[i] & 0xff, 16);
                        if(hex.length() == 1){
                            encodedPart.append('0');
                        }
                        encodedPart.append(hex);
                        column += 3;
                    }
                    if(column > 76){
                        needFolding = true;
                        lastIndex /= 2;
                        continue RETRY;
                    }
                }
                
                result.append(";\r\n ").append(name);
                if(needFolding){
                    result.append('*').append(sequenceNo);
                    sequenceNo++;
                }
                if(!CESWasWritten && needWriteCES){
                    result.append("*=");
                    CESWasWritten = true;
                    result.append(encoding).append('\'');
                    if(lang != null) result.append(lang);
                    result.append('\'');
                }else if(encoded){
                    result.append("*=");
                }else{
                    result.append('=');
                }
                result.append(new String(encodedPart));
                value = value.substring(lastIndex);
                break;
            }
        }
        return result.toString();
    }
    
    private boolean isAllAscii(String text){
        for(int i = 0; i < text.length(); i++){
            if(text.charAt(i) > 0x7f){ // non-ascii
                return false;
            }
        }
        return true;
    }
    
    @SuppressWarnings("unchecked")
    private String[] toStringArray(Object obj){
        if(obj == null){
            return null;
        }
        String[] result = null;
        if(obj instanceof String[]){
            result = (String[])obj;
        }else if(obj instanceof Collection){
            final Collection<Object> col = (Collection<Object>)obj;
            if(col.size() != 0){
                result = new String[col.size()];
                int index = 0;
                for(Object element : col){
                    result[index++] = element.toString();
                }
            }
        }else{
            result = CSVReader.toArray(
                obj.toString(),
                ',',
                '\\',
                '"',
                "",
                null,
                true,
                false,
                true,
                false
            );
        }
        return result;
    }
    
    private InternetAddress[] createAddressArray(
        Map<Object, WritableElement> elementMap,
        String[] defaultAddress,
        String[] defaultPersonals,
        String[] defaultPersonalEncodings,
        String defaultPersonalEncoding,
        String addressKey,
        String personalKey,
        String personalEncodingKey,
        boolean isAddressValidate
    ) throws MessageWriteException{
        final String[] address = getStringArrayValue(
            elementMap,
            defaultAddress,
            addressKey
        );
        if(address == null || address.length == 0){
            return null;
        }
        String[] personals = null;
        String[] personalEncodings = null;
        if(address == defaultAddress){
            personals = defaultPersonals;
            personalEncodings = defaultPersonalEncodings;
        }else{
            personals = getStringArrayValue(
                elementMap,
                null,
                personalKey
            );
            if(personals != null && personals.length != address.length){
                throw new MessageWriteException("It is necessary to set address and personals to the same length.");
            }
            
            personalEncodings = getStringArrayValue(
                elementMap,
                null,
                personalEncodingKey
            );
            if(personalEncodings != null && personalEncodings.length != address.length){
                throw new MessageWriteException("It is necessary to set address and personalEncodings to the same length.");
            }
        }
        final InternetAddress[] internetAddress
             = new InternetAddress[address.length];
        for(int i = 0; i < address.length; i++){
            internetAddress[i] = createAddress(
                address[i],
                personals == null ? null : personals[i],
                personalEncodings == null ? defaultPersonalEncoding : personalEncodings[i],
                isAddressValidate
            );
        }
        return internetAddress;
    }
    
    private InternetAddress createAddress(
        Map<Object, WritableElement> elementMap,
        String defaultAddress,
        String defaultPersonal,
        String defaultPersonalEncoding,
        String addressKey,
        String personalKey,
        String personalEncodingKey,
        boolean isAddressValidate
    ) throws MessageWriteException{
        final String address = getStringValue(
            elementMap,
            defaultAddress,
            addressKey
        );
        if(address == null){
            return null;
        }
        String personal = getStringValue(
            elementMap,
            defaultPersonal,
            personalKey
        );
        String personalEncoding = getStringValue(
            elementMap,
            defaultPersonalEncoding,
            personalEncodingKey
        );
        return createAddress(
            address,
            personal,
            personalEncoding,
            isAddressValidate
        );
    }
    
    private InternetAddress createAddress(
        String address,
        String personal,
        String personalEncoding,
        boolean isAddressValidate
    ) throws MessageWriteException{
        InternetAddress internetAddress = null;
        try{
            if(personal != null && personalEncoding != null){
                internetAddress = new InternetAddress(
                    address,
                    personal,
                    personalEncoding
                );
            }else if(personal != null){
                internetAddress = new InternetAddress(
                    address,
                    personal
                );
            }else{
                internetAddress = new InternetAddress(address);
            }
            if(isAddressValidate){
                internetAddress.validate();
            }
        }catch(UnsupportedEncodingException e){
            throw new MessageWriteException(e);
        }catch(AddressException e){
            throw new MessageWriteException(e);
        }
        return internetAddress;
    }
    
    private String[] getStringArrayValue(Map<Object, WritableElement> elementMap, String[] defaultVal, String key){
        String[] val = defaultVal;
        if(key != null){
            final WritableElement tmpElement
                 = (WritableElement)elementMap.get(key);
            if(tmpElement != null){
                Object tmpVal = tmpElement.toObject();
                if(tmpVal != null){
                    final String[] tmpArray = toStringArray(tmpVal);
                    if(tmpArray != null && tmpArray.length != 0){
                        val = tmpArray;
                    }
                }
            }
        }
        return val;
    }
    
    private String getStringValue(Map<Object, WritableElement> elementMap, String defaultVal, String key){
        Object val = getValue(elementMap, defaultVal, key);
        return val == null ? null : val.toString();
    }
    
    private int getIntValue(Map<Object, WritableElement> elementMap, int defaultVal, String key){
        Integer val = (Integer)getValue(elementMap, null, key);
        return val == null ? defaultVal : val.intValue();
    }
    
    private Object getValue(Map<Object, WritableElement> elementMap, Object defaultVal, String key){
        Object val = defaultVal;
        if(key != null){
            final WritableElement tmpElement
                 = (WritableElement)elementMap.get(key);
            if(tmpElement != null){
                Object tmpVal = tmpElement.toObject();
                if(tmpVal != null){
                    val = tmpVal;
                }
            }
        }
        return val;
    }
    
    @SuppressWarnings("unchecked")
    private File[] getFileArrayValue(Map<Object, WritableElement> elementMap, String key){
        File[] val = null;
        if(key != null){
            final WritableElement tmpElement
                 = (WritableElement)elementMap.get(key);
            if(tmpElement != null){
                Object tmpVal = tmpElement.toObject();
                if(tmpVal != null){
                    if(tmpVal instanceof File[]){
                        val = (File[])tmpVal;
                    }else if(tmpVal instanceof Collection){
                        final Collection<Object> col = (Collection<Object>)tmpVal;
                        if(col.size() != 0){
                            val = new File[col.size()];
                            int index = 0;
                            for(Object element : col){
                                val[index++] = (File)element;
                            }
                        }
                    }
                }
            }
        }
        return val;
    }
}
