/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.service.writer;

import java.lang.reflect.InvocationTargetException;
import java.util.*;
import java.io.Serializable;

import jp.ossc.nimbus.beans.*;
import jp.ossc.nimbus.core.*;
import jp.ossc.nimbus.lang.*;
import jp.ossc.nimbus.service.log.Logger;

/**
 * CӂBeanWritableRecordɕϊ{@link WritableRecordFactory}T[rXB<p>
 *
 * @author Y.Tokuda
 */
public class DefaultWritableRecordFactoryService extends ServiceBase
 implements WritableRecordFactory,
            DefaultWritableRecordFactoryServiceMBean,
            Serializable{
    
    private static final long serialVersionUID = 5249532509800152052L;
    
    private static final String ITERATE_SUFFIX = "*";
    
    /** tH[}bg` */
    protected String format;
    
    /** tH[}bg`񂩂琶ParsedElement^ArrayList */
    protected List<ParsedElement> parsedElements;
    
    /** WritableElement̎NXnbV */
    protected Map<String, Class<WritableElement>> writableElementClassMap;
    
    /** WritableElement̎T[rXnbV */
    protected Map<String, ServiceName> writableElementServiceNameMap;
    
    protected Properties formatKeyMappings;
    protected Map<String, Property> formatKeyPropertyMap;
    
    protected Map<String, Properties> iterateFormatKeyMappings;
    protected Map<String, Map<String, Property>> iterateformatKeyPropertyMap;
    
    protected Map<String, String> iterateFormats;
    protected Map<String, List<ParsedElement>> iterateFormatMap;
    
    // WritableRecordFactoryServiceMBeanJavaDoc
    public void setWritableElementClass(String key, Class<WritableElement> clazz){
        writableElementClassMap.put(key, clazz);
    }
    
    // WritableRecordFactoryServiceMBeanJavaDoc
    public Class<WritableElement> getWritableElementClass(String key){
        return writableElementClassMap == null ? null : writableElementClassMap.get(key);
    }
    
    // WritableRecordFactoryServiceMBeanJavaDoc
    public void setWritableElementServiceName(String key, ServiceName name){
        writableElementServiceNameMap.put(key, name);
    }
    
    // WritableRecordFactoryServiceMBeanJavaDoc
    public ServiceName getWritableElementServiceName(String key){
        return writableElementServiceNameMap == null ? null : writableElementServiceNameMap.get(key);
    }
    
    // WritableRecordFactoryServiceMBeanJavaDoc
    public void setFormat(String fmt){
        format = fmt;
    }
    
    // WritableRecordFactoryServiceMBeanJavaDoc
    public String getFormat(){
        return format;
    }
    
    // WritableRecordFactoryServiceMBeanJavaDoc
    public void setFormatKeyMapping(Properties mapping){
        formatKeyMappings = mapping;
    }
    // WritableRecordFactoryServiceMBeanJavaDoc
    public Properties getFormatKeyMapping(){
        return formatKeyMappings;
    }
    
    // WritableRecordFactoryServiceMBeanJavaDoc
    public void setFormatKeyProperty(String key, Property prop){
        prop.setIgnoreNullProperty(true);
        formatKeyPropertyMap.put(key, prop);
    }
    // WritableRecordFactoryServiceMBeanJavaDoc
    public Property getFormatKeyProperty(String key){
        return formatKeyPropertyMap == null ? null : formatKeyPropertyMap.get(key);
    }
    
    // WritableRecordFactoryServiceMBeanJavaDoc
    public void setIterateFormatKeyMapping(String key, Properties mapping){
        iterateFormatKeyMappings.put(key, mapping);
    }
    // WritableRecordFactoryServiceMBeanJavaDoc
    public Properties getIterateFormatKeyMapping(String key){
        return iterateFormatKeyMappings.get(key);
    }
    
    // WritableRecordFactoryServiceMBeanJavaDoc
    public void setIterateFormat(String key, String format){
        iterateFormats.put(key, format);
    }
    // WritableRecordFactoryServiceMBeanJavaDoc
    public String getIterateFormat(String key){
        return (String)iterateFormats.get(key);
    }
    
    /**
     * sB<p>
     *
     * @exception Exception T[rX̐Ɏsꍇ
     */
    public void createService() throws Exception {
        writableElementClassMap = new HashMap<String, Class<WritableElement>>();
        writableElementServiceNameMap = new HashMap<String, ServiceName>();
        formatKeyPropertyMap = new HashMap<String, Property>();
        iterateFormatKeyMappings = new HashMap<String, Properties>();
        iterateformatKeyPropertyMap = new HashMap<String, Map<String, Property>>();
        iterateFormats = new HashMap<String, String>();
        iterateFormatMap = new HashMap<String, List<ParsedElement>>();
    }
    
    /**
     * JnsB<p>
     *
     * @exception Exception T[rX̊JnɎsꍇ
     */    
    public void startService()throws Exception{
        
        //^ꂽNXŃCX^X𐶐ł邩ǂmF
        Collection<Class<WritableElement>> implClasses = writableElementClassMap.values();
        for(Class<WritableElement> implClass : implClasses){
            implClass.newInstance();
        }
        
        parsedElements = parseFormat(format);
        
        if(formatKeyMappings != null && formatKeyMappings.size() != 0){
            formatKeyPropertyMap.clear();
            for(Map.Entry<Object, Object> entry : formatKeyMappings.entrySet()){
                final String key = (String)entry.getKey();
                final String prop = (String)entry.getValue();
                if(prop != null && prop.length() != 0){
                    Property property = PropertyFactory.createProperty(prop);
                    property.setIgnoreNullProperty(true);
                    formatKeyPropertyMap.put(key, property);
                }
            }
        }
        
        if(iterateFormats.size() != 0){
            for(Map.Entry<String, String> entry : iterateFormats.entrySet()){
                final String itrKey = entry.getKey();
                final String format = entry.getValue();
                iterateFormatMap.put(itrKey, parseFormat(format));
                if(!iterateFormatKeyMappings.containsKey(itrKey)){
                    throw new IllegalArgumentException(
                        "IterateFormatKeyMapping that corresponds to \"" + itrKey + "\" is not specified."
                    );
                }
            }
        }
        if(iterateFormatKeyMappings.size() != 0){
            for(Map.Entry<String, Properties> entry : iterateFormatKeyMappings.entrySet()){
                final String itrKey = entry.getKey();
                final Properties props = entry.getValue();
                final Map<String, Property> mapping = new HashMap<String, Property>();
                for(Map.Entry<Object, Object> entry2 : props.entrySet()){
                    final String key = (String)entry2.getKey();
                    final String prop = (String)entry2.getValue();
                    if(prop != null && prop.length() != 0){
                        Property property = PropertyFactory.createProperty(prop);
                        property.setIgnoreNullProperty(true);
                        mapping.put(key, property);
                    }
                }
                iterateformatKeyPropertyMap.put(itrKey, mapping);
            }
        }
    }
    
    /**
     * ~sB<p>
     *
     * @exception Exception T[rX̒~Ɏsꍇ
     */    
    public void stopService() throws Exception{
        parsedElements = null;
        iterateformatKeyPropertyMap.clear();
        iterateFormatMap.clear();
    }
    /**
     * jsB<p>
     *
     * @exception Exception T[rX̔jɎsꍇ
     */            
    public void destroyService() throws Exception{
        writableElementClassMap = null;
        writableElementServiceNameMap = null;
        formatKeyPropertyMap = null;
        iterateformatKeyPropertyMap = null;
        iterateFormats = null;
        iterateFormatMap = null;
    }
    
    /**
     * w肳ꂽo͗vfMap{@link WritableRecord}𐶐B<p>
     * {@link #setFormat(String)}Ŏw肳ꂽtH[}bgɏ]āA{@link WritableElement}𐶐Ai[WritableRecordԂB<br>
     * setFormat(String)Ŏw肳ꂽtH[}bgɁAL[w肳Ăꍇ́Ao͗vfMaploāA{@link #setWritableElementClass(String, Class)}܂{@link #setWritableElementServiceName(String, ServiceName)}Ń}bsOꂽWritableElementCX^XɊi[B<br>
     * 
     * @param elements o͗vfMapIuWFNg
     * @return MessageWriterT[rX̓̓IuWFNgƂȂWritableRecord
     */
    public WritableRecord createRecord(Object elements){
        if(getState() != State.STARTED){
            throw new IllegalServiceStateException(this);
        }
        
        final WritableRecord record = new WritableRecord();
        if(parsedElements == null || parsedElements.size() == 0){
            for(String key : getElementKeys(elements)){
                final WritableElement element = createElement(
                    key,
                    getElementValue(key, elements, formatKeyPropertyMap)
                );
                record.addElement(element);
            }
        }else{
            for(int i = 0, imax = parsedElements.size(); i < imax; i++){
                final ParsedElement parsedElement = parsedElements.get(i);
                if(parsedElement.isKeyElement()){
                    final String key = parsedElement.getValue();
                    final WritableElement element = createElement(
                        key,
                        getElementValue(key, elements, formatKeyPropertyMap)
                    );
                    if(element != null){
                        record.addElement(element);
                    }
                }else{
                    final SimpleElement element = new SimpleElement();
                    element.setValue(parsedElement.getValue());
                    record.addElement(element);
                }
            }
        }
        return record;
    }
    
    /**
     * w肳ꂽo͗vfMapL[̏W擾B<p>
     *
     * @param elements o͗vf̃}bv
     * @return L[̏W
     */
    @SuppressWarnings("unchecked")
    protected Set<String> getElementKeys(Object elements){
        if(elements instanceof Map){
            return ((Map<String, Object>)elements).keySet();
        }else{
            return SimpleProperty.getPropertyNames(elements);
        }
    }
    
    /**
     * w肳ꂽo͗vf̃}bvAw肳ꂽL[̒l擾B<p>
     *
     * @param key o͗vf}bṽL[
     * @param elements o͗vf}bv
     * @param propMapping L[ƃvpeB̃}bsO
     * @return o͗vf̃L[ɊYl
     */
    @SuppressWarnings("unchecked")
    protected Object getElementValue(
        String key,
        Object elements,
        Map<String, Property> propMapping
    ){
        if(propMapping != null && propMapping.containsKey(key)){
            final Logger logger = getLogger();
            final Property prop = propMapping.get(key);
            try{
                return prop.getProperty(elements);
            }catch(NoSuchPropertyException e){
                return null;
            }catch(InvocationTargetException e){
                logger.write("DWRF_00001", e, key, prop);
                return null;
            }
        }else if(elements instanceof Map){
            return ((Map<Object,Object>)elements).get(key);
        }else{
            return elements;
        }
    }
    
    /**
     * o͗vfw肳ꂽL[̒loāAΉ{@link WritableElement}𐶐B<p>
     * <br>
     * 
     * @param key L[
     * @param val o͗vf
     * @return WritableElementCX^X
     */
    @SuppressWarnings("unchecked")
    protected WritableElement createElement(
        String key,
        Object val
    ){
        if(key.endsWith(ITERATE_SUFFIX)){
            Object[] array = null;
            if(val == null){
                return createSingleElement(key, val);
            }else if(val instanceof Collection){
                Collection<Object> col = (Collection<Object>)val;
                array = col.toArray();
            }else if(val.getClass().isArray()){
                array = (Object[])val;
            }else{
                return createSingleElement(key, val);
            }
            if(array == null || array.length == 0){
                return null;
            }
            IterateWritableElement itrElement = new IterateWritableElement(key);
            if(iterateFormatMap.containsKey(key)){
                final List<ParsedElement> parsedElements
                     = iterateFormatMap.get(key);
                final Map<String, Property> itrPropMapping
                     = iterateformatKeyPropertyMap.get(key);
                for(int i = 0; i < array.length; i++){
                    for(int j = 0, jmax = parsedElements.size(); j < jmax; j++){
                        final ParsedElement parsedElem = parsedElements.get(j);
                        WritableElement element = null;
                        if(parsedElem.isKeyElement()){
                            final String itrElementKey = parsedElem.getValue();
                            element = createSingleElement(
                                itrElementKey,
                                getElementValue(
                                    itrElementKey,
                                    array[i],
                                    itrPropMapping
                                )
                            );
                        }else{
                            element = new SimpleElement(parsedElem.getValue());
                        }
                        if(element != null){
                            itrElement.addElement(element);
                        }
                    }
                }
            }else{
                for(int i = 0; i < array.length; i++){
                    final WritableElement element
                         = createSingleElement(key, array[i]);
                    if(element != null){
                        itrElement.addElement(element);
                    }
                }
            }
            return itrElement;
        }else{
            return createSingleElement(key, val);
        }
    }
    
    /**
     * o͗vfw肳ꂽL[̒loāAΉ{@link WritableElement}𐶐B<p>
     * {@link #getElementValue(String, Object, Map)}ŁAo͗vfw肳ꂽL[̒loB{@link #getWritableElementClass(String)}܂{@link #getWritableElementServiceName(String)}ŁAL[ɊY{@link WritableElement}肵ÃCX^X擾āAL[ƒlݒ肵ĕԂB<br>
     * L[ɊY{@link WritableElement}łȂꍇ́A{@link SimpleElement}NXgpB<br>
     * 
     * @param key L[
     * @param val o͗vf
     * @return WritableElementCX^X
     */
    protected WritableElement createSingleElement(String key, Object val){
        WritableElement element = null;
        
        final Class<WritableElement> elementClass = getWritableElementClass(key);
        if(elementClass != null){
            try{
                element = elementClass.newInstance();
            }catch(IllegalAccessException e){
                // NȂ͂
                element = new SimpleElement();
            }catch(InstantiationException e){
                // NȂ͂
                element = new SimpleElement();
            }
        }else{
            final ServiceName elementServiceName = getWritableElementServiceName(key);
            if(elementServiceName != null){
                try{
                    element = (WritableElement)ServiceManagerFactory
                        .getServiceObject(elementServiceName);
                }catch(ServiceNotFoundException e){
                    element = new SimpleElement();
                }
            }else{
                element = new SimpleElement();
            }
        }
        
        element.setKey(key);
        element.setValue(val);
        
        return element;
    }
    
    /**
     * tH[}bgp[X\bhB<p>
     * setFormat(String)ŁAZbgꂽtH[}bgp[XAParsedElementListԂ
     * <ol>
     * <li>'%'ň͂񂾕L[[hƔFB</li>
     * <li>'\%'LqƁA'%'ƂĔFB</li>
     * <li>'\\'ƋLqƁA'\'ƂĔFB</li>
     * <li>tH[}bgnull܂͋󕶎̂ƂAnullԂB</li>
     * <li>tH[}bgɂāA"%"Ŏn߂L[[h̋Lq%ŕĂȂꍇAIllegalArgumentExceptionthrowB</li>
     * <li>() tH[}bg" %D% łB"̏ꍇA3ParsedElement܂ListԋpB</li> 
     *         <ul>
     *            <li>ŏParsedElementmValue" "AmIsKeyWordfalse</li>
     *            <li>2ԖڂParsedElementmValue"D"AmIsKeyWordtrue</li>
     *            <li>3ԖڂParsedElementmValue" łB"AmIsKeyWordfalse</li>
     *         </ul>
     * <li>tH[}bgnull͋󕶎̏ꍇAnullԂB</li>
     * </ol>
     * 
     * @return ParsedElementList
     */
    protected List<ParsedElement> parseFormat(String format){
        //
        if(format == null || format.length() == 0){
            //nullԂ
            return null;
        }
        final List<ParsedElement> result = new ArrayList<ParsedElement>();
        final StringBuilder word = new StringBuilder();
        boolean isStartKey = false;
        boolean isEscape = false;
        for(int i = 0, max = format.length(); i < max; i++){
            final char c = format.charAt(i);
            switch(c){
            case '%':
                if(isEscape){
                    word.append(c);
                    isEscape = false;
                }else if(isStartKey){
                    if(word.length() != 0){
                        //L[[hƂĒǉ
                        final ParsedElement elem = new ParsedElement(
                            word.toString(),
                            true
                        );
                        result.add(elem);
                        word.setLength(0);
                        isStartKey = false;
                    }else{
                        throw new IllegalArgumentException(
                            "Keyword must not be null. : " + format
                        );
                    }
                }else{
                    if(word.length() > 0){
                        //Œ胁bZ[WƂĒǉ
                        final ParsedElement elem = new ParsedElement(
                            word.toString(),
                            false
                        );
                        result.add(elem);
                        word.setLength(0);
                    }
                    isStartKey = true;
                }
                break;
            case '\\':
                if(isEscape){
                    word.append(c);
                    isEscape = false;
                }else{
                    isEscape = true;
                }
                break;
            default:
                if(isEscape){
                    throw new IllegalArgumentException(
                        "'\' is escape character. : " + format
                    );
                }else{
                    word.append(c);
                }
                break;
            }
        }
        if(isEscape){
            throw new IllegalArgumentException(
                "'\' is escape character. : " + format
            );
        }
        if(isStartKey){
            throw new IllegalArgumentException(
                "'%' is key separator character. : " + format
            );
        }
        if(word.length() > 0){
            //Œ胁bZ[WƂĒǉ
            final ParsedElement elem = new ParsedElement(
                word.toString(),
                false
            );
            result.add(elem);
            word.setLength(0);
        }
        return result;
    }
    
    /**
     * p[XꂽtH[}bg̗vf\NXB<p>
     * String̒lƁAkeyۂ̎(boolean)lB
     * 
     * @author Y.Tokuda
     */
    protected static class ParsedElement implements Serializable{
        
        private static final long serialVersionUID = 6554326776504636150L;
        
        //oϐ
        protected String value;
        protected boolean isKeyElement;
        
        /**
         * RXgN^B<p>
         *
         * @param val tH[}bg̗vf
         * @param isKey L[ǂtO
         */
        public ParsedElement(String val, boolean isKey){
            value = val;
            isKeyElement = isKey;
        }
        
        /**
         * tH[}bg̗vf擾B<p>
         *
         * @return tH[}bg̗vf
         */
        public String getValue(){
            return value;
        }
        
        /**
         * ̗vfL[ǂ𔻒肷B<p>
         *
         * @return ̗vfL[̏ꍇtrue
         */
        public boolean isKeyElement(){
            return isKeyElement;
        }
    }
}
