/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.service.websocket.jetty;

import java.io.IOException;
import java.util.Map;
import java.util.HashMap;
import java.util.Collections;
import java.util.Timer;
import java.util.TimerTask;

import org.eclipse.jetty.websocket.WebSocket;

import jp.ossc.nimbus.core.ServiceBase;

/**
 * {@link PingSender}T[rXB<p>
 * 
 * @author M.Takata
 */
public class DefaultPingSenderService extends ServiceBase implements PingSender, DefaultPingSenderServiceMBean{
    
    private static final long serialVersionUID = 7948288564532727379L;
    
    protected long pingInterval = 30000l;
    protected int maxRetryCount;
    protected int closeCodeForNoPong = 1000;
    protected boolean isAllowNoPong = false;
    
    protected Map<WebSocket.Connection, PingContainer> pingContainerMap;
    protected Timer pingTimer;
    
    @Override
    public void setPingInterval(long interval){
        pingInterval = interval;
    }
    @Override
    public long getPingInterval(){
        return pingInterval;
    }
    
    @Override
    public void setMaxRetryCount(int max){
        maxRetryCount = max;
    }
    @Override
    public int getMaxRetryCount(){
        return maxRetryCount;
    }
    
    @Override
    public void setCloseCodeForNoPong(int code){
        closeCodeForNoPong = code;
    }
    @Override
    public int getCloseCodeForNoPong(){
        return closeCodeForNoPong;
    }
    
    @Override
    public void setAllowNoPong(boolean isAllow){
        isAllowNoPong = isAllow;
    }
    @Override
    public boolean isAllowNoPong(){
        return isAllowNoPong;
    }
    
    @Override
    public void createService() throws Exception{
        pingContainerMap = Collections.synchronizedMap(new HashMap<WebSocket.Connection, PingContainer>());
    }
    
    @Override
    public void startService() throws Exception{
        pingTimer = new Timer(true);
    }
    
    @Override
    public void stopService() throws Exception{
        if(pingTimer == null){
            pingTimer.cancel();
            pingTimer = null;
        }
        pingContainerMap.clear();
    }
    
    @Override
    public void destroyService() throws Exception{
        pingContainerMap = null;
    }
    
    @Override
    public void regist(WebSocket.Connection connection, WebSocket.FrameConnection frameConnection){
        pingContainerMap.put(connection, new PingContainer(connection, frameConnection));
    }
    
    @Override
    public void unregist(WebSocket.Connection connection){
        if(pingContainerMap != null){
            pingContainerMap.remove(connection);
        }
    }
    
    @Override
    public void onPong(WebSocket.Connection connection, byte[] data, int offset, int length){
        if(pingContainerMap != null){
            PingContainer container = pingContainerMap.get(connection);
            if(container != null){
                container.onPong(data, offset, length);
            }
        }
    }
    
    protected class PingContainer{
        
        protected WebSocket.Connection connection;
        protected WebSocket.FrameConnection frameConnection;
        protected TimerTask pingTask;
        protected long lastPingTime = -1l;
        protected long lastPongTime = -1l;
        protected long retryCount = 0;
        
        public PingContainer(WebSocket.Connection connection, WebSocket.FrameConnection frameConnection){
            this.connection = connection;
            this.frameConnection = frameConnection;
            pingTask = new PingTimerTask();
            pingTimer.scheduleAtFixedRate(pingTask, 0l, pingInterval);
        }
        
        public void onPong(byte[] data, int offset, int length){
            lastPongTime = System.currentTimeMillis();
        }
        
        public void close(){
            if(pingTask != null){
                pingTask.cancel();
            }
            if(connection != null){
                connection.close(closeCodeForNoPong, null);
                connection = null;
                frameConnection = null;
            }
        }
        
        protected class PingTimerTask extends TimerTask{
            
            @Override
            public void run(){
                if(!isReceivePong()){
                    if(maxRetryCount <= retryCount){
                        PingContainer.this.close();
                        return;
                    }else{
                        retryCount++;
                    }
                }else{
                    retryCount = 0;
                }
                try{
                    sendPing(null, 0, 0);
                    lastPingTime = System.currentTimeMillis();
                }catch(IOException e){
                }
            }
            
            protected void sendPing(byte[] data, int offset, int length) throws IOException{
                if(frameConnection.isPing((byte)0x02)){
                    frameConnection.sendControl((byte)0x02, data, offset, length);
                }else if(frameConnection.isPing((byte)0x09)){
                    frameConnection.sendControl((byte)0x09, data, offset, length);
                }
            }
            
            protected boolean isReceivePong(){
                if(isAllowNoPong){
                    return true;
                }
                if(lastPingTime > 0){
                    if(lastPongTime > 0){
                        return lastPingTime <= lastPongTime;
                    }else{
                        return false;
                    }
                }else{
                    return true;
                }
            }
        }
    }
}
