/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.service.log;

import java.util.*;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.commons.logging.LogConfigurationException;

import jp.ossc.nimbus.beans.ServiceNameEditor;
import jp.ossc.nimbus.core.*;

/**
 * NimbuspJakarta Commons LoggingLogFactorygNXB<p>
 * vpeBt@C"commons-logging.properties"ɁA
 * <pre>
 * org.apache.commons.logging.LogFactory=jp.ossc.nimbus.service.log.NimbusLogFactory
 * </pre>
 * w肷鎖ŁA{@link LogFactory}̎NXƂāÃNXgp\ɂȂB܂AVXevpeBłl̎w肪\łB<br>
 * <p>
 * ̃Ot@NǵA{@link CommonsLogFactory}C^tF[XT[rXgp{@link Log}CX^X𐶐B<br>
 * ̂߁ACommonsLogFactoryC^tF[XT[rX`āA̒`[hĂKvBCommonsLogFactoryC^tF[XT[rX[hĂȂꍇA܂́ANĂȂꍇ́AftHg{@link LogFactory}gpLogCX^X𐶐B<br>
 * ftHgLogFactorýAorg.apache.commons.logging.impl.LogFactoryImplgpBAAftHgLogFactoryύX鎖\ŁAvpeBt@C"commons-logging.properties"ɁA
 * <pre>
 * jp.ossc.nimbus.service.log.NimbusLogFactory.DefaultLogFactory=org.apache.commons.logging.impl.Log4jFactory
 * </pre>
 * ̂悤Ɏw肷鎖ŁAύXłB܂AVXevpeBłl̎w肪\łB<br>
 * <p>
 * {@link CommonsLogFactory}C^tF[XT[rXƂāA{@link DefaultCommonsLogFactoryService}񋟂ĂB<br>
 * DefaultCommonsLogFactoryServicéAN{@link LogFactory#getFactory()}ĂяoāÃt@Ng̃CX^X擾A{@link #setCommonsLogFactory(CommonsLogFactory)}Ŏg̃t@Ngɐݒ肷B̂߁AOq"org.apache.commons.logging.LogFactory"vpeB̐ݒƁADefaultCommonsLogFactoryServicẽT[rX`݂̂ŁAgp\łB<br>
 * AALogFactory.getFactory()Ŏ擾łLogFactoryCX^X́AĂяoXbhɊ֘AtꂽNX[_PʂŎ擾B̂߁ADefaultCommonsLogFactoryServicẽ[hsXbhɊ֘AtꂽNX[_ƁA{@link LogFactory#getLog(String)}ĂяoXbhɊ֘AtꂽNX[_قȂꍇ́AL̐ݒ݂̂ł́Ãt@NgDefaultCommonsLogFactoryService̎QƂ𓾂鎖łȂB<br>
 * <p>
 * {@link LogFactory#getLog(Class)}A{@link LogFactory#getLog(String)}ĂяoXbhɊ֘AtꂽNX[_ƁACommonsLogFactory̎T[rX[hXbhɊ֘AtꂽNX[_قȂꍇ́AvpeBt@C"commons-logging.properties"ɁA
 * <pre>
 * jp.ossc.nimbus.service.log.NimbusLogFactory.CommonsLogFactoryName=Nimbus#CommonsLog
 * </pre>
 * ̂悤ɁACommonsLogFactory̎T[rX̃T[rXw肷KvB<br>
 * 
 * @author M.Takata
 * @see CommonsLogFactory
 */
public class NimbusLogFactory extends LogFactory implements java.io.Serializable{
    
    private static final long serialVersionUID = -3343921992875545571L;
    
    /**
     * {@link CommonsLogFactory}ݒ肳ĂȂɎgp{@link LogFactory}̎NXw肷vpeBB<p>
     * vpeBt@C"commons-logging.properties"ɁÃvpeBw肷B܂́AVXevpeBŎw肷B<br>
     */
    public static final String DEFAULT_FACTORY_PROPERTY =
        "jp.ossc.nimbus.service.log.NimbusLogFactory.DefaultLogFactory";
    
    /**
     * {@link CommonsLogFactory}̃T[rXw肷vpeBB<p>
     * vpeBt@C"commons-logging.properties"ɁÃvpeBw肷B܂́AVXevpeBŎw肷B<br>
     * {@link LogFactory#getLog(Class)}A{@link LogFactory#getLog(String)}ĂяoXbh̃NX[_ƁACommonsLogFactory̎T[rX[hXbh̃NX[_قȂꍇ́ÃvpeBw肷KvB҂̃NX[_ꍇ́ÃvpeBw肷Kv͂ȂB<br>
     */
    public static final String FACTORY_NAME_PROPERTY =
        "jp.ossc.nimbus.service.log.NimbusLogFactory.CommonsLogFactoryName";
    
    /**
     * {@link #DEFAULT_FACTORY_PROPERTY}̎w肪ȂꍇɁA{@link LogFactory}̎NXB<p>
     */
    public static final String DEFAULT_FACTORY_DEFAULT =
        "org.apache.commons.logging.impl.LogFactoryImpl";
    
    /**
     * {@link Log}𐶐{@link CommonsLogFactory}B<p>
     */
    private CommonsLogFactory logFactory;
    
    /**
     * ftHg{@link LogFactory}B<p>
     */
    private LogFactory deafultLogFactory;
    
    /**
     * Ǘ}bvB<p>
     * <table border="1">
     *   <tr bgcolor="#CCCCFF"><th colspan="2">L[</th><th colspan="2">l</th></tr>
     *   <tr bgcolor="#CCCCFF"><th>^</th><th>e</th><th>^</th><th>e</th></tr>
     *   <tr><td>Object</td><td></td><td>Object</td><td>l</td></tr>
     * </table>
     */
    private Map<String, Object> attributes = new HashMap<String, Object>();
    
    /**
     * {@link Log}CX^XǗ}bvB<p>
     * <table border="1">
     *   <tr bgcolor="#CCCCFF"><th colspan="2">L[</th><th colspan="2">l</th></tr>
     *   <tr bgcolor="#CCCCFF"><th>^</th><th>e</th><th>^</th><th>e</th></tr>
     *   <tr><td>Object</td><td>LogCX^Xʏ</td><td>Log</td><td>LogCX^X</td></tr>
     * </table>
     */
    private Map<Object, Log> logInstances = new HashMap<Object, Log>();
    
    /**
     * {@link CommonsLogFactory}ݒ肷B<p>
     *
     * @param factory CommonsLogFactoryIuWFNg
     */
    public void setCommonsLogFactory(CommonsLogFactory factory){
        if(logFactory != null && logFactory == factory){
            return;
        }
        logFactory = factory;
        if(factory != null){
            final String[] names = getAttributeNames();
            for(String name : names){
                factory.setAttribute(name, getAttribute(name));
            }
        }
        for(Map.Entry<Object, Log> entry : logInstances.entrySet()){
            final Object key = entry.getKey();
            final LogWrapper log = (LogWrapper)entry.getValue();
            if(factory != null){
                if(key instanceof Class<?>){
                    log.setRealLog(factory.getInstance((Class<?>)key));
                }else{
                    log.setRealLog(factory.getInstance((String)key));
                }
                log.real();
            }else{
                log.dummy();
            }
        }
    }
    
    /**
     * w肳ꂽL[ɑΉLogCX^X擾B<p>
     * 
     * @param key L[
     * @return LogCX^X
     */
    private Log getInstance(final Object key){
        if(logInstances.containsKey(key)){
            return logInstances.get(key);
        }
        
        if(deafultLogFactory == null){
            deafultLogFactory = createDefaultLogFactory();
        }
        LogWrapper log = null;
        if(logFactory != null){
            if(key instanceof Class<?>){
                log = new LogWrapper(
                    logFactory.getInstance((Class<?>)key),
                    deafultLogFactory.getInstance((Class<?>)key)
                );
            }else{
                log = new LogWrapper(
                    logFactory.getInstance((String)key),
                    deafultLogFactory.getInstance((String)key)
                );
            }
            logInstances.put(key, log);
            return log;
        }
        
        String factoryName
             = System.getProperty(FACTORY_NAME_PROPERTY);
        if(factoryName == null){
            factoryName = (String)getAttribute(FACTORY_NAME_PROPERTY);
        }
        if(factoryName == null){
            if(key instanceof Class<?>){
                log = new LogWrapper(
                    deafultLogFactory.getInstance((Class<?>)key)
                );
            }else{
                log = new LogWrapper(
                    deafultLogFactory.getInstance((String)key)
                );
            }
            logInstances.put(key, log);
            return log;
        }
        
        final ServiceNameEditor editor = new ServiceNameEditor();
        editor.setAsText(factoryName);
        final ServiceName name = (ServiceName)editor.getValue();
        LogWrapper tmpLog = null;
        if(key instanceof Class<?>){
            tmpLog = new LogWrapper(deafultLogFactory.getInstance((Class<?>)key));
        }else{
            tmpLog = new LogWrapper(deafultLogFactory.getInstance((String)key));
        }
        log = tmpLog;
        logInstances.put(key, log);
        final String managerName = name.getServiceManagerName();
        final String serviceName = name.getServiceName();
        if(!ServiceManagerFactory.isRegisteredManager(managerName)){
            waitRegistrationManager(managerName, serviceName);
            return log;
        }
        final ServiceManager manager
             = ServiceManagerFactory.findManager(managerName);
        if(!manager.isRegisteredService(serviceName)){
            waitRegistrationService(manager, serviceName);
            return log;
        }
        final Service service = manager.getService(serviceName);
        waitStartService(service);
        return log;
    }
    
    /**
     * CommonsLogFactoryT[rXo^ServiceManagerAServiceManagerFactoryɓo^̂ҋ@āACommonsLogFactoryT[rXݒ肷B<p>
     * T[rXtargetMngServiceManagerServiceManagerFactoryɓo^ƁAT[rXtargetServicẽT[rX擾Ă݂B擾łȂꍇ́A{@link #waitRegistrationService(ServiceManager, String)}ĂяoāAT[rXo^̂ҋ@B擾łꍇ́A{@link #waitStartService(Service)}ĂяoāAT[rXJn̂ҋ@B<br>
     *
     * @param targetMng CommonsLogFactoryT[rXo^ServiceManager̃T[rX
     * @param targetService CommonsLogFactorỹT[rX
     */
    private void waitRegistrationManager(
        final String targetMng,
        final String targetService
    ){
        ServiceManagerFactory.addRegistrationListener(
            new RegistrationListener(){
                public void registered(RegistrationEvent e){
                    final ServiceManager manager
                         = (ServiceManager)e.getRegistration();
                    if(!manager.getServiceName().equals(targetMng)){
                        return;
                    }
                    ServiceManagerFactory.removeRegistrationListener(this);
                    Service service = null;
                    try{
                        service = manager.getService(targetService);
                    }catch(ServiceNotFoundException ex){
                        waitRegistrationService(
                            manager,
                            targetService
                        );
                    }
                    if(service != null){
                        waitStartService(service);
                    }
                }
                public void unregistered(RegistrationEvent e){
                }
            }
        );
    }
    
    /**
     * CommonsLogFactoryT[rXServiceManagerɓo^̂ҋ@āACommonsLogFactoryT[rXݒ肷B<p>
     * T[rXtargetServiceCommonsLogFactoryServiceManagerɓo^ƁA{@link #waitStartService(Service)}ĂяoāAT[rXJn̂ҋ@B<br>
     *
     * @param targetMng CommonsLogFactoryT[rXo^ServiceManager
     * @param targetService CommonsLogFactorỹT[rX
     * @param log LogWrapperIuWFNg
     */
    private void waitRegistrationService(
        final ServiceManager targetMng,
        final String targetService
    ){
        targetMng.addRegistrationListener(
            new RegistrationListener(){
                public void registered(RegistrationEvent e){
                    final Service service
                         = (Service)e.getRegistration();
                    if(!service.getServiceName().equals(targetService)){
                        return;
                    }
                    targetMng.removeRegistrationListener(this);
                    waitStartService(service);
                }
                public void unregistered(RegistrationEvent e){
                }
            }
        );
    }
    
    /**
     * CommonsLogFactoryT[rXJn̂ҋ@āACommonsLogFactoryT[rXݒ肷B<p>
     * T[rXserviceJnƁA{@link #setCommonsLogFactory(CommonsLogFactory)}ĂяoāACommonsLogFactory̎QƂݒ肷B<br>
     * ܂AT[rXservice~ƁA{@link #setCommonsLogFactory(CommonsLogFactory)}ĂяoāACommonsLogFactory̎QƂjB<br>
     *
     * @param targetService CommonsLogFactoryT[rX
     */
    private void waitStartService(final Service service){
        Service targetService = null;
        if(!(service instanceof ServiceStateListenable)){
            final String managerName = service.getServiceManagerName();
            final ServiceManager mng = ServiceManagerFactory.findManager(
                managerName
            );
            targetService = mng;
        }else{
            targetService = service;
        }
        final ServiceStateListenable broad = ServiceManagerFactory
            .getServiceStateListenable(
                targetService.getServiceManagerName(),
                targetService.getServiceName()
            );
        if(broad != null){
            broad.addServiceStateListener(
                new ServiceStateListener(){
                    public void stateChanged(ServiceStateChangeEvent e)
                     throws Exception{
                        CommonsLogFactory factory = null;
                        switch(service.getState()){
                        case STARTED:
                            try{
                                factory = (CommonsLogFactory)
                                    ServiceManagerFactory.getServiceObject(
                                        service.getServiceManagerName(),
                                        service.getServiceName()
                                );
                            }catch(ServiceNotFoundException ex){
                                factory = null;
                            }
                            break;
                        case STOPPED:
                            factory = null;
                            break;
                        case DESTROYED:
                            broad.removeServiceStateListener(this);
                            factory = null;
                            break;
                        default:
                        }
                        setCommonsLogFactory(factory);
                    }
                    public boolean isEnabledState(Service.State state){
                        return state == Service.State.STARTED
                             || state == Service.State.STOPPED
                             || state == Service.State.DESTROYED;
                    }
                }
            );
        }
        if(service.getState() == Service.State.STARTED){
            CommonsLogFactory factory = null;
            try{
                factory = (CommonsLogFactory)ServiceManagerFactory.getServiceObject(
                    service.getServiceManagerName(),
                    service.getServiceName()
                );
            }catch(ServiceNotFoundException ex){
                // N蓾Ȃ
            }
            setCommonsLogFactory(factory);
        }
    }
    
    /**
     * ftHgLogFactory𐶐B<p>
     *
     * @return ftHgLogFactory
     */
    private LogFactory createDefaultLogFactory()
     throws LogConfigurationException {
        LogFactory factory = null;
        try{
            final String factoryClass
                 = System.getProperty(DEFAULT_FACTORY_PROPERTY);
            if(factoryClass != null){
                final ClassLoader classLoader
                     = Thread.currentThread().getContextClassLoader();
                factory = newFactory(factoryClass, classLoader);
            }
        }catch(SecurityException e){
            factory = null;
        }
        
        if(factory == null){
            final String factoryClass = (String)getAttribute(DEFAULT_FACTORY_PROPERTY);
            if(factoryClass != null){
                final ClassLoader classLoader
                     = Thread.currentThread().getContextClassLoader();
                factory = newFactory(factoryClass, classLoader);
            }
        }
        
        if(factory == null){
            final ClassLoader classLoader
                 = Thread.currentThread().getContextClassLoader();
            factory = newFactory(DEFAULT_FACTORY_DEFAULT, classLoader);
        }
        
        if(factory != null){
            final String[] names = getAttributeNames();
            for(String name : names){
                factory.setAttribute(name, getAttribute(name));
            }
        }
        
        return factory;
    }
    
    /**
     * Ŏw肵NXIuWFNgɊ֘At{@link Log}CX^X擾B<p>
     *
     * @param clazz 擾LogCX^XʂL[ƂȂNXIuWFNg
     * @return Ŏw肵NXIuWFNgɊ֘At{@link Log}CX^X
     * @exception LogConfigurationException LogCX^X̍쐬Ɏsꍇ
     */
    public Log getInstance(@SuppressWarnings("rawtypes") Class clazz) throws LogConfigurationException{
        return getInstance((Object)clazz);
    }
    
    /**
     * Ŏw肵OɊ֘At{@link Log}CX^X擾B<p>
     *
     * @param name 擾LogCX^Xʂ閼O
     * @return Ŏw肵OɊ֘At{@link Log}CX^X
     * @exception LogConfigurationException LogCX^X̍쐬Ɏsꍇ
     */
    public Log getInstance(String name) throws LogConfigurationException{
        return getInstance((Object)name);
    }
    
    /**
     * 쐬{@link Log}CX^XJB<p>
     */
    public void release(){
        logInstances.clear();
        if(logFactory != null){
            logFactory.release();
        }
    }
    
    /**
     * l擾B<p>
     * "commons-logging.properties"Őݒ肵vpeBƂĊi[B<p>
     *
     * @param name 
     * @return l
     * @see #getAttributeNames()
     * @see #removeAttribute(String)
     * @see #setAttribute(String, Object)
     */
    public Object getAttribute(String name){
        if(logFactory != null){
            return logFactory.getAttribute(name);
        }
        return attributes.get(name);
    }
    
    /**
     * ̔z擾B<p>
     * "commons-logging.properties"Őݒ肵vpeBƂĊi[B<p>
     *
     * @return ̔z
     * @see #getAttribute(String)
     * @see #removeAttribute(String)
     * @see #setAttribute(String, Object)
     */
    public String[] getAttributeNames(){
        if(logFactory != null){
            return logFactory.getAttributeNames();
        }
        return attributes.keySet()
            .toArray(new String[attributes.size()]);
    }
    
    /**
     * 폜B<p>
     * "commons-logging.properties"Őݒ肵vpeBƂĊi[B<p>
     *
     * @param name 
     * @return l
     * @see #getAttribute(String)
     * @see #getAttributeNames()
     * @see #setAttribute(String, Object)
     */
    public void removeAttribute(String name){
        attributes.remove(name);
        if(logFactory != null){
            logFactory.removeAttribute(name);
        }
    }
    
    /**
     * ݒ肷B<p>
     * "commons-logging.properties"Őݒ肵vpeBƂĊi[B<p>
     *
     * @param name 
     * @param value l
     * @see #getAttribute(String)
     * @see #getAttributeNames()
     * @see #removeAttribute(String)
     */
    public void setAttribute(String name, Object value){
        attributes.put(name, value);
        if(logFactory != null){
            logFactory.setAttribute(name, value);
        }
    }
    
    private class LogWrapper implements org.apache.commons.logging.Log{
        
        private Log logger;
        private Log dummyLogger;
        private Log currentLogger;
        
        public LogWrapper(Log dummyLogger){
            this(null, dummyLogger);
        }
        
        public LogWrapper(Log logger, Log dummyLogger){
            this.logger = logger;
            this.dummyLogger = dummyLogger;
            if(logger != null){
                currentLogger = logger;
            }else{
                currentLogger = dummyLogger;
            }
        }
        
        public void setRealLog(Log logger){
            LogWrapper.this.logger = logger;
        }
        
        public void real(){
            currentLogger = logger;
        }
        
        public void dummy(){
            currentLogger = dummyLogger;
        }
        
        public void trace(Object message){
            currentLogger.trace(message);
        }
        
        public void trace(Object message, Throwable t){
            currentLogger.trace(message, t);
        }
        
        public void debug(Object message){
            currentLogger.debug(message);
        }
        
        public void debug(Object message, Throwable t){
            currentLogger.debug(message, t);
        }
        
        public void info(Object message){
            currentLogger.info(message);
        }
        
        public void info(Object message, Throwable t){
            currentLogger.info(message, t);
        }
        
        public void warn(Object message){
            currentLogger.warn(message);
        }
        
        public void warn(Object message, Throwable t){
            currentLogger.warn(message, t);
        }
        
        public void error(Object message){
            currentLogger.error(message);
        }
        
        public void error(Object message, Throwable t){
            currentLogger.error(message, t);
        }
        
        public void fatal(Object message){
            currentLogger.fatal(message);
        }
        
        public void fatal(Object message, Throwable t) {
            currentLogger.fatal(message, t);
        }
        
        public boolean isTraceEnabled() {
            return currentLogger.isTraceEnabled();
        }
        
        public boolean isDebugEnabled() {
            return currentLogger.isDebugEnabled();
        }
        
        public boolean isInfoEnabled() {
            return currentLogger.isInfoEnabled();
        }
        
        public boolean isWarnEnabled() {
            return currentLogger.isWarnEnabled();
        }
        
        public boolean isErrorEnabled() {
            return currentLogger.isErrorEnabled();
        }
        
        public boolean isFatalEnabled() {
            return currentLogger.isFatalEnabled();
        }
    }
}