/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.service.jndi;

import java.io.StringWriter;
import java.io.PrintWriter;
import java.util.*;
import javax.naming.*;

import jp.ossc.nimbus.core.*;
import jp.ossc.nimbus.daemon.*;
import jp.ossc.nimbus.service.cache.*;
import jp.ossc.nimbus.service.distribute.*;

/**
 * [gIuWFNgLbVtJndiFinderT[rXB<p>
 * JNDIIntialContext̏vpeB𑮐ƂĐݒłB<br>
 * ܂A{@link jp.ossc.nimbus.service.cache.CacheMap LbV}bv}T[rX𑮐ƂĐݒ肷ƁAJndiFinderlookup[gIuWFNgLbV鎖łB<br>
 * ɁAlookup̒ʐMG[̃gCAJNDIT[o̒IȐ`FbNȂǂ̋@\B<br>
 * <p>
 * ȉɁAT[rX`B<br>
 * <pre>
 * &lt;?xml version="1.0" encoding="Shift_JIS"?&gt;
 * 
 * &lt;nimbus&gt;
 *     
 *     &lt;manager name="Sample"&gt;
 *         
 *         &lt;service name="JndiFinder"
 *                  code="jp.ossc.nimbus.service.jndi.DefaultJndiFinderService"&gt;
 *             &lt;attribute name="Environment"&gt;
 *                 java.naming.factory.initial=org.jnp.interfaces.NamingContextFactory
 *                 java.naming.factory.url.pkgs=org.jboss.naming:org.jnp.interfaces
 *                 java.naming.provider.url=localhost
 *             &lt;/attribute&gt;
 *         &lt;/service&gt;
 *         
 *     &lt;/manager&gt;
 *     
 * &lt;/nimbus&gt;
 * </pre>
 * 
 * @author Y.Tokuda
 * @see CacheMap
 */
public class DefaultJndiFinderService extends ServiceBase
 implements JndiFinder, KeepAliveChecker, DefaultJndiFinderServiceMBean{
    
    private static final long serialVersionUID = 2361330897642105726L;
    
    /**
     * JNDIT[omFpJNDIB<p>
     */
    private static final String ROOT_CONTEXT = "/";
    
    /**
     * JNDI IntialContextvpeBB<p>
     */
    private Properties contextEnv;
    
    /**
     * JNDIReLXgB<p>
     */
    private InitialContext initialCtx;
    
    /**
     * [gIuWFNgLbVT[rXB<p>
     */
    private ServiceName remoteObjCacheServiceName;
    
    /**
     * [gIuWFNgLbVT[rXB<p>
     */
    private CacheMap<String, Object> remoteObjCache;
    
    /**
     * JNDIvtBNXB<p>
     * ftHg͋󕶎B<br>
     */
    private String jndiPrefix = "";
    
    /**
     * lookupG[̃gC񐔁B<p>
     * ftHǵAgCȂB<br>
     */
    private int lookupRetryCount = 0;
    
    /**
     * lookupG[̃gCԊu [msec]B<p>
     * ftHǵA1bB<br>
     */
    private long retryInterval = 1000;
    
    /**
     * gCΏۂ̗ONXzB<p>
     */
    private Class<Throwable>[] retryExceptionClasses = DEFAULT_RETRY_EXCXEPTION;
    
    /**
     * JNDIT[o̐mF邩ǂ̃tOB<p>
     */
    private boolean isAliveCheckJNDIServer;
    
    /**
     * JNDIT[o̐Ă邩ǂ̃tOB<p>
     */
    private boolean isAliveJNDIServer;
    
    /**
     * JNDIT[o̐mFԊu[msec]B<p>
     */
    private long aliveCheckJNDIServerInterval = 60000;
    
    /**
     * {@link Daemon}IuWFNgB<p>
     */
    private Daemon daemon;
    
    private boolean isLoggingDeadJNDIServer = true;
    
    private boolean isLoggingRecoverJNDIServer = true;
    
    private String deadJNDIServerLogMessageId = JNDI_SERVER_DEAD_MSG_ID;
    
    private String recoverJNDIServerLogMessageId = JNDI_SERVER_RECOVER_MSG_ID;
    
    private List<KeepAliveListener> keepAliveListeners;
    
    /**
     * T[rX̐sB<p>
     *
     * @exception Exception T[rX̐Ɏsꍇ
     */
    public void createService() throws Exception{
        daemon = new Daemon(new KeepAliveCheckDaemon());
        daemon.setName("Nimbus JndiCheckDaemon " + getServiceNameObject());
        keepAliveListeners = new ArrayList<KeepAliveListener>();
    }
    
    /**
     * CacheMapݒ肷B
     */
    public void setCacheMap(CacheMap<String, Object> remoteObjCache) {
        this.remoteObjCache = remoteObjCache;
    }

    /**
     * T[rX̊JnsB<p>
     *
     * @exception Exception T[rX̊JnɎsꍇ
     */
    public void startService() throws Exception{
        
        //LbVT[rX̎擾
        if(remoteObjCacheServiceName != null){
            remoteObjCache = ServiceManagerFactory
                .getServiceObject(remoteObjCacheServiceName);
        }
        
        if(contextEnv == null){
            initialCtx = new InitialContext();
        }else{
            initialCtx = new InitialContext(contextEnv);
        }
        
        isAliveJNDIServer = true;
        
        if(isAliveCheckJNDIServer){
            // f[N
            daemon.start();
        }
    }
    
    /**
     * T[rX̒~sB<p>
     *
     * @exception Exception T[rX̒~Ɏsꍇ
     */
    public void stopService() throws Exception{
        
        // f[~
        daemon.stop();
        
        initialCtx.close();
    }
    
    /**
     * T[rX̔jsB<p>
     *
     * @exception Exception T[rX̔jɎsꍇ
     */
    public void destory() throws Exception{
        initialCtx = null;
        remoteObjCache = null;
        contextEnv = null;
        remoteObjCacheServiceName = null;
        retryExceptionClasses = null;
        daemon = null;
        keepAliveListeners = null;
    }
    
    // JndiFinderJavaDoc
    public <T> T lookup() throws NamingException{
        return this.<T>lookup("");
    }
    
    private boolean isRetryException(NamingException e){
        if(retryExceptionClasses != null && retryExceptionClasses.length != 0){
            for(int i = 0; i < retryExceptionClasses.length; i++){
                if(retryExceptionClasses[i].isInstance(e)){
                    return true;
                }
            }
        }
        return false;
    }
    
    // JndiFinderJavaDoc
    @SuppressWarnings("unchecked")
    public <T> T lookup(String name) throws NamingException{
        T result = null;
        String key = jndiPrefix + name;
        
        //LbVT[rXݒ肳ĂȂALbVT[rXT
        if(remoteObjCache != null){
            result = (T)remoteObjCache.get(key);
            if(result != null){
                return result;
            }
        }
        
        result = (T)lookupInternal(key);
        
        // lookupŃReLXg擾łALbV[hł΁A
        // 擾ContextLbV
        if(result != null && remoteObjCache != null){
            remoteObjCache.put(key, result);
        }
        
        return result;
    }
    
    private Object lookupInternal(String key) throws NamingException{
        Object result = null;
        
        try{
            result = initialCtx.lookup(key);
        }catch(NamingException e){
            //ԂăgCB
            if(lookupRetryCount <= 0 || !isRetryException(e)){
                throw e;
            }
        }
        
        if(result == null){
            for(int rcont = 0; rcont < lookupRetryCount; rcont++){
                //gCsleep
                try{
                    Thread.sleep(retryInterval);
                }catch(InterruptedException e){}
                
                try{
                    result = initialCtx.lookup(key);
                    break;
                }catch(NamingException e){
                    //ԂăgCB
                    if(rcont == lookupRetryCount - 1
                         || !isRetryException(e)){
                        throw e;
                    }
                }
            }
        }
        
        return result; 
    }
    
    // KeepAliveCheckerJavaDoc
    public void addKeepAliveListener(KeepAliveListener listener){
        synchronized(keepAliveListeners){
            keepAliveListeners.add(listener);
        }
    }
    
    // KeepAliveCheckerJavaDoc
    public void removeKeepAliveListener(KeepAliveListener listener){
        synchronized(keepAliveListeners){
            keepAliveListeners.remove(listener);
        }
    }
    
    // KeepAliveCheckerJavaDoc
    public void clearKeepAliveListener(){
        synchronized(keepAliveListeners){
            keepAliveListeners.clear();
        }
    }
    
    protected class KeepAliveCheckDaemon implements DaemonRunnable<Object>{
        
        /**
         * f[JnɌĂяoB<p>
         * 
         * @return trueԂ
         */
        public boolean onStart(){
            return true;
        }
        
        /**
         * f[~ɌĂяoB<p>
         * 
         * @return trueԂ
         */
        public boolean onStop(){
            return true;
        }
        
        /**
         * f[fɌĂяoB<p>
         * 
         * @return trueԂ
         */
        public boolean onSuspend(){
            return true;
        }
        
        /**
         * f[ĊJɌĂяoB<p>
         * 
         * @return trueԂ
         */
        public boolean onResume(){
            return true;
        }
        
        /**
         * 莞sleepɃ[gReLXglookupĕԂB<p>
         * 
         * @param ctrl DaemonControlIuWFNg
         * @return [gReLXgIuWFNg܂NamingException
         */
        public Object provide(DaemonControl ctrl){
            try{
                Thread.sleep(aliveCheckJNDIServerInterval);
            }catch(InterruptedException e){
                Thread.interrupted();
            }
            if(isAliveCheckJNDIServer){
                try{
                    return lookupInternal(ROOT_CONTEXT);
                }catch(NamingException e){
                    return e;
                }
            }else{
                return null;
            }
        }
        
        /**
         * lookupedObjœnꂽIuWFNgB<p>
         * isAliveJNDIServertruȅԂŁAlookupedObjœnꂽIuWFNgNamingException̏ꍇAJNDIT[o񂾎|̃G[Oo͂B<br>
         * isAliveJNDIServerfalsȅԂŁAlookupedObjœnꂽIuWFNgNamingExceptionłȂꍇAJNDIT[oA|̒ʒmOo͂B<br>
         *
         * @param lookupedObj [gReLXgIuWFNg
         * @param ctrl DaemonControlIuWFNg
         */
        public void consume(Object lookupedObj, DaemonControl ctrl){
            if(!isAliveCheckJNDIServer){
                return;
            }
            if(isAliveJNDIServer){
                if(lookupedObj instanceof NamingException){
                    isAliveJNDIServer = false;
                    clearCache();
                    synchronized(keepAliveListeners){
                        for(int i = 0, imax = keepAliveListeners.size(); i < imax; i++){
                            final KeepAliveListener keepAliveListener
                                 = keepAliveListeners.get(i);
                            keepAliveListener.onDead(DefaultJndiFinderService.this);
                        }
                    }
                    // G[Oo
                    if(isLoggingDeadJNDIServer){
                        getLogger().write(
                            deadJNDIServerLogMessageId,
                            getJNDIServerInfo(),
                            (NamingException)lookupedObj
                        );
                    }
                }
            }else{
                if(!(lookupedObj instanceof NamingException)){
                    isAliveJNDIServer = true;
                    synchronized(keepAliveListeners){
                        for(int i = 0, imax = keepAliveListeners.size(); i < imax; i++){
                            final KeepAliveListener keepAliveListener
                                 = keepAliveListeners.get(i);
                            keepAliveListener.onRecover(DefaultJndiFinderService.this);
                        }
                    }
                    if(isLoggingRecoverJNDIServer){
                        // ʒmOo
                        getLogger().write(
                            recoverJNDIServerLogMessageId,
                            getJNDIServerInfo()
                        );
                    }
                }
            }
        }
        
        private Object getJNDIServerInfo(){
            Object result = null;
            try{
                result = getEnvironment().get("java.naming.provider.url");
            }catch(NamingException e){
            }
            if(result == null){
                result = "localhost";
            }
            return result;
        }
        
        /**
         * ȂB<p>
         */
        public void garbage(){
        }
    }
    
    // CachedJndiFinderServiceMBeanJavaDoc
    public void setEnvironment(Properties prop){
        contextEnv = prop;
    }
    
    // CachedJndiFinderServiceMBeanJavaDoc
    public Properties getEnvironment() throws NamingException{
        if(contextEnv != null){
            return contextEnv;
        }else if(initialCtx != null){
            final Properties prop = new Properties();
            prop.putAll(initialCtx.getEnvironment());
            return prop;
        }
        return null;
    }
    
    // CachedJndiFinderServiceMBeanJavaDoc
    public void setCacheMapServiceName(ServiceName name){
        remoteObjCacheServiceName = name;
    }
    
    // CachedJndiFinderServiceMBeanJavaDoc
    public ServiceName getCacheMapServiceName(){
        return remoteObjCacheServiceName;
    }
    
    // CachedJndiFinderServiceMBeanJavaDoc
    public void setPrefix(String prefix){
        jndiPrefix = prefix;
    }
    
    // CachedJndiFinderServiceMBeanJavaDoc
    public String getPrefix(){
        return jndiPrefix;
    }
    
    // CachedJndiFinderServiceMBeanJavaDoc
    public void setRetryCount(int num){
        lookupRetryCount = num;
    }
    
    // CachedJndiFinderServiceMBeanJavaDoc
    public int getRetryCount(){
        return lookupRetryCount;
    }
    
    // CachedJndiFinderServiceMBeanJavaDoc
    public void setRetryInterval(long interval){
        retryInterval = interval;
    }
    
    // CachedJndiFinderServiceMBeanJavaDoc
    public long getRetryInterval(){
        return retryInterval;
    }
    
    // CachedJndiFinderServiceMBeanJavaDoc
    public void setRetryExceptionClasses(Class<Throwable>[] classes){
        retryExceptionClasses = classes;
    }
    
    // CachedJndiFinderServiceMBeanJavaDoc
    public Class<Throwable>[] getRetryExceptionClasses(){
        return retryExceptionClasses;
    }
    
    // CachedJndiFinderServiceMBeanJavaDoc
    public void setAliveCheckJNDIServer(boolean isCheck){
        isAliveCheckJNDIServer = isCheck;
        if(isCheck && getState() == State.STARTED && !daemon.isRunning()){
            daemon.start();
        }
    }
    
    // CachedJndiFinderServiceMBeanJavaDoc
    public boolean isAliveCheckJNDIServer(){
        return isAliveCheckJNDIServer;
    }
    
    // CachedJndiFinderServiceMBeanJavaDoc
    public void setAliveCheckJNDIServerInterval(long interval){
        aliveCheckJNDIServerInterval = interval;
    }
    
    // CachedJndiFinderServiceMBeanJavaDoc
    public long getAliveCheckJNDIServerInterval(){
        return aliveCheckJNDIServerInterval;
    }
    
    // CachedJndiFinderServiceMBeanJavaDoc
    public boolean isAliveJNDIServer(){
        if(getState() != State.STARTED){
            return false;
        }else if(isAliveCheckJNDIServer){
            return isAliveJNDIServer;
        }else{
            try{
                lookupInternal(ROOT_CONTEXT);
                return true;
            }catch(NamingException e){
                return false;
            }
        }
    }
    
    // KeepAliveCheckerJavaDoc
    public boolean isAlive(){
        return isAliveJNDIServer();
    }
    
    // CachedJndiFinderServiceMBeanJavaDoc
    public void setLoggingDeadJNDIServer(boolean isOutput){
        isLoggingDeadJNDIServer = isOutput;
    }
    
    // CachedJndiFinderServiceMBeanJavaDoc
    public boolean isLoggingDeadJNDIServer(){
        return isLoggingDeadJNDIServer;
    }
    
    // CachedJndiFinderServiceMBeanJavaDoc
    public void setLoggingRecoverJNDIServer(boolean isOutput){
        isLoggingRecoverJNDIServer = isOutput;
    }
    
    // CachedJndiFinderServiceMBeanJavaDoc
    public boolean isLoggingRecoverJNDIServer(){
        return isLoggingRecoverJNDIServer;
    }
    
    // CachedJndiFinderServiceMBeanJavaDoc
    public void setDeadJNDIServerLogMessageId(String id){
        deadJNDIServerLogMessageId = id;
    }
    
    // CachedJndiFinderServiceMBeanJavaDoc
    public String getDeadJNDIServerLogMessageId(){
        return deadJNDIServerLogMessageId;
    }
    
    // CachedJndiFinderServiceMBeanJavaDoc
    public void setRecoverJNDIServerLogMessageId(String id){
        recoverJNDIServerLogMessageId = id;
    }
    
    // CachedJndiFinderServiceMBeanJavaDoc
    public String getRecoverJNDIServerLogMessageId(){
        return recoverJNDIServerLogMessageId;
    }
    
    // CachedJndiFinderServiceMBeanJavaDoc
    public void clearCache(){
        if(remoteObjCache != null){
            remoteObjCache.clear();
        }
    }
    
    // CachedJndiFinderServiceMBeanJavaDoc
    public void clearCache(String name){
        if(remoteObjCache != null){
            remoteObjCache.remove(name);
        }
    }
    
    // CachedJndiFinderServiceMBeanJavaDoc
    public String listContext() throws NamingException{
        if(initialCtx == null){
            return null;
        }
        StringWriter sw = new StringWriter();
        PrintWriter pw = new PrintWriter(sw);
        pw.println("<pre>");
        listContext(pw, initialCtx, "@");
        pw.println("</pre>");
        return sw.toString();
    }
    
    private void listContext(PrintWriter pw, Context context, String indent) throws NamingException{
        NamingEnumeration<Binding> list = context.listBindings("");
        while(list.hasMore()){
            Binding item = list.next();
            String className = item.getClassName();
            String name = item.getName();
            pw.println(indent + className + "@" + name);
            Object o = item.getObject();
            if(o instanceof javax.naming.Context){
                listContext(pw, (Context)o, indent + "@");
            }
        }
    }
}
