/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.service.cache;

import java.util.*;
import java.lang.reflect.*;

import jp.ossc.nimbus.core.*;
import jp.ossc.nimbus.beans.*;

/**
 * vZTCYӂꌟ؃T[rXB<p>
 * ȉɁA̎gpTCYő僁̔𒴂Ɨ\zꂽꍇɂӂ邠ӂꌟ؃T[rX̃T[rX`B<br>
 * <pre>
 * &lt;?xml version="1.0" encoding="Shift_JIS"?&gt;
 * 
 * &lt;nimbus&gt;
 *     
 *     &lt;manager name="Sample"&gt;
 *         
 *         &lt;service name="CalculateMemorySizeOverflowValidator"
 *                  code="jp.ossc.nimbus.service.cache.CalculateMemorySizeOverflowValidatorService"/&gt;
 *         
 *     &lt;/manager&gt;
 *     
 * &lt;/nimbus&gt;
 * </pre>
 * {@link MemorySizeOverflowValidatorService}ƈقȂ̂́AJVM̃q[vTCYłӂꌟ؂̂ł͂ȂALbVꂽIuWFNggp郁TCY̗_lłӂꌟ؂ƂłB<br>
 * IuWFNggp郁TCY̗_ľvZ@́Aȉ̒ʂłB<br>
 * <ul>
 *   <li>{@link #setMemorySize(String, String)}Ŏw肳ꂽ_lꍇAgpB</li>
 *   <li>NXɐ錾ĂSẴCX^XϐA̐錾ĂNX̏găTCYvZĉZB́AtB[h錾ɕKvȃvZ邾Ȃ̂ŁA̕ϐ̒l̂̃gpʂ͌vZȂB</li>
 *   <li>IuWFNgɐ錾Ăgetter\bhŎ擾łprimitive^ȊÕIuWFNg̃TCYvZĉZB{@link #setCalculateProperty(boolean)}ł̉ZON/OFF𐧌łBCX^Xϐ̒l̂̃gpʂZBAAgetter݂ȂCX^Xϐ̒l̂̃gpʂ͉ZȂB</li>
 *   <li>z^̃IuWFNǵAz * 4 + 12oCgƂČvZBAACX^XɃANZXłȂꍇ́Az񒷂0Ɖ肷B܂ACX^XɃANZXłꍇ́Aezvf̃IuWFNg̃TCYZB</li>
 *   <li>java.util.Collection^java.util.Map^̃IuWFNǵACX^XɃANZXłꍇ͊evf̃IuWFNg̃TCYZB</li>
 * </ul>
 * ̗_ĺAtNVAPIŒׂ鎖ł͈͂܂łAvZȂ߁AKgpʂƈvȂBXɂāAgpʂȒlɂȂB]āÃT[rXgpꍇ́AL̗_lłǂ܂ŎgpʂƂ̂ꂪ邩z肷KvB<br>
 *
 * @author M.Takata
 */
public class CalculateMemorySizeOverflowValidatorService<E> extends ServiceBase
 implements OverflowValidator<E>, CacheRemoveListener<E>, java.io.Serializable,
            CalculateMemorySizeOverflowValidatorServiceMBean{
    
    private static final long serialVersionUID = 6454857430979865088L;
    
    private static final char KILO_UNIT = 'K';
    private static final char MEGA_UNIT = 'M';
    private static final char GIGA_UNIT = 'G';
    
    private static final long KILO_BYTE = 1024;
    private static final long MEGA_BYTE = KILO_BYTE * KILO_BYTE;
    private static final long GIGA_BYTE = MEGA_BYTE * KILO_BYTE;
    
    
    private String maxMemorySizeStr = "32M";
    private long maxMemorySize = 32 * MEGA_BYTE;
    
    private Map<CachedReference<E>, Long> references;
    private Map<Class<?>, Number> memorySizeMap;
    private Map<Class<?>, Number> tmpMemorySizeMap;
    private long currentUsedMemorySize;
    private boolean isCalculateOnValidate;
    private boolean isCalculateProperty;
    
    {
        final Runtime runtime = Runtime.getRuntime();
        try{
            maxMemorySize = runtime.maxMemory() / 2;
            maxMemorySizeStr = Long.toString(maxMemorySize);
        }catch(NoSuchMethodError err){
        }
    }
    
    // CalculateMemorySizeOverflowValidatorServiceMBeanJavaDoc
    @Override
    public void setMaxMemorySize(String size)
     throws IllegalArgumentException{
        maxMemorySize = convertMemorySize(size);
        maxMemorySizeStr = size;
    }
    
    // CalculateMemorySizeOverflowValidatorServiceMBeanJavaDoc
    @Override
    public String getMaxMemorySize(){
        return maxMemorySizeStr;
    }
    
    // CalculateMemorySizeOverflowValidatorServiceMBeanJavaDoc
    @Override
    public void setCalculateProperty(boolean isCalculate){
        isCalculateProperty = isCalculate;
    }
    
    // CalculateMemorySizeOverflowValidatorServiceMBeanJavaDoc
    @Override
    public boolean isCalculateProperty(){
        return isCalculateProperty;
    }
    
    // CalculateMemorySizeOverflowValidatorServiceMBeanJavaDoc
    @Override
    public void setCalculateOnValidate(boolean isCalculate){
        isCalculateOnValidate = isCalculate;
    }
    
    // CalculateMemorySizeOverflowValidatorServiceMBeanJavaDoc
    @Override
    public boolean isCalculateOnValidate(){
        return isCalculateOnValidate;
    }
    
    // CalculateMemorySizeOverflowValidatorServiceMBeanJavaDoc
    @Override
    public int size(){
        return references == null ? 0 : references.size();
    }
    
    // CalculateMemorySizeOverflowValidatorServiceMBeanJavaDoc
    @Override
    public long getCurrentUsedMemorySize(){
        return currentUsedMemorySize;
    }
    
    // CalculateMemorySizeOverflowValidatorServiceMBeanJavaDoc
    @Override
    public void setMemorySize(String className, String size)
     throws ClassNotFoundException{
        final Class<?> clazz = Class.forName(
            className,
            true,
            NimbusClassLoader.getInstance()
        );
        long val = convertMemorySize(size);
        memorySizeMap.put(clazz, new Long(val));
    }
    
    // CalculateMemorySizeOverflowValidatorServiceMBeanJavaDoc
    @Override
    public String getMemorySize(String className) throws ClassNotFoundException{
        if(memorySizeMap == null){
            return null;
        }
        final Class<?> clazz = Class.forName(
            className,
            true,
            NimbusClassLoader.getInstance()
        );
        Number number = (Number)memorySizeMap.get(clazz);
        return number == null ? null : String.valueOf(number.longValue());
    }
    
    // CalculateMemorySizeOverflowValidatorServiceMBeanJavaDoc
    @Override
    public Map<Class<?>, Number> getMemorySizeMap(){
        return memorySizeMap;
    }
    
    /**
     * T[rX̐sB<p>
     * CX^Xϐ̏sB
     *
     * @exception Exception T[rX̐Ɏsꍇ
     */
    @Override
    public void createService() throws Exception{
        references = Collections.synchronizedMap(new HashMap<CachedReference<E>, Long>());
        memorySizeMap = Collections.synchronizedMap(new HashMap<Class<?>, Number>());
        memorySizeMap.put(Byte.TYPE, new Short((short)1));
        memorySizeMap.put(Boolean.TYPE, new Short((short)1));
        memorySizeMap.put(Character.TYPE, new Short((short)2));
        memorySizeMap.put(Short.TYPE, new Short((short)2));
        memorySizeMap.put(Integer.TYPE, new Short((short)4));
        memorySizeMap.put(Float.TYPE, new Short((short)4));
        memorySizeMap.put(Long.TYPE, new Short((short)8));
        memorySizeMap.put(Double.TYPE, new Short((short)8));
        memorySizeMap.put(Class.class, new Short((short)0));
        memorySizeMap.put(Method.class, new Short((short)0));
        memorySizeMap.put(Field.class, new Short((short)0));
        memorySizeMap.put(Constructor.class, new Short((short)0));
        tmpMemorySizeMap = Collections.synchronizedMap(new HashMap<Class<?>, Number>());
    }
    
    /**
     * T[rX̒~sB<p>
     *
     * @exception Exception T[rX̒~Ɏsꍇ
     */
    @Override
    public void stopService() throws Exception{
        tmpMemorySizeMap.clear();
    }
    
    /**
     * T[rX̔jsB<p>
     * CX^Xϐ̊JsB
     *
     * @exception Exception T[rX̔jɎsꍇ
     */
    @Override
    public void destroyService() throws Exception{
        reset();
        references = null;
    }
    
    private long convertMemorySize(String size)
     throws IllegalArgumentException{
        long value = 0L;
        boolean isValid = true;
        
        if(size == null){
            isValid = false;
        }else{
            final int length = size.length();
            if(length == 0){
                isValid = false;
            }else{
                final char unit = Character.toUpperCase(
                    size.charAt(length - 1)
                );
                String tmpSize = null;
                long unitValue = 0;
                switch(unit){
                case KILO_UNIT:
                    tmpSize = size.substring(0, length - 1);
                    unitValue = KILO_BYTE;
                    break;
                case MEGA_UNIT:
                    tmpSize = size.substring(0, length - 1);
                    unitValue = MEGA_BYTE;
                    break;
                case GIGA_UNIT:
                    tmpSize = size.substring(0, length - 1);
                    unitValue = GIGA_BYTE;
                    break;
                default:
                    tmpSize = size;
                    unitValue = 1;
                }
                try{
                    value = (long)(Double.parseDouble(tmpSize) * (long)unitValue);
                }catch(NumberFormatException e){
                    isValid = false;
                }
            }
        }
        if(value < 0){
            isValid = false;
        }
        if(!isValid){
            throw new IllegalArgumentException("Invalid size : " + size);
        }
        return value;
    }
    
    private long calculateMemorySize(Object obj){
        if(obj == null){
            return 0l;
        }
        final Class<?> clazz = obj.getClass();
        return calculateMemorySize(clazz, obj, true, new ArrayList<Class<?>>());
    }
    
    private long roundUp(long val, int base){
        long tmp = val % base;
        return tmp == 0 ? val : (val + base - tmp);
    }
    
    @SuppressWarnings("unchecked")
    private long calculateMemorySize(Class<?> clazz, Object obj, boolean isRoundUp, List<Class<?>> stack){
        if(stack.contains(clazz)){
            return 0l;
        }
        int index = 0;
        try{
            index = stack.size();
            stack.add(clazz);
            Number size = (Number)memorySizeMap.get(clazz);
            if(size != null){
                return size.longValue();
            }
            if(obj == null){
                size = (Number)tmpMemorySizeMap.get(clazz);
                if(size != null){
                    return size.longValue();
                }
            }
            long result = 0l;
            if(clazz.isInterface()){
                // CX^X̌^Ȃ̂Œ߂
                result = 8l;
            }else if(clazz.isArray()){
                if(obj == null){
                    // z̃TCYȂ̂Œ0ƂĒ߂
                    result = 12l;
                }else{
                    final int length = Array.getLength(obj);
                    result = length * 4 + 12l;
                    result = roundUp(result, 8);
                    for(int i = 0; i < length; i++){
                        Object element = Array.get(obj, i);
                        if(element != null){
                            result += calculateMemorySize(
                                element.getClass(),
                                element,
                                false,
                                stack
                            );
                        }
                    }
                }
                result = roundUp(result, 8);
            }else if(String.class.equals(clazz)){
                result = calculateMemorySize(clazz, null, true, stack);
                if(obj != null){
                    final String str = (String)obj;
                    result += str.length() * (4l + 2l);
                    result = roundUp(result, 8);
                }
            }else if(Collection.class.isAssignableFrom(clazz)){
                result = calculateMemorySize(clazz, null, true, stack);
                if(obj != null){
                    final Collection<Object> col = (Collection<Object>)obj;
                    if(col.size() != 0){
                        try{
                            for(Object element : col){
                                if(element != null){
                                    result += calculateMemorySize(
                                        element.getClass(),
                                        element,
                                        true,
                                        stack
                                    );
                                }
                            }
                        }catch(ConcurrentModificationException e){}
                    }
                }
            }else if(Map.class.isAssignableFrom(clazz)){
                result = calculateMemorySize(clazz, null, true, stack);
                if(obj != null){
                    final Map<Object, Object> map = (Map<Object, Object>)obj;
                    if(map.size() != 0){
                        try{
                            for(Map.Entry<Object, Object> entry : map.entrySet()){
                                Object key = entry.getKey();
                                if(key != null){
                                    result += calculateMemorySize(
                                        key.getClass(),
                                        key,
                                        true,
                                        stack
                                    );
                                }
                                Object value = entry.getValue();
                                if(value != null){
                                    result += calculateMemorySize(
                                        value.getClass(),
                                        value,
                                        true,
                                        stack
                                    );
                                }
                            }
                        }catch(ConcurrentModificationException e){}
                    }
                }
            }else{
                Class<?> tmpClass = clazz;
                result += 8;
                do{
                    final Field[] fields = tmpClass.getDeclaredFields();
                    if(fields != null){
                        for(int i = 0; i < fields.length; i++){
                            final Field field = fields[i];
                            if(Modifier.isStatic(field.getModifiers())){
                                continue;
                            }
                            final Class<?> fieldType = field.getType();
                            result += calculateFieldMemorySize(
                                fieldType,
                                true
                            );
                        }
                        if(isRoundUp){
                            result = roundUp(result, 8);
                        }
                    }
                    tmpClass = tmpClass.getSuperclass();
                }while(tmpClass != null);
                if(obj != null && isCalculateProperty){
                    final SimpleProperty[] props
                         = SimpleProperty.getProperties(obj);
                    for(int i = 0; i < props.length; i++){
                        if(!props[i].isReadable(obj)){
                            continue;
                        }
                        Object val = null;
                        Class<?> type = null;
                        try{
                            final Method method = props[i].getReadMethod(obj);
                            if(Modifier.isStatic(method.getModifiers())){
                                continue;
                            }
                            type = props[i].getPropertyType(obj);
                            if(type.isPrimitive()){
                                continue;
                            }
                            val = props[i].getProperty(obj);
                        }catch(InvocationTargetException e){
                        }catch(NoSuchPropertyException e){
                        }
                        if(val != null){
                            result += calculateMemorySize(
                                type,
                                val,
                                true,
                                stack
                            );
                        }
                    }
                }
            }
            if(obj == null){
                tmpMemorySizeMap.put(clazz, new Long(result));
            }
            return result;
        }finally{
            stack.remove(index);
        }
    }
    
    private long calculateFieldMemorySize(Class<?> clazz, boolean isRoundUp){
        if(clazz.isPrimitive()){
            if(Byte.TYPE.equals(clazz) || Boolean.TYPE.equals(clazz)){
                return 1l;
            }else if(Character.TYPE.equals(clazz) || Short.TYPE.equals(clazz)){
                return 2l;
            }else if(Integer.TYPE.equals(clazz) || Float.TYPE.equals(clazz)){
                return 4l;
            }else{
                return 8l;
            }
        }else{
            if(clazz.isArray()){
                // z̃TCYȂ̂Œ0ƂĒ߂
                return 16l;
            }else{
                return 4l;
            }
        }
    }
    
    /**
     * LbVQƂǉB<p>
     * œnꂽLbVQƂێBɁA{@link CachedReference#addCacheRemoveListener(CacheRemoveListener)}ŁA{@link CacheRemoveListener}ƂĎgo^B<br>
     *
     * @param ref LbVQ
     */
    @Override
    public void add(CachedReference<E> ref){
        if(references == null || ref == null){
            return;
        }
        synchronized(references){
            if(!references.containsKey(ref)){
                Long size = null;
                if(!isCalculateOnValidate){
                    size = new Long(calculateMemorySize(ref.get(this)));
                    if(currentUsedMemorySize < 0){
                        currentUsedMemorySize = 0;
                    }
                    currentUsedMemorySize += size.longValue();
                }
                references.put(ref, size);
                ref.addCacheRemoveListener(this);
            }
        }
    }
    
    /**
     * LbVQƂ폜B<p>
     * œnꂽLbVQƂŕێĂꍇ́AjBɁA{@link CachedReference#removeCacheRemoveListener(CacheRemoveListener)}ŁA{@link CacheRemoveListener}ƂĎgo^B<br>
     *
     * @param ref LbVQ
     */
    @Override
    public void remove(CachedReference<E> ref){
        if(references == null || ref == null){
            return;
        }
        synchronized(references){
            if(references.containsKey(ref)){
                final Long size = (Long)references.remove(ref);
                ref.removeCacheRemoveListener(this);
                if(!isCalculateOnValidate && size != null){
                    currentUsedMemorySize -= size.longValue();
                    if(currentUsedMemorySize < 0){
                        currentUsedMemorySize = 0;
                    }
                }
            }
        }
    }
    
    /**
     * q[v̎gpłӂꌟ؂sB<p>
     * ȉ̌vZŁAӂꐔvZBAAvZʂ̏ꍇ́A0ƂB<br>
     * LbVTCY~igpq[v]׃q[vjiőq[v]׃q[vj
     *
     * @return ӂꌟ؂sʂӂꂪꍇAӂꐔԂBӂȂꍇ́A0Ԃ
     */
    @Override
    public int validate(){
        if(references == null || references.size() == 0){
            return 0;
        }
        synchronized(references){
            if(getState() != State.STARTED){
                return 0;
            }
            if(isCalculateOnValidate){
                synchronized(references){
                    currentUsedMemorySize = 0;
                    for(Map.Entry<CachedReference<E>, Long> entry : references.entrySet()){
                        final CachedReference<E> ref = entry.getKey();
                        final long size = calculateMemorySize(ref.get(this));
                        if(currentUsedMemorySize < 0){
                            currentUsedMemorySize = 0;
                        }
                        currentUsedMemorySize += size;
                    }
                }
            }
            
            if(currentUsedMemorySize < maxMemorySize){
                return 0;
            }
            double usedAverage = currentUsedMemorySize / references.size();
            long overflowMemorySize = currentUsedMemorySize - maxMemorySize;
            final double overflowSize = overflowMemorySize / usedAverage;
            return overflowSize > 0.0 ? (int)Math.ceil(overflowSize) : 0;
        }
    }
    
    /**
     * ӂꌟ؂s邽߂ɕێĂB<p>
     * {@link #add(CachedReference)}œnꂽLbVQƂSĔjB<br>
     */
    @Override
    public void reset(){
        if(references != null){
            references.clear();
        }
        currentUsedMemorySize = 0;
    }
    
    /**
     * LbV폜ꂽLbVQƂ̒ʒm󂯂B<p>
     * {@link #remove(CachedReference)}ĂяoB<br>
     *
     * @param ref LbV폜ꂽLbVQ
     */
    @Override
    public void removed(CachedReference<E> ref){
        remove(ref);
    }
}
