/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.service.cache;

import java.util.*;

import jp.ossc.nimbus.core.ServiceBase;

/**
 * LbVTCYӂꌟ؃T[rXB<p>
 * őLbVłӂ؂OverflowValidatorłB<br>
 * ȉɁALbVTCY10𒴂Ƃӂ邠ӂꌟ؃T[rX̃T[rX`B<br>
 * <pre>
 * &lt;?xml version="1.0" encoding="Shift_JIS"?&gt;
 * 
 * &lt;nimbus&gt;
 *     
 *     &lt;manager name="Sample"&gt;
 *         
 *         &lt;service name="CacheSizeOverflowValidator"
 *                  code="jp.ossc.nimbus.service.cache.CacheSizeOverflowValidatorService"&gt;
 *             &lt;attribute name="MaxSize"&gt;10&lt;/attribute&gt;
 *         &lt;/service&gt;
 *         
 *     &lt;/manager&gt;
 *     
 * &lt;/nimbus&gt;
 * </pre>
 *
 * @author M.Takata
 */
public class CacheSizeOverflowValidatorService<E> extends ServiceBase
 implements OverflowValidator<E>, CacheRemoveListener<E>, java.io.Serializable,
            CacheSizeOverflowValidatorServiceMBean{
    
    private static final long serialVersionUID = -2810585852541528435L;
    
    /**
     * őLbVB<p>
     */
    private int maxSize;
    
    /**
     * ӂ臒lB<p>
     */
    private int overflowThreshold;
    
    /**
     * LbVQƂ̏WB<p>
     */
    private Set<CachedReference<E>> references;
    
    // CacheSizeOverflowValidatorServiceMBeanJavaDoc
    @Override
    public void setMaxSize(int size) throws IllegalArgumentException{
        if(size < 0){
            throw new IllegalArgumentException("Invalid size : " + size);
        }
        maxSize = size;
    }
    
    // CacheSizeOverflowValidatorServiceMBeanJavaDoc
    @Override
    public int getMaxSize(){
        return maxSize;
    }
    
    // CacheSizeOverflowValidatorServiceMBeanJavaDoc
    public void setOverflowThreshold(int threshold){
        overflowThreshold = threshold;
    }
    
    // CacheSizeOverflowValidatorServiceMBeanJavaDoc
    public int getOverflowThreshold(){
        return overflowThreshold;
    }
    
    /**
     * T[rX̐sB<p>
     * CX^Xϐ̏sB<br>
     * 
     * @exception Exception T[rX̐Ɏsꍇ
     */
    @Override
    public void createService() throws Exception{
        references = Collections.synchronizedSet(new HashSet<CachedReference<E>>());
    }
    
    /**
     * T[rX̔jsB<p>
     * {@link #reset()}ĂяoB܂ACX^Xϐ̎QƂjB<br>
     *
     * @exception Exception T[rX̔jɎsꍇ
     */
    @Override
    public void destroyService() throws Exception{
        reset();
        references = null;
    }
    
    /**
     * LbVőLbV𒴂Ă邩؂B<p>
     * őLbV́A{@link #setMaxSize(int)}Őݒ肳ꂽlB<br>
     *
     * @return LbVőLbV𒴂ĂꍇAĂ鐔ԂBĂȂꍇ́A0Ԃ
     */
    @Override
    public int validate(){
        if(references == null || references.size() == 0){
            return 0;
        }
        synchronized(references){
            if(getState() != State.STARTED){
                return 0;
            }
            if(getMaxSize() == 0){
                return 0;
            }
            int overflowSize = references.size() - getMaxSize();
            if(overflowSize > 0 && getOverflowThreshold() > 0){
                overflowSize = (getMaxSize() - getOverflowThreshold());
            }
            return overflowSize > 0 ? overflowSize : 0;
        }
    }
    
    // OverflowValidatorJavaDoc
    @Override
    public void add(CachedReference<E> ref){
        if(references == null || ref == null){
            return;
        }
        synchronized(references){
            if(!references.contains(ref)){
                references.add(ref);
                ref.addCacheRemoveListener(this);
            }
        }
    }
    
    // OverflowValidatorJavaDoc
    @Override
    public void remove(CachedReference<E> ref){
        if(references == null || ref == null){
            return;
        }
        synchronized(references){
            if(references.contains(ref)){
                references.remove(ref);
                ref.removeCacheRemoveListener(this);
            }
        }
    }
    
    // OverflowValidatorJavaDoc
    @SuppressWarnings("unchecked")
    @Override
    public void reset(){
        if(references != null){
            synchronized(references){
                final CachedReference<E>[] refs = references.toArray(new CachedReference[references.size()]);
                for(CachedReference<E> ref : refs){
                    remove(ref);
                }
                references.clear();
            }
        }
    }
    
    // CacheSizeOverflowValidatorServiceMBeanJavaDoc
    @Override
    public int size(){
        return references == null ? 0 : references.size();
    }
    
    /**
     * {@link #add(CachedReference)}Œǉꂽ{@link CachedReference}̃LbVIuWFNg폜ꂽꍇɌĂяoB<p>
     * 폜ꂽLbVQƂ{@link #remove(CachedReference)}ŁAOverflowValidator폜B<br>
     *
     * @param ref 폜ꂽLbVIuWFNg̃LbVQ
     */
    @Override
    public void removed(CachedReference<E> ref){
        if(references == null){
            return;
        }
        remove(ref);
    }
}
