#
# Cookbook Name:: apt_utils
# Recipe Spec:: apt-cacher-ng_spec
#
# Copyright 2015-2017, whitestar
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#

require_relative '../spec_helper'

describe 'apt_utils::apt-cacher-ng' do
  let(:chef_run_on_non_debian) {
    ChefSpec::SoloRunner.new(platform: 'centos', version: '7.0') {|node|
      node.set['apt_utils']['apt-cacher-ng']['CacheDir'] = '/path/to/cache'
      node.set['apt_utils']['apt-cacher-ng']['LogDir'] = '/path/to/log'
    }.converge(described_recipe)
  }

  it 'does nothing on non debian family.' do
    expected = expect(chef_run_on_non_debian)
    expected.to_not install_package('apt-cacher-ng')
    expected.to_not enable_service('apt-cacher-ng')
    expected.to_not start_service('apt-cacher-ng')
    expected.to_not create_directory('/path/to/cache')
    expected.to_not create_directory('/path/to/log')
    expected.to_not render_file('/etc/apt-cacher-ng/acng.conf')
  end

  let(:chef_run_on_debian) {
    ChefSpec::SoloRunner.new(platform: 'debian', version: '8.1') {|node|
      node.set['apt_utils']['apt-cacher-ng']['CacheDir'] = '/path/to/cache'
      node.set['apt_utils']['apt-cacher-ng']['LogDir'] = '/path/to/log'
    }.converge(described_recipe)
  }

  it 'installs apt-cacher-ng package on debian family.' do
    expected = expect(chef_run_on_debian)
    expected.to install_package('apt-cacher-ng')
    expected.to enable_service('apt-cacher-ng')
    expected.to start_service('apt-cacher-ng')
  end

  it 'creates some directries on debian family.' do
    expected = expect(chef_run_on_debian)
    expected.to create_directory('/path/to/cache')
    expected.to create_directory('/path/to/log')
  end

  it 'renders /etc/apt-cacher-ng/acng.conf on debian family.' do
    expect(chef_run_on_debian).to \
      render_file('/etc/apt-cacher-ng/acng.conf')
      .with_content(%r{^CacheDir:\s+/path/to/cache$})
  end

  it 'notifies apt-cacher-ng service\'s restart after rendering on debian family.' do
    expect(chef_run_on_debian.template('/etc/apt-cacher-ng/acng.conf')).to \
      notify('service[apt-cacher-ng]').to(:restart).delayed
  end
end
