#
# Cookbook Name:: krb5
# Recipe Spec:: admin_spec
#
# Copyright 2015, whitestar
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#

require_relative '../spec_helper'

describe 'krb5::admin' do
  let(:chef_run_on_debian) {
    ChefSpec::SoloRunner.new(platform: 'debian', version: '7.6') {|node|
      node.set['krb5']['libdefaults']['default_realm'] = 'GRID.EXAMPLE.COM'
    }.converge(described_recipe)
  }

  it 'installs krb5-admin-server and krb5-kdc packages on debian family.' do
    expected = expect(chef_run_on_debian)
    expected.to install_package('krb5-admin-server')
    expected.to enable_service('krb5-admin-server')
    expected.to start_service('krb5-admin-server')
    expected.to install_package('krb5-kdc')
    expected.to enable_service('krb5-kdc')
    expected.to start_service('krb5-kdc')
  end

  it 'renders /etc/krb5kdc/kdc.conf on debian family.' do
    expect(chef_run_on_debian).to \
      render_file('/etc/krb5kdc/kdc.conf').
        with_content(%r|^\s+GRID\.EXAMPLE\.COM\s+=\s+{$|)
  end

  let(:chef_run_on_rhel) {
    ChefSpec::SoloRunner.new(platform: 'centos', version: '7.0') {|node|
      node.set['krb5']['libdefaults']['default_realm'] = 'GRID.EXAMPLE.COM'
    }.converge(described_recipe)
  }

  it 'installs krb5-server on rhel family.' do
    expected = expect(chef_run_on_rhel)
    expected.to install_package('krb5-server')
    expected.to enable_service('kadmin')
    expected.to enable_service('krb5kdc')
  end

  it 'renders /var/kerberos/krb5kdc/kdc.conf on rhel family.' do
    expected = expect(chef_run_on_rhel)
    expected.to create_template('/var/kerberos/krb5kdc/kdc.conf').with(mode: '0600')
    expected.to \
      render_file('/var/kerberos/krb5kdc/kdc.conf').
        with_content(%r|^\s+GRID\.EXAMPLE\.COM\s+=\s+{$|)
  end

  it 'writes log.' do
    expect(chef_run_on_rhel).to \
      write_log <<-EOM
Note:
You must initialize a Kerberos realm in the first installation:
  [Debian]
    $ sudo krb5_newrealm
    $ sudo service krb5-kdc restart
    $ sudo service krb5-admin-server restart
  [CentOS]
    $ sudo kdb5_util create -s
    $ sudo service kadmin restart
    $ sudo service krb5kdc restart
EOM
  end
end

