/*
 * Copyright (c) 2005- Shinji Kashihara.
 * All rights reserved. This program are made available under
 * the terms of the Eclipse Public License v1.0 which accompanies
 * this distribution, and is available at epl-v10.html.
 */
package jp.sourceforge.mergedoc.pleiades.aspect;

import java.io.IOException;

import javassist.CannotCompileException;
import javassist.CtBehavior;
import javassist.CtClass;
import javassist.CtConstructor;
import javassist.NotFoundException;
import jp.sourceforge.mergedoc.pleiades.aspect.resource.ExcludeClassNameCache;
import jp.sourceforge.mergedoc.pleiades.aspect.resource.TransformedClassCache;
import jp.sourceforge.mergedoc.pleiades.log.Logger;

/**
 * 翻訳処理を埋め込むためのバイトコード変換を行うトランスフォーマーです。
 * <p>
 * @author cypher256
 */
public class TranslationTransformer extends AbstractTransformer {

	/** ロガー */
	@SuppressWarnings("unused")
	private static final Logger log = Logger.getLogger(TranslationTransformer.class);

	/**
	 * バイトコードを変換します。
	 * <p>
	 * {@link TranslationEditor 翻訳エディター} を使用し、バイトコードを変換します。
	 * ただし、{@link ExcludeClassNameCache 変換除外クラス名キャッシュ}
	 * に処理するクラス名が含まれる場合、何も行いません。
	 */
	@Override
	protected byte[] transform(
			String classId,
			String className,
			byte[] bytecode)
		throws CannotCompileException, NotFoundException, IOException {

		// 変換除外クラスの場合は何もしない
		ExcludeClassNameCache excludeList = ExcludeClassNameCache.getInstance();
		if (excludeList.contains(classId)) {
			return null;
		}

		// 変換済みクラス・キャッシュがある場合はそれを返す
		TransformedClassCache classCache = TransformedClassCache.getInstance();
		byte[] cachedBytecode = classCache.get(classId);
		if (cachedBytecode != null) {
			return cachedBytecode;
		}

		// バイトコードに翻訳アスペクトを埋め込み
		CtClass ctClass = createCtClass(bytecode);

		try {
			byte[] transformedBytecode = transformClass(ctClass);

			// 次回起動用の情報を作成
			if (classCache.isActive()) {

				if (transformedBytecode == null) {
					if (!classCache.contains(classId)) {

						// 変換対象外の場合は、変換除外リストに追加
						excludeList.addNextLaunch(classId);
					}
				} else {
					// 変換した場合は、変換済みクラス・キャッシュに追加
					classCache.putNextLaunch(classId, transformedBytecode);
				}
			}
			return transformedBytecode;

		} catch (CannotCompileException e) {

			// 関連クラス未ロードでコンパイル不可、次回変換除外
			excludeList.addNextLaunch(classId);
			throw e;
		}
	}

	/**
	 * クラス・オブジェクトに翻訳アスペクトを埋め込みます。
	 * <p>
	 * @param ctClass 変換対象クラス・オブジェクト
	 * @return 埋め込み後のバイトコード。埋め込み対象でない場合は null。
	 */
	protected byte[] transformClass(CtClass ctClass)
		throws CannotCompileException, NotFoundException, IOException {

		long start = System.nanoTime();

		// コンストラクター、メソッドの変換
		TranslationEditor editor = new TranslationEditor(ctClass);
		for (CtBehavior ctBehavior : ctClass.getDeclaredBehaviors()) {

			// コードを検査し、呼び出し部分を編集
			ctBehavior.instrument(editor);

			// メソッドを編集
			editor.editBehavior(ctBehavior);
		}

		// スタティック・イニシャライザーの変換
		CtConstructor ctInitializer = ctClass.getClassInitializer();
		if (ctInitializer != null) {
			ctInitializer.instrument(editor);
		}

		Analyses.end(TranslationTransformer.class, "transformClass", start);
		return editor.toBytecode();
	}
}
