/*
 * Copyright (c) 2005- Shinji Kashihara.
 * All rights reserved. This program are made available under
 * the terms of the Eclipse Public License v1.0 which accompanies
 * this distribution, and is available at epl-v10.html.
 */
package jp.sourceforge.mergedoc.pleiades;

import java.io.File;
import java.io.UnsupportedEncodingException;
import java.lang.instrument.Instrumentation;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import jp.sourceforge.mergedoc.pleiades.aspect.Analyses;
import jp.sourceforge.mergedoc.pleiades.aspect.LauncherTransformer;
import jp.sourceforge.mergedoc.pleiades.log.Logger;
import jp.sourceforge.mergedoc.pleiades.log.PopupFileLogger;
import jp.sourceforge.mergedoc.pleiades.resource.Files;
import jp.sourceforge.mergedoc.pleiades.resource.PropertySet;

/**
 * Pleiades を起動するためのエージェントです。
 * <p>
 * @author cypher256
 */
public class Pleiades {

	/** ロガー */
	private static Logger log;

	/** Pleiades パッケージ名 */
	public static final String PACKAGE_NAME = Pleiades.class.getPackage().getName();

	/** OS 定数 */
	private static enum OS {
		LINUX, MAC, WINDOWS,
	}

	/** OS */
	private static OS os;

	/** Pleiades 起動オプション */
	private static PleiadesOption pleiadesOption;

	/** コンフィグレーション・パス */
	private static File pleiadesConfigurationPath;

	/** Eclipse ホーム */
	private static File eclipseHome;

	/** p2agent バンドル・ホーム */
	private static File bundleHome;

	/** インストゥルメンテーション */
	private static Instrumentation instrumentation;

	/**
	 * 起動エントリー・ポイントとなると premain メソッドです。
	 * <p>
	 * @param option -javaagent 起動引数
	 * @param inst インストゥルメンテーション
	 * @throws Throwable
	 * @throws Throwable 処理できないエラーが発生した場合
	 */
	public static void premain(final String option, final Instrumentation inst) throws Throwable {

		long start = System.nanoTime();

		try {
			instrumentation = inst;

			// Pleiades 起動オプション作成
			pleiadesOption = new PleiadesOption(option);

			// 初期化
			init();

		} catch (Throwable e) {

			e.printStackTrace();
			throw e;
		}

		try {
			// Swing のフォントを初期化。下記の理由により Windows 以外は処理しない。
			//---------------------------------------------------------------------------
			// ・Mac OSX では非同期で行うと強制終了。
			// ・同期処理でもタイミングによりイベントディスパッチスレッドで NPE が発生する。
			// ・Fedora6, 7 や CentOS では NPE が発生する。
			if (os == OS.WINDOWS) {

				// Aptana でファイルタブから「Local Filesystem」を開くと VM クラッシュ。
				// L&F 変更は問題が多いため廃止。
				/*
				Asyncs.execute(new Runnable() {
					public void run() {
						try {
							initSwingFont();
						} catch (Throwable e) {
							log.debug("Swing の LookAndFeel 設定に失敗しました。" + e);
						}
					}
				});
				*/
			} else if (os == OS.MAC) {
				pleiadesOption.setNoMnemonic(true);
			}
			Analyses.end(Pleiades.class, "premain", start);

			// 起動トランスフォーマー開始
			inst.addTransformer(new LauncherTransformer());
			log.info("Pleiades AOP 動的翻訳コンテナーを開始しました。Pleiades 起動オプション:" + option);

		} catch (Throwable e) {

			StringBuilder msg = new StringBuilder();
			msg.append("Pleiades AOP 動的翻訳コンテナーの起動に失敗しました。\n");
			if (pleiadesConfigurationPath != null && pleiadesConfigurationPath.exists()) {
				msg.append(pleiadesConfigurationPath);
				msg.append("\n上記ディレクトリーを削除して、");
			}
			msg.append("起動オプションに -clean を指定して起動してください。");
			log.fatal(e, msg.toString());

			System.exit(-1);
		}
	}

	/**
	 * 起動に必要な情報を初期化します。
	 * @throws UnsupportedEncodingException
	 */
	private static void init() throws UnsupportedEncodingException {

		long start = System.nanoTime();

		// OS 判定
		String osName = System.getProperty("os.name", "").toLowerCase();
		if (osName.contains("mac")) {
			os = OS.MAC;
		} else if (osName.contains("windows")) {
			os = OS.WINDOWS;
		} else {
			os = OS.LINUX;
		}

		// config.ini の位置を元に Eclipse ホームを取得
		File configIniFile = getConfigIniFile();
		File configPath = configIniFile.getParentFile();
		eclipseHome = configPath.getParentFile();

		// config.ini からユーザー・コンフィグレーション・パスの取得
		final String CONFIG_AREA_KEY = "osgi.configuration.area";
		PropertySet configIni = new PropertySet(configIniFile);
		String configArea = configIni.get(CONFIG_AREA_KEY);
		if (configArea == null) {
			configArea = System.getProperty(CONFIG_AREA_KEY);
		}
		File userConfigPath = null;
		if (configArea != null) {
			Matcher mat = Pattern.compile("^@([\\w\\.]+)(.+)$").matcher(configArea);
			if (mat.find()) {
				String base = System.getProperty(mat.group(1));
				if (base != null) {
					File baseDir = new File(base);
					if (baseDir.exists()) {
						String childPath = mat.group(2);
						userConfigPath = new File(baseDir, childPath);
					}
				}
			}
		}

		// p2agent のバンドル・ホームを取得
		String osgiJarPath = configIni.get("osgi.framework");
		if (osgiJarPath != null) {
			File jarFile = new File(osgiJarPath.replace("file:", ""));
			if (jarFile.exists()) {
				bundleHome = jarFile.getParentFile().getParentFile();
			}
		}

		// config.ini に指定がある場合
		if (userConfigPath != null) {
			configPath = userConfigPath;
			pleiadesConfigurationPath = new File(configPath, PACKAGE_NAME);
		}
		// config.ini に指定がない場合
		else {
			if (os == OS.WINDOWS) {
				// Windows の場合は configPath デフォルト
				// <インストールディレクトリ>/configuration/
				pleiadesConfigurationPath = new File(configPath, PACKAGE_NAME);

			} else {
				// Linux、Mac の場合はマルチユーザ環境での書き込み権なしに対応
				// <ユーザホーム>/.eclipse/<パッケージ名>/<インストールパスの - 区切り>/
				File base = new File(System.getProperty("user.home"), ".eclipse/" + PACKAGE_NAME);
				String instPath = eclipseHome.getPath().replace("/", "-");
				pleiadesConfigurationPath = new File(base, instPath);
			}
		}
		pleiadesConfigurationPath.mkdirs();

		// ロガーの初期化
		File logFile = new File(pleiadesConfigurationPath, "pleiades.log");
		Logger.init(PopupFileLogger.class, pleiadesOption.getLogLevel(), logFile);
		log = Logger.getLogger(Pleiades.class);

		log.info("Pleiades 構成保管パス: " + pleiadesConfigurationPath);
		log.info("Eclipse ホーム: " + eclipseHome);

		Analyses.end(Pleiades.class, "init", start);
	}

	/**
	 * config.ini ファイルを取得します。
	 * @return config.ini ファイル
	 */
	private static File getConfigIniFile() {

		final String CONFIG_INI = "config.ini";
		final String CONFIG_DIR_INI = "configuration/" + CONFIG_INI;
		File configIniFile = new File(System.getProperty("user.dir"),
				CONFIG_DIR_INI);
		if (configIniFile.exists()) {
			return configIniFile;
		}
		File pleiadesDir = Files.resourceRoot.getParentFile().getParentFile().getParentFile();

		// config ディレクトリがデフォルトの名前の場合の検索
		for (File dir = pleiadesDir; dir != null; dir = dir.getParentFile()) {
			configIniFile = new File(dir, CONFIG_DIR_INI);
			if (configIniFile.exists()) {
				return configIniFile;
			}
		}

		// config ディレクトリが -configuration で変更されている場合の検索
		for (File dir = pleiadesDir; dir != null; dir = dir.getParentFile()) {
			for (File d : dir.listFiles(Files.createDirectoryFilter())) {
				configIniFile = new File(d, CONFIG_INI);
				if (configIniFile.exists()) {
					return configIniFile;
				}
			}
		}
		throw new IllegalStateException(CONFIG_INI + " が見つかりません。");
	}

	// L&F 変更は問題が多いため廃止。2010.02.07
	/**
	 * Swing で使用するフォントを初期化します。
	 * Eclipse のプラグインは通常 SWT が使用されていますが、
	 * 一部のプラグインでは Swing も利用されています。
	 * <p>
	 * @throws ClassNotFoundException LookAndFeel クラスが見つからなかった場合
	 * @throws InstantiationException クラスの新しいインスタンスを生成できなかった場合
	 * @throws IllegalAccessException クラスまたは初期化子にアクセスできない場合
	 * @throws UnsupportedLookAndFeelException lnf.isSupportedLookAndFeel() が false の場合
	private static void initSwingFont() throws ClassNotFoundException,
			InstantiationException, IllegalAccessException,
			UnsupportedLookAndFeelException {

		long start = System.nanoTime();

		// LookAndFeel の設定
		UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
		Toolkit.getDefaultToolkit().setDynamicLayout(true);

		// フォント設定
		Object propoFont = new FontUIResource("MS UI Gothic",	Font.TRUETYPE_FONT, 12);
		Object fixedFont = new FontUIResource("MS Gothic",		Font.TRUETYPE_FONT, 12);

		// 一旦すべてのコンポーネントのフォントをプロポーショナルにする。
		// フォントリソースかの判定は値を取得し instanceof FontUIResource が
		// 安全だが、UIDefaults の Lazy Value を生かすため末尾の文字列で判定。
		for (Object keyObj : UIManager.getLookAndFeelDefaults().keySet()) {
			String key = keyObj.toString();
			if (key.endsWith("font") || key.endsWith("Font")) {
				UIManager.put(key, propoFont);
			}
		}

		// 特定コンポーネントのフォントを等幅にする
		UIManager.put("TextPane.font", fixedFont);
		UIManager.put("TextArea.font", fixedFont);

		Analyses.end(Pleiades.class, "initSwingFont<THREAD>", start);
	}
	 */

	/**
	 * インストゥルメンテーションを取得します。
	 * <p>
	 * @return インストゥルメンテーション
	 */
	public static Instrumentation getInstrumentation() {
		return instrumentation;
	}

	/**
	 * Pleiades 起動オプションを取得します。
	 * <p>
	 * @return Pleiades 起動オプション
	 */
	public static PleiadesOption getPleiadesOption() {
		return pleiadesOption;
	}

	/**
	 * Pleiades コンフィグレーション・パスからの相対パスを指定してファイルを取得します。
	 * <p>
	 * 通常、Pleiades コンフィグレーション・パスは
	 * configuration/jp.sourceforge.mergedoc.pleiades
	 * で、Eclipse の config.ini の指定がある場合はそれが優先されます。
	 * <p>
	 * @param path Pleiades コンフィグレーション・パスからの相対パス
	 * @return ファイル
	 */
	public static File getResourceFile(String path) {
		return new File(pleiadesConfigurationPath, path);
	}

	/**
	 * Eclipse ホームを取得します。
	 * <p>
	 * @return Eclipse ホーム
	 */
	public static File getEclipseHome() {
		return eclipseHome;
	}

	/**
	 * p2agent バンドル・ホームを取得します。
	 * 通常は Eclipse ホームと同じですが、p2agent を使用した場合、
	 * インストール時に指定したバンドル・プール・ロケーションとなり、
	 * このディレクトリー配下の plugins に基本的なプラグイン群が格納されます。
	 * <p>
	 * @return Eclipse バンドル・ホーム
	 */
	public static File getBundleHome() {
		return bundleHome;
	}

	/**
	 * OS が Windows か判定します。
	 * @return Windows の場合は true
	 */
	public static boolean isWindows() {
		return os == OS.WINDOWS;
	}
}
