/*
 * Copyright (c) 2005- Shinji Kashihara.
 * All rights reserved. This program are made available under
 * the terms of the Eclipse Public License v1.0 which accompanies
 * this distribution, and is available at epl-v10.html.
 */
package jp.sourceforge.mergedoc.pleiades.generator.nls;

import static jp.sourceforge.mergedoc.pleiades.util.UnMnemonicProperties.toPropertyKey;
import static jp.sourceforge.mergedoc.pleiades.util.UnMnemonicProperties.toPropertyValue;

import java.util.Map.Entry;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import jp.sourceforge.mergedoc.pleiades.log.Logger;
import jp.sourceforge.mergedoc.pleiades.util.FastProperties;
import jp.sourceforge.mergedoc.pleiades.util.UnMnemonicProperties;

/**
 * 翻訳ルールに従って訳を最適化します。
 * <p>
 * @author cypher256
 */
public class Optimizer {

	/** ロガー */
	private static final Logger log = Logger.getLogger(Optimizer.class);

	/** 用語検証プロパティー */
	private static final FastProperties vTermProp =
		new FastProperties("props/validation-term.properties");

	/** 翻訳プロパティー (すべて) */
	private static final FastProperties transProp =
		new FastProperties("translation.properties");

	/**
	 * メイン。
	 * @param args
	 */
	public static void main(String... args) {
		
		FastProperties inProp = new UnMnemonicProperties("props/_a.properties");
		FastProperties outProp = new Optimizer().apply(inProp);
		outProp.store("props/_a_out.properties", "");
	}
	
	/**
	 * コンストラクタ。
	 */
	public Optimizer() {
	}

	/**
	 * 翻訳ルールを適用します。
	 * <p>
	 * @param inProp ルールを適用するプロパティー
	 * @return ルール適用後のプロパティー
	 */
	public FastProperties apply(FastProperties inProp) {

		FastProperties outProp = new FastProperties();
		
		for (Entry<String, String> entry : inProp.entrySet()) {

			String en = entry.getKey();
			String ja = entry.getValue();

			//-----------------------------------------------------------------
			// 原文と訳文が同じものは読み飛ばし
			//-----------------------------------------------------------------
			
			if (en.equals(ja)) {
				log.info("原文と訳文が同じであるためエントリー読み飛ばし\n" +
						"英: " + toPropertyKey(en) + "\n");
				continue;
			}

			//-----------------------------------------------------------------
			// 訳語の統一
			//-----------------------------------------------------------------

			// 翻訳ルールによる置換（例：下さい→ください）
			String enBefore = en;
			String jaBefore = ja;
			
			for (Entry<String, String> rep : vTermProp.entrySet()) {

				String ng = rep.getKey();
				String ok = rep.getValue();
				
				if (!ok.equals("") && ja.matches(".*?" + ng + ".*")) {
					Pattern pat = Pattern.compile(ng);
					Matcher mat = pat.matcher(ja);
					StringBuffer sb = new StringBuffer();
					while (mat.find()) {mat.appendReplacement(sb, ok);}
					mat.appendTail(sb);
					ja = sb.toString();
				}
			}
			if (!ja.equals(jaBefore)) {
				log.info("翻訳ルールによる置換\n" +
						"置換前: " + toPropertyValue(jaBefore) + "\n" +
						"置換後: " + toPropertyValue(ja) + "\n");
			}

			//-----------------------------------------------------------------
			// 前後の空白合わせ
			//-----------------------------------------------------------------
			
			String leading =
				en.matches("\\s+.*") ?
				en.replaceFirst("^(\\s+).+$", "$1") :
				"";
			
			String trailing =
				en.matches(".+?\\s+") ?
				en.replaceFirst("^.+?(\\s+)$", "$1") :
				"";
			
			String e = en.trim();
			String j = ja.trim();
			jaBefore = ja;
			ja = leading + j + trailing;
			
			if (!ja.equals(jaBefore)) {
				log.info("前後空白合わせによる置換\n" +
						"英: " + toPropertyKey(en) + "\n" +
						"置換前: " + toPropertyValue(jaBefore) + "\n" +
						"置換後: " + toPropertyValue(ja) + "\n");
			}

			//-----------------------------------------------------------------
			// 原文末尾に「.」が無い場合は訳文末尾の「。」を除去
			//-----------------------------------------------------------------
			
			if (!e.endsWith(".") && j.endsWith("。")) {
				
				jaBefore = ja;
				ja = ja.replaceFirst("。(\\s*)$", "$1");
				
				log.info("訳文末尾の「。」除去\n" +
						"英: " + toPropertyKey(en) + "\n" +
						"除去前: " + toPropertyValue(jaBefore) + "\n" +
						"除去後: " + toPropertyValue(ja) + "\n");
			}
			
			//-----------------------------------------------------------------
			// 末尾 (Incubation)
			//-----------------------------------------------------------------
			if (e.endsWith("(Incubation)")) {
				
				enBefore = en;
				jaBefore = ja;
				
				en = en.replaceFirst("\\s*\\(Incubation\\)\\s*$", "");
				ja = ja.replaceFirst("\\s*\\(インキュベーション.*?\\)\\s*$", "");
				
				log.info("末尾 (Incubation) の除去\n" +
						"置換前: " + toPropertyKey(enBefore) + "\n" +
						"　　　  " + toPropertyValue(jaBefore) + "\n" +
						"置換後: " + toPropertyKey(en) + "\n" +
						"　　　  " + toPropertyValue(ja) + "\n");
			}

			//-----------------------------------------------------------------
			// ルール適用後、翻訳辞書に存在するものは読み飛ばし
			//-----------------------------------------------------------------
			
			String transJa = transProp.get(en);
			if (transJa != null) {

				if (transJa.equals(ja)) {
					// 原文も訳文も同一の訳の場合は読み飛ばし
					continue;
				} else {
					// 原文のみが同一の場合はログ出力し、読み飛ばし
					/*
					log.info("翻訳辞書に原文が同じ、訳文が異なるエントリーが存在するため読み飛ばし\n" +
							"英: " + toPropertyKey(en) + "\n" +
							"日: " + toPropertyValue(ja) + "\n" +
							"既: " + toPropertyValue(transJa) + "\n");
					*/
					continue;
				}
			}
			
			outProp.put(en, ja);
		}
		
		log.info("入力プロパティー数:" + inProp.size());
		log.info("出力プロパティー数:" + outProp.size());
		
		return outProp;
	}
}
