/*
 * Copyright (c) 2005- Shinji Kashihara.
 * All rights reserved. This program are made available under
 * the terms of the Eclipse Public License v1.0 which accompanies
 * this distribution, and is available at epl-v10.html.
 */
package jp.sourceforge.mergedoc.pleiades.aspect.resource;

import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;

import jp.sourceforge.mergedoc.pleiades.aspect.advice.JointPoint;
import jp.sourceforge.mergedoc.pleiades.log.Logger;
import jp.sourceforge.mergedoc.pleiades.util.UnMnemonicProperties;

/**
 * 翻訳ユーティリティーです。
 * <p>
 * このクラスのメソッドは、バイトコード変換されたクラスから
 * 呼び出しやすいように、static メソッドで構成されています。
 * <p>
 * @author cypher256
 */
public class Translations {

	/** ロガー */
	private static final Logger log = Logger.getLogger(Translations.class);

	/** 翻訳辞書 */
	private static final DynamicTranslationDictionary dictionary =
		DynamicTranslationDictionary.getInstance();

	/** コンボの値を保持するマップ */
	private static final Map<String, String> backupMap =
		new HashMap<String, String>();

	/**
	 * インスタンス化できません。
	 */
	private Translations() {
	}

	/**
	 * 指定された英語リソース文字列を翻訳します。
	 * <p>
	 * @param enValue 英語リソース文字列
	 * @param jointPoint ジョイント・ポイント
	 * @return 翻訳後文字列。翻訳できない場合は引数と同じ。
	 */
	public static String translate(String enValue, JointPoint jointPoint) {

		if (enValue == null) return null;
		return dictionary.lookup(enValue, jointPoint);
	}

	/**
	 * 指定された英語リソース文字列の配列を翻訳します。
	 * <p>
	 * @param enValues 英語リソース文字列の配列
	 * @param jointPoint ジョイント・ポイント
	 * @return 翻訳後文字列。翻訳できない場合は引数と同じ。
	 */
	public static String[] translate(String[] enValues, JointPoint jointPoint) {

		if (enValues == null) return null;
		for (int i = 0; i < enValues.length; i++) {
			enValues[i] = translate(enValues[i], jointPoint);
		}
		return enValues;
	}

	/**
	 * 指定された英語リソース・オブジェクトを翻訳します。
	 * <p>
	 * @param enValue 英語リソース・オブジェクト
	 * @param jointPoint ジョイント・ポイント
	 * @return 翻訳後文字列。翻訳できない場合は引数と同じ。
	 */
	public static Object translate(Object enValue, JointPoint jointPoint) {

		if (enValue == null) {
			return null;
		} else if (enValue instanceof String) {
			return translate((String) enValue, jointPoint);
		} else if (enValue instanceof String[]) {
			return translate((String[]) enValue, jointPoint);
		} else if (enValue instanceof Object[]) {
			return translate((Object[]) enValue, jointPoint);
		}
		return enValue;
	}

	/**
	 * 指定された英語リソース・オブジェクトを翻訳します。
	 * <p>
	 * @param enValue 英語リソース・オブジェクト
	 * @param jointPoint ジョイント・ポイント
	 * @return 翻訳後文字列。翻訳できない場合は引数と同じ。
	 */
	public static Object[] translate(Object enValue[], JointPoint jointPoint) {

		if (enValue instanceof String[]) {
			return translate((String[]) enValue, jointPoint);
		}
		return enValue;
	}

	/**
	 * 指定された英語リソース文字列をメニュー用に翻訳します。
	 * <p>
	 * @param enValue 英語リソース文字列
	 * @param jointPoint ジョイント・ポイント
	 * @return 翻訳後文字列。翻訳できない場合は引数と同じ。
	 */
	public static String translateMenu(String enValue, JointPoint jointPoint) {

		if (enValue == null) return null;
		String[] splitsAccelerator = enValue.split("\t");
		String enMenuText = splitsAccelerator[0];

		// 現在このメソッド呼び出しは限定されているため、先頭 & 除去は不要
		/*
		if (!enMenuText.matches("\\p{ASCII}+")) {
			// 日本語メニューの新規項目先頭に & が付加されてしまうのを回避
			enMenuText = enMenuText.replaceFirst("^&", "");
		}
		*/
		if (splitsAccelerator.length == 1) {
			return dictionary.lookup(enMenuText, jointPoint);
		} else {
			return dictionary.lookup(enMenuText, jointPoint) + "\t" + splitsAccelerator[1];
		}
	}

	/**
	 * 指定された英語リソース文字列を翻訳します。
	 * ニーモニックは処理されません。
	 * <p>
	 * @param enValue 英語リソース文字列
	 * @param jointPoint ジョイント・ポイント
	 * @return 翻訳後文字列。翻訳できない場合は引数と同じ。
	 */
	public static String translateIgnoreMnemonic(String enValue, JointPoint jointPoint) {

		if (enValue == null) return null;
		return dictionary.lookupIgnoreMnemonic(enValue, jointPoint);
	}

	/**
	 * 指定された英語リソース文字列をバックアップし、翻訳します。
	 * <p>
	 * @param enValue 英語リソース文字列
	 * @param jointPoint ジョイント・ポイント
	 * @return 翻訳後文字列。翻訳できない場合は引数と同じ。
	 */
	public static String translateBackup(String enValue, JointPoint jointPoint) {

		if (enValue == null) return null;
		String jaValue = dictionary.lookup(enValue, jointPoint);
		backupMap.put(jaValue, enValue);
		return jaValue;
	}

	/**
	 * 指定された英語リソース文字列をバックアップし、翻訳します。
	 * ニーモニックは処理されません。
	 * <p>
	 * @param enValue 英語リソース文字列
	 * @param jointPoint ジョイント・ポイント
	 * @return 翻訳後文字列。翻訳できない場合は引数と同じ。
	 */
	public static String translateBackupIgnoreMnemonic(String enValue, JointPoint jointPoint) {

		if (enValue == null) return null;
		String jaValue = dictionary.lookupIgnoreMnemonic(enValue, jointPoint);
		backupMap.put(jaValue, enValue);
		return jaValue;
	}

	/**
	 * 指定された英語リソース文字列配列をバックアップし、翻訳します。
	 * ニーモニックは処理されません。
	 * <p>
	 * @param enValues 英語リソース文字列
	 * @param jointPoint ジョイント・ポイント
	 * @return 翻訳後文字列。翻訳できない場合は引数と同じ。
	 */
	public static String[] translateBackupIgnoreMnemonic(String[] enValues, JointPoint jointPoint) {

		if (enValues == null) return null;
		for (int i = 0; i < enValues.length; i++) {
			enValues[i] = translateBackupIgnoreMnemonic(enValues[i], jointPoint);
		}
		return enValues;
	}

	/**
	 * translateBackup で翻訳された日本語リソース文字列を英語に戻します。
	 * <p>
	 * @param jaValue 日本語リソース文字列
	 * @param jointPoint ジョイント・ポイント
	 * @return 英語文字列。戻せない場合は引数と同じ。
	 */
	public static String restore(String jaValue, JointPoint jointPoint) {

		if (jaValue == null) return null;
		String enValue = backupMap.get(jaValue);
		if (enValue == null) {
			return jaValue;
		}
		return enValue;
	}

	/**
	 * 指定された英語リソース文字列を翻訳します。
	 * 空白が 1 つ以上含まれる場合のみ翻訳されます。
	 * <p>
	 * @param enValue 英語リソース文字列
	 * @param jointPoint ジョイント・ポイント
	 * @return 翻訳後文字列。翻訳できない場合は引数と同じ。
	 */
	public static String translateSentence(String enValue, JointPoint jointPoint) {

		if (enValue == null) return null;
		if (enValue.contains(" ")) {
			return dictionary.lookup(enValue, jointPoint);
		}
		return enValue;
	}

	/**
	 * 指定された日本用ニーモニックを含むリソース文字列を翻訳します。
	 * 引数に含まれる日本用ニーモニックは翻訳前に除去されます。
	 * <p>
	 * @param enValue 日本用ニーモニックが含まれたリソース文字列
	 * @param jointPoint ジョイント・ポイント
	 * @return 翻訳後文字列。翻訳できない場合は引数と同じ。
	 */
	public static String translateWithoutJaMnemonic(String enValue, JointPoint jointPoint) {

		if (enValue == null) return null;
		enValue = enValue.replaceFirst(
				"\\(\\&" + UnMnemonicProperties.MNEMONIC_CHARS + "\\)", "");
		return dictionary.lookupIgnoreMnemonic(enValue, jointPoint);
	}

	/**
	 * 指定した値をデバッグ・ログとして出力します。
	 * <p>
	 * @param obj オブジェクト
	 */
	public static void debug(Object obj) {

		StringBuilder sb = new StringBuilder();
		sb.append("Translations#debug ");
		if (obj != null) {
			sb.append(obj.getClass().getName());
		}
		sb.append("[");
		sb.append(obj);
		sb.append("]");
		log.info(sb.toString());
	}

	/**
	 * 指定した値をデバッグ・ログとして出力します。
	 * <p>
	 * @param enValue 英語リソース・オブジェクト
	 * @param jointPoint ジョイント・ポイント
	 */
	public static void debug(Object enValue, JointPoint jointPoint) {

		StringBuilder sb = new StringBuilder();
		sb.append("Translations#debug ");
		if (jointPoint != null) {
			sb.append("(");
			sb.append(jointPoint.getClassName());
			sb.append("#");
			sb.append(jointPoint.getMethodName());
			sb.append(") ");
		}
		if (enValue != null) {
			sb.append(enValue.getClass().getName());
		}
		sb.append("[");
		sb.append(enValue);
		sb.append("] -> [");
		sb.append(translate(enValue, jointPoint));
		sb.append("]");
		log.info(sb.toString());
	}

	/**
	 * 指定した値をデバッグ・ログとして出力します。
	 * <p>
	 * @param enValue 英語リソース・オブジェクト
	 * @param jointPoint ジョイント・ポイント
	 * @param key キー
	 */
	public static void debug(Object enValue, JointPoint jointPoint, String key) {

		StringBuilder sb = new StringBuilder();
		sb.append("Translations#debug ");
		if (jointPoint != null) {
			sb.append("(");
			sb.append(jointPoint.getClassName());
			sb.append("#");
			sb.append(jointPoint.getMethodName());
			sb.append(") ");
		}
		if (enValue != null) {
			sb.append(enValue.getClass().getName());
		}
		sb.append("[");
		sb.append(enValue);
		sb.append("] -> [");
		sb.append(translate(enValue, jointPoint));
		sb.append("]");
		
		sb.append(" key=");
		sb.append(key);
		log.info(sb.toString());
	}

	/**
	 * 指定した値をデバッグ・ログとして出力します。
	 * <p>
	 * @param enValues 英語リソース・オブジェクトの配列
	 * @param jointPoint ジョイント・ポイント
	 */
	public static void debug(String[] enValues, JointPoint jointPoint) {

		StringBuilder sb = new StringBuilder();
		sb.append("Translations#debug ");
		if (jointPoint != null) {
			sb.append("(");
			sb.append(jointPoint.getClassName());
			sb.append("#");
			sb.append(jointPoint.getMethodName());
			sb.append(") ");
		}
		if (enValues != null) {
			sb.append(enValues.getClass().getName());
		}
		sb.append(Arrays.asList(enValues));
		sb.append(" -> ");
		sb.append(Arrays.asList(translate(enValues, jointPoint)));
		log.info(sb.toString());
	}

	/**
	 * 現在のスレッドのスタックトレースをログに出力します。
	 * <p>
	 * @param value 英語リソース・オブジェクト
	 * @param jointPoint ジョイント・ポイント
	 */
	public static void printStackTrace(Object value, JointPoint jointPoint) {
		log.info("Translator#debug " +
				(value == null ? "" : value.getClass().getName()) +
				"[" + value + "] -> [" + translate(value, jointPoint) + "]",
				new Exception());
	}
}
