/*
 * Copyright (c) 2005- Shinji Kashihara.
 * All rights reserved. This program are made available under
 * the terms of the Eclipse Public License v1.0 which accompanies
 * this distribution, and is available at epl-v10.html.
 */
package jp.sourceforge.mergedoc.pleiades.aspect.resource;

import java.lang.reflect.Field;

import jp.sourceforge.mergedoc.pleiades.aspect.ITimingLogger;
import jp.sourceforge.mergedoc.pleiades.aspect.Pleiades;
import jp.sourceforge.mergedoc.pleiades.aspect.advice.JointPoint;
import jp.sourceforge.mergedoc.pleiades.log.Logger;

/**
 * 時間計測ログ機能付きの辞書です。
 * <p>
 * @author cypher256
 */
public class DynamicTranslationLoggingDictionary extends DynamicTranslationDictionary implements ITimingLogger {

	/** ロガー */
	private static final Logger log = Logger.getLogger(DynamicTranslationLoggingDictionary.class);

	/** convert メソッドの所要時間 */
	private volatile long applyConverterTime;

	/** convert メソッドの呼び出し回数 */
	private volatile long applyConverterCall;

	/** lookup メソッドの所要時間 */
	private volatile long lookupTime;

	/** lookup メソッドの呼び出し回数 */
	private volatile long lookupCall;

	/** lookupNoMnemonic メソッドの所要時間 */
	private volatile long lookupIgnoreMnemonicTime;

	/** lookupNoMnemonic メソッドの呼び出し回数 */
	private volatile long lookupIgnoreMnemonicCall;

	/** getJaValue メソッドの所要時間 */
	private volatile long getValueTime;

	/** getJaValue メソッドの呼び出し回数 */
	private volatile long getValueCall;

	/** getJaValueByRegex メソッドの所要時間 */
	private volatile long getValueToTrimTime;

	/** getJaValueByRegex メソッドの呼び出し回数 */
	private volatile long getValueToTrimCall;

	/** getJaValueByRegex メソッドの所要時間 */
	private volatile long getValueByRegexTime;

	/** getJaValueByRegex メソッドの呼び出し回数 */
	private volatile long getValueByRegexCall;

	/** isNoTranslation メソッドの所要時間 */
	private volatile long isNoTranslationTime;

	/** isNoTranslation メソッドの呼び出し回数 */
	private volatile long isNoTranslationCall;

	/** editMnemonicEnToJa メソッドの所要時間 */
	private volatile long editMnemonicEnToJaTime;

	/** editMnemonicEnToJa メソッドの呼び出し回数 */
	private volatile long editMnemonicEnToJaCall;

	/**
	 * この辞書を構築します。
	 */
	DynamicTranslationLoggingDictionary() {
		Pleiades.registTimingLogger(this);
	}

	/**
	 * 親クラスの同メソッドに時間計測ログ出力を追加したメソッドです。
	 */
	@Override
	protected void applyConverter() {

		long startTime = System.currentTimeMillis();
		super.applyConverter();
		synchronized (this) {
			applyConverterTime += System.currentTimeMillis() - startTime;
			applyConverterCall++;
		}
	}

	/**
	 * 親クラスの同メソッドに時間計測ログ出力を追加したメソッドです。
	 */
	@Override
	public String lookup(String enValue, JointPoint jointPoint) {

		long startTime = System.currentTimeMillis();
		String result = super.lookup(enValue, jointPoint);
		synchronized (this) {
			lookupTime += System.currentTimeMillis() - startTime;
			lookupCall++;
		}
		return result;
	}

	/**
	 * 親クラスの同メソッドに時間計測ログ出力を追加したメソッドです。
	 */
	@Override
	public String lookupIgnoreMnemonic(String enValue, JointPoint jointPoint) {

		long startTime = System.currentTimeMillis();
		String result = super.lookupIgnoreMnemonic(enValue, jointPoint);
		synchronized (this) {
			lookupIgnoreMnemonicTime += System.currentTimeMillis() - startTime;
			lookupIgnoreMnemonicCall++;
		}
		return result;
	}

	/**
	 * 親クラスの同メソッドに時間計測ログ出力を追加したメソッドです。
	 */
	@Override
	protected String getValue(String enNoMnemonic) {

		long startTime = System.currentTimeMillis();
		String result = super.getValue(enNoMnemonic);
		synchronized (this) {
			getValueTime += System.currentTimeMillis() - startTime;
			getValueCall++;
		}
		return result;
	}

	/**
	 * 親クラスの同メソッドに時間計測ログ出力を追加したメソッドです。
	 */
	@Override
	protected String getValueToTrim(String enNoMnemonic) {

		long startTime = System.currentTimeMillis();
		String result = super.getValueToTrim(enNoMnemonic);
		synchronized (this) {
			getValueToTrimTime += System.currentTimeMillis() - startTime;
			getValueToTrimCall++;
		}
		return result;
	}

	/**
	 * 親クラスの同メソッドに時間計測ログ出力を追加したメソッドです。
	 */
	@Override
	protected String getValueByRegex(String enNoMnemonic) {

		long startTime = System.currentTimeMillis();
		String result = super.getValueByRegex(enNoMnemonic);
		synchronized (this) {
			getValueByRegexTime += System.currentTimeMillis() - startTime;
			getValueByRegexCall++;
		}
		return result;
	}

	/**
	 * 親クラスの同メソッドに時間計測ログ出力を追加したメソッドです。
	 */
	@Override
	protected boolean isNoTranslation(String enValueNonMnemonic, JointPoint jointPoint) {

		long startTime = System.currentTimeMillis();
		boolean result = super.isNoTranslation(enValueNonMnemonic, jointPoint);
		synchronized (this) {
			isNoTranslationTime += System.currentTimeMillis() - startTime;
			isNoTranslationCall++;
		}
		return result;
	}

	/**
	 * 親クラスの同メソッドに時間計測ログ出力を追加したメソッドです。
	 */
	@Override
	protected String editMnemonicEnToJa(String enValue,
			String enValueNonMnemonic, String result) {

		long startTime = System.currentTimeMillis();
		String jaResult = super.editMnemonicEnToJa(enValue, enValueNonMnemonic, result);
		synchronized (this) {
			editMnemonicEnToJaTime += System.currentTimeMillis() - startTime;
			editMnemonicEnToJaCall++;
		}
		return jaResult;
	}

	/**
	 * 時間計測ログを出力します。
	 * @return 計測時間合計 (秒)
	 */
	public long logTime() {

		String msg = "%-30s %8d 回呼出計 %6.3f 秒";

		log.debug(String.format(msg, "applyConverter", applyConverterCall, applyConverterTime / 1000d));
		log.debug(String.format(msg, "lookupIgnoreMnemonic", lookupIgnoreMnemonicCall, lookupIgnoreMnemonicTime / 1000d));
		log.debug(String.format(msg, "lookup", lookupCall, lookupTime / 1000d));
		log.debug(String.format(msg, "  editMnemonicEnToJa", editMnemonicEnToJaCall, editMnemonicEnToJaTime / 1000d));
		log.debug(String.format(msg, "  getValue", getValueCall, getValueTime / 1000d));
		log.debug(String.format(msg, "    isNoTranslation", isNoTranslationCall, isNoTranslationTime / 1000d));
		log.debug(String.format(msg, "    getValueToTrim", getValueToTrimCall, getValueToTrimTime / 1000d));
		log.debug(String.format(msg, "      getValueByRegex", getValueByRegexCall, getValueByRegexTime / 1000d));

		long summuryTime =
			applyConverterTime + lookupIgnoreMnemonicTime + lookupTime +
			editMnemonicEnToJaTime + getValueTime + isNoTranslationTime +
			getValueToTrimTime + getValueByRegexTime;

		log.debug(String.format("クラス計 %6.3f 秒", summuryTime / 1000d));

		return summuryTime;
	}

	/**
	 * 計測値をリセットします。
	 */
	public void reset() {

		synchronized (this) {
			Field[] fields = getClass().getDeclaredFields();
			for (int i = 0; i < fields.length; i++) {
				Field field = fields[i];
				if (field.getType() == Long.TYPE) {
					try {
						field.setAccessible(true);
						field.set(this, 0L);
					} catch (Exception e) {
						log.error("フィールドの初期化に失敗しました。", e);
					}
				}
			}
		}
	}
}
