/*
 * Copyright (c) 2005- Shinji Kashihara.
 * All rights reserved. This program are made available under
 * the terms of the Eclipse Public License v1.0 which accompanies
 * this distribution, and is available at epl-v10.html.
 */
package jp.sourceforge.mergedoc.pleiades.aspect.advice;

import java.io.File;
import java.io.FileNotFoundException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;

import jp.sourceforge.mergedoc.pleiades.aspect.Pleiades;
import jp.sourceforge.mergedoc.pleiades.aspect.advice.JointPoint.EditPoint;
import jp.sourceforge.mergedoc.pleiades.log.Logger;
import jp.sourceforge.mergedoc.pleiades.util.Files;

import org.apache.commons.lang.builder.ToStringBuilder;
import org.apache.commons.lang.builder.ToStringStyle;

/**
 * アスペクト・マッピングです。<br>
 * ジョイント・ポイントとポイント・カットの関連付けを保持します。
 * <p>
 * @author cypher256
 */
public class AspectMapping {

	/** ロガー */
	private static final Logger log = Logger.getLogger(AspectMapping.class);

	/** 設定ファイル名 */
	private static final String CONFIG_FILE_NAME = "pleiades-config.xml";

	/** このクラスのインスタンス */
	private static final AspectMapping singleton = new AspectMapping();

	/**
	 * アスペクト・マッピングを取得します。
	 * <p>
	 * @return アスペクト・マッピング
	 */
	public static AspectMapping getInstance() {
		return singleton;
	}

	/** ジョイント・ポイントとポイント・カットを保持するマップ */
	private Map<JointPoint, PointCut> jointMap = new HashMap<JointPoint, PointCut>();

	/** ウィービング対象のメソッド呼び出し名前セット */
	private Set<String> methodCalls = new HashSet<String>();

	/**
	 * アスペクト・マッピングを構築します。
	 */
	private AspectMapping() {

	    File configFile = Files.getResourceFile(CONFIG_FILE_NAME);
		try {
		    if (!configFile.exists()) {
		    	throw new FileNotFoundException(configFile.getPath());
		    }
		    SAXParserFactory factory = SAXParserFactory.newInstance();
		    SAXParser parser = factory.newSAXParser();
		    parser.parse(configFile, new AspectMappingAssembler(jointMap));

		    for (JointPoint jointPoint : jointMap.keySet()) {
				if (jointPoint.getEditPoint() == EditPoint.CALL) {
					methodCalls.add(
							jointPoint.getClassName() + "#" +
							jointPoint.getMethodName());
				}
			}
			log.info("Pleiades 構成ファイルをロードしました。");

		} catch (Exception e) {

			String msg = "Pleiades 構成ファイルのロードに失敗しました。";
			Exception ies = new IllegalStateException(msg, e);
			Pleiades.abort(ies);
		}
	}

	/**
	 * ポイント・カットを取得します。
	 * <p>
	 * @param jointPoint ジョイント・ポイント
	 * @return ポイント・カット
	 */
	public PointCut getPointCut(JointPoint jointPoint) {

		String descBackup = jointPoint.getDescriptor();
		jointPoint.setDescriptor(null);
		PointCut pointCut = jointMap.get(jointPoint);
		jointPoint.setDescriptor(descBackup);
		if (pointCut == null) {
			pointCut = jointMap.get(jointPoint);
		}
		return pointCut;
	}

	/**
	 * メソッド呼び出しがウィービング対象か判定します。
	 * <p>
	 * @param className クラス名
	 * @param methodName メソッド名
	 * @return ウィービング対象の場合は true
	 */
	public boolean containesMethodCall(String className, String methodName) {
		return methodCalls.contains(className + "#" + methodName);
	}

	/**
	 * このオブジェクトの文字列表現を取得します。
	 */
	public String toString() {
		return new ToStringBuilder(this, ToStringStyle.MULTI_LINE_STYLE)
			.append(jointMap)
			.toString();
	}
}
