/*
 * Copyright (c) 2005- Shinji Kashihara.
 * All rights reserved. This program are made available under
 * the terms of the Eclipse Public License v1.0 which accompanies
 * this distribution, and is available at epl-v10.html.
 */
package jp.sourceforge.mergedoc.pleiades.aspect;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.zip.Deflater;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import java.util.zip.ZipOutputStream;

import jp.sourceforge.mergedoc.pleiades.log.Logger;

import org.apache.commons.io.IOUtils;

/**
 * クラス・キャッシュです。
 * <p>
 * @author cypher256
 */
abstract public class AbstractClassCache {

	/** ロガー */
	private final Logger log = Logger.getLogger(getClass());

	/** クラス・キャッシュ・ファイル */
	private final File file = getFile();

	/** 変換済みクラス・マップ */
	private final Map<String, byte[]> map = new HashMap<String, byte[]>();

	/** ロード時の辞書サイズ */
	private int loadedSize;

	/**
	 * キャッシュを保存するファイルを取得します。
	 * @return キャッシュ・ファイル
	 */
	abstract protected File getFile();

	/**
	 * ログなどのメッセージに使用するキャッシュ名を取得します。
	 * @return キャッシュ名
	 */
	abstract protected String getName();

	/**
	 * クラス・キャッシュを構築します。
	 */
	protected AbstractClassCache() {

		if (Pleiades.getPleiadesOption().isClean()) {
			log.info("-clean により" + getName() + "はロードされません。");

		} else if (!file.exists()) {
			log.info(getName() + "が存在しません。");

		} else {

			ZipInputStream in = null;
			try {
				in = new ZipInputStream(new BufferedInputStream(new FileInputStream(file)));
				ZipEntry inEntry = null;
				while ((inEntry = in.getNextEntry()) != null) {
					byte[] bytecode = IOUtils.toByteArray(in);
					map.put(inEntry.getName(), bytecode);
				}
				log.info(getName() + "をロードしました。" + map.size());

			} catch (IOException e) {
				String msg = getName() + " のロードに失敗しました。";
				Exception ise = new IllegalStateException(msg, e);
				Pleiades.abort(ise);

			} finally {
				IOUtils.closeQuietly(in);
			}
		}

		// ロード時の辞書サイズを保存
		loadedSize = map.size();
	}

	/**
	 * 追加されたクラスをキャッシュとして永続化します。
	 */
	public void store() {

		if (map.size() > loadedSize) {

			ZipOutputStream out = null;
			try {
				out = new ZipOutputStream(new BufferedOutputStream(new FileOutputStream(file)));
				out.setLevel(Deflater.BEST_SPEED);

				for (Entry<String, byte[]> entry : map.entrySet()) {
					out.putNextEntry(new ZipEntry(entry.getKey()));
					out.write(entry.getValue());
				}
				log.info(getName() + "を保管しました。" +
						loadedSize + " -> " + map.size());

			} catch (IOException e) {
				log.error(getName() + "の保管に失敗しました。", e);

			} finally {
				IOUtils.closeQuietly(out);
			}
		}
	}

	/**
	 * キャッシュへバイトコードを追加します。
	 * <p>
	 * @param className クラス名
	 * @param bytecode バイトコード
	 */
	public void put(String className, byte[] bytecode) {
		map.put(className, bytecode);
	}

	/**
	 * キャッシュからバイトコードを取得します。
	 * <p>
	 * @param className クラス名
	 * @return バイトコード
	 */
	public byte[] get(String className) {
		return map.get(className);
	}
}
