/*
 * Copyright (c) 2005- Shinji Kashihara.
 * All rights reserved. This program are made available under
 * the terms of the Eclipse Public License v1.0 which accompanies
 * this distribution, and is available at epl-v10.html.
 */
package jp.sourceforge.mergedoc.pleiades.util;

import java.io.File;

/**
 * キー、値に含まれるニーモニックを格納時に除去する Properties です。
 * <p>
 * @author cypher256
 */
public class UnMnemonicProperties extends FastProperties {

	/** シリアル・バージョン ID */
	private static final long serialVersionUID = -4066795589235888438L;

	/** ニーモニック文字の正規表現ブロック */
	public static final String MNEMONIC_CHARS = "[\\w\\.@]";

	/**
	 * ニーモニック除去プロパティーを構築します。
	 */
	public UnMnemonicProperties() {
	}

	/**
	 * パスを指定してニーモニック除去プロパティーを構築します。<br>
	 * パスがディレクトリーの場合は、その直下にあるプロパティー・ファイルが
	 * すべてロードされます。
	 * <p>
	 * @param path パス
	 */
	public UnMnemonicProperties(String path) {
		super(path);
	}

	/**
	 * ルートパスを指定してニーモニック除去プロパティーを構築します。<br>
	 * パスがディレクトリーの場合は、その直下にあるプロパティー・ファイルが
	 * すべてロードされます。
	 * <p>
	 * @param file プロパティー・ファイルまたはディレクトリー
	 */
	public UnMnemonicProperties(File file) {
		super(file);
	}
	
	/**
	 * 指定したキーと値からニーモニックを除去し、この
	 * プロパティーにセットします。
	 */
	@Override
	public String put(String key, String value) {
		return super.put(
				removeEnMnemonic(key),
				removeJaMnemonic(value));
	}

	/**
	 * 指定した値に含まれる (&A) のような日本語ニーモニックを除去します。
	 * ニーモニック対象文字と中括弧も含めて除去されます。
	 * <p>
	 * @param value 値
	 * @return ニーモニック除去後の値
	 */
	private String removeJaMnemonic(String value) {

		// 日本語向けニーモニックの除去
		value = value.replaceFirst("\\(\\&" + MNEMONIC_CHARS + "\\)", "");

		// 日本語向けニーモニックのバグを除去 (&)
		value = value.replaceFirst("\\(\\&\\)", "");

		return value;
	}

	//////////////////////////////////////////////////////////////////////////
	// 以下、public static ユーティリティー・メソッド

	/**
	 * 英語リソース文字列からニーモニック制御文字 & を除去します。
	 * ニーモニック対象文字自体は除去されません。
	 * <p>
	 * @param enValue 英語リソース文字列
	 * @return ニーモニック制御文字除去後の文字列
	 */
	public static String removeEnMnemonic(String enValue) {

		if (hasEnMnemonic(enValue)) {
			return enValue.replaceFirst(
					"\\&(" + MNEMONIC_CHARS + ")", "$1");
		}
		return enValue;
	}

	/**
	 * (&A) のような日本用のニーモニックが既に含まれているか判定します。
	 * <p>
	 * @param value リソース文字列
	 * @return 含まれている場合は true
	 */
	public static boolean hasJaMnemonic(String value) {
		return value.matches(
				"(?s)^.*?\\(\\&" + MNEMONIC_CHARS + "\\).*$");
	}

	/**
	 * 英語ニーモニックが含まれているか判定します。
	 * &lt; などはニーモニックではないので false を返します。
	 * <p>
	 * @param value 値
	 * @return 含まれている場合は true
	 */
	public static boolean hasEnMnemonic(String value) {

		// &nbsp; などがある場合、ニーモニックが含まれないことを前提とした判定。
		// これ以外の & は日本語文字列に & が含まれるかを辞書で判断。
		return
			value.contains("&") &&
			!value.contains("& ") && // & の後に半角空白
			!value.contains("&lt;") &&
			!value.contains("&gt;") &&
			!value.contains("&amp;") &&
			!value.contains("&quot;") &&
			!value.contains("&nbsp;");
	}

	/**
	 * 文字列をプロパティーのキー（左辺）文字列表現に変換します。
	 * <p>
	 * @param key 文字列
	 * @return プロパティーのキー文字列表現
	 */
	public static String toPropertyKey(String key) {
		return key
			.replaceAll("[ :=#!\t\\\\]", "\\\\$0")
			.replaceAll("\r", "\\\\r")
			.replaceAll("\n", "\\\\n");
	}

	/**
	 * 文字列をプロパティーの値（右辺）文字列表現に変換します。
	 * <p>
	 * @param value 文字列
	 * @return プロパティーの値文字列表現
	 */
	public static String toPropertyValue(String value) {
		return value
			.replaceAll("#", "\\\\$0")
			.replaceAll("\r", "\\\\r")
			.replaceAll("\n", "\\\\n");
	}
}
