/*
 * Copyright (c) 2005- Shinji Kashihara.
 * All rights reserved. This program are made available under
 * the terms of the Eclipse Public License v1.0 which accompanies
 * this distribution, and is available at epl-v10.html.
 */
package jp.sourceforge.mergedoc.pleiades.util;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Properties;
import java.util.Map.Entry;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;

/**
 * 高速プロパティー・クラスです。 
 * このクラスは java.util.Properties から同期処理を省いて高速化したものです。
 * <p>
 * @author cypher256
 */
public class FastProperties extends HashMap<String, String> {

	// このクラスは起動初期段階で使用されるため、ロガーは使用しない
	
	/**
	 * コンストラクタです。
	 */
	public FastProperties() {
	}

	/**
	 * コンストラクタです。
	 * <p>
	 * @param path パス
	 */
	public FastProperties(String path) {
		load(path);
	}

	/**
	 * コンストラクタです。
	 * <p>
	 * @param file プロパティー・ファイルまたはディレクトリー
	 */
	public FastProperties(File file) {
		load(file);
	}

	/**
	 * プロパティーを追加します。
	 * @param prop プロパティー
	 */
	public void putProperties(Properties prop) {
		for (Entry<Object, Object> entry : prop.entrySet()) {
			put((String) entry.getKey(), (String) entry.getValue());
		}
	}
	
	/**
	 * プロパティー・ファイルをロードします。
	 * <p>
	 * @param path パス
	 */
	public void load(String path) {

		File resourceFile = Files.getResourceFile(path);
		load(resourceFile);
	}

	/**
	 * プロパティー・ファイルをロードします。<br>
	 * パスがディレクトリーの場合は、その直下にあるプロパティー・ファイルが
	 * すべてロードされます。
	 * <p>
	 * @param file プロパティー・ファイルまたはディレクトリー
	 */
	public void load(File file) {

		// ディレクトリーの場合
		if (file.isDirectory()) {

			File[] files = file.listFiles();
			Arrays.sort(files); // 昇順

			for (File f: files) {
				if (f.isFile() && f.getName().endsWith(".properties")) {
					loadFile(f);
				}
			}
		}
		// ファイルの場合
		else {
			loadFile(file);
		}
	}

	/**
	 * プロパティー・ファイルをロードします。
	 * 引数の入力ストリームはクローズされません。
	 * <p>
	 * @param is 入力ストリーム
	 */
	public void load(InputStream is) {

		try {
			loadInternal(is);
			
		} catch (IOException e) {
			throw new IllegalArgumentException(e);
		}
	}

	/**
	 * 指定されたプロパティーをソートしてファイルに保管します。
	 * <p>
	 * @param path パス
	 * @param title ヘッダーに出力するタイトル
	 */
	public void store(String path, String title) {

		File resourceFile = Files.getResourceFile(path);
		store( resourceFile, title);
	}

	/**
	 * 指定されたプロパティーをソートしてファイルに保管します。
	 * <p>
	 * @param file ファイル
	 * @param title ヘッダーに出力するタイトル
	 */
	public void store(File file, String title) {

		// 既存ファイルと内容が同じであれば保存しない (コメントの日付更新抑止)
		if (file.exists()) {
			FastProperties old = new FastProperties(file);
			if (old.equals(this)) {
				System.out.println("既存ファイルと内容が同じであるため保存しません。" + file);
				// 注）ロガーはログ設定前に初期化されてしまうため使用しない
				return;
			}
		}
		
		OutputStream os = null;
		try {
			os = new BufferedOutputStream(new FileOutputStream(file));
			
			List<String> headerList = new LinkedList<String>();
			headerList.add(saveConvertComment("このファイルは Pleiades により生成されました。"
					+ size() + " エントリー。"));
			headerList.add("###############################################################################");
			headerList.add("# ");
			for (String s : title.split("\\n")) {
				headerList.add("# " + saveConvertComment(s));
			}
			headerList.add("# ");
			headerList.add("###############################################################################");
			String header = StringUtils.join(headerList, "\n");
			store(os, header);
			
		} catch (IOException e) {
			throw new IllegalArgumentException(file.toString(), e);

		} finally {
			IOUtils.closeQuietly(os);
		}
	}

	// ------------------------------------------------------- protected
	
	protected InputStream getInputStream(File file) throws FileNotFoundException {
		return new BufferedInputStream(new FileInputStream(file));
	}

	// ------------------------------------------------------- private

	private static final long serialVersionUID = 1L;

	private void loadFile(File file) {

		InputStream is = null;
		try {
			is = getInputStream(file);
			if (is == null) {
				return;
			}
			loadInternal(is);
			
		} catch (IOException e) {
			throw new IllegalArgumentException(e);
			
		} finally {
			IOUtils.closeQuietly(is);
		}
	}
	
	private void loadInternal(InputStream inStream) throws IOException {
		
		char[] convtBuf = new char[1024];
		LineReader lr = new LineReader(inStream);

		int limit;
		int keyLen;
		int valueStart;
		char c;
		boolean hasSep;
		boolean precedingBackslash;

		while ((limit = lr.readLine()) >= 0) {
			c = 0;
			keyLen = 0;
			valueStart = limit;
			hasSep = false;

			precedingBackslash = false;
			while (keyLen < limit) {
				c = lr.lineBuf[keyLen];
				if ((c == '=' || c == ':') && !precedingBackslash) {
					valueStart = keyLen + 1;
					hasSep = true;
					break;
				} else if ((c == ' ' || c == '\t' || c == '\f')
						&& !precedingBackslash) {
					valueStart = keyLen + 1;
					break;
				}
				if (c == '\\') {
					precedingBackslash = !precedingBackslash;
				} else {
					precedingBackslash = false;
				}
				keyLen++;
			}
			while (valueStart < limit) {
				c = lr.lineBuf[valueStart];
				if (c != ' ' && c != '\t' && c != '\f') {
					if (!hasSep && (c == '=' || c == ':')) {
						hasSep = true;
					} else {
						break;
					}
				}
				valueStart++;
			}
			String key = loadConvert(lr.lineBuf, 0, keyLen, convtBuf);
			String value = loadConvert(lr.lineBuf, valueStart, limit
					- valueStart, convtBuf);
			put(key, value);
		}
	}

	private void store(OutputStream out, String comments) throws IOException {
		
		BufferedWriter writer = new BufferedWriter(new OutputStreamWriter(out, "8859_1"));
		if (comments != null) {
			writeln(writer, "#" + comments);
		}
		writeln(writer, "#" + new Date().toString());

		// キーでソート
		List<String> keyList = new ArrayList<String>(keySet());
		Collections.<String>sort(keyList, String.CASE_INSENSITIVE_ORDER);
		
		for (String key : keyList) {
			String val = get(key);
			key = saveConvert(key, true);
			val = saveConvert(val, false);
			writeln(writer, key + "=" + val);
		}
		writer.flush();
	}
	
	class LineReader {
		
		public LineReader(InputStream inStream) {
			this.inStream = inStream;
		}

		byte[] inBuf = new byte[8192];
		char[] lineBuf = new char[1024];
		int inLimit = 0;
		int inOff = 0;
		InputStream inStream;

		int readLine() throws IOException {
			int len = 0;
			char c = 0;

			boolean skipWhiteSpace = true;
			boolean isCommentLine = false;
			boolean isNewLine = true;
			boolean appendedLineBegin = false;
			boolean precedingBackslash = false;
			boolean skipLF = false;

			while (true) {
				if (inOff >= inLimit) {
					inLimit = inStream.read(inBuf);
					inOff = 0;
					if (inLimit <= 0) {
						if (len == 0 || isCommentLine) {
							return -1;
						}
						return len;
					}
				}
				c = (char) (0xff & inBuf[inOff++]);
				if (skipLF) {
					skipLF = false;
					if (c == '\n') {
						continue;
					}
				}
				if (skipWhiteSpace) {
					if (c == ' ' || c == '\t' || c == '\f') {
						continue;
					}
					if (!appendedLineBegin && (c == '\r' || c == '\n')) {
						continue;
					}
					skipWhiteSpace = false;
					appendedLineBegin = false;
				}
				if (isNewLine) {
					isNewLine = false;
					if (c == '#' || c == '!') {
						isCommentLine = true;
						continue;
					}
				}

				if (c != '\n' && c != '\r') {
					lineBuf[len++] = c;
					if (len == lineBuf.length) {
						int newLength = lineBuf.length * 2;
						if (newLength < 0) {
							newLength = Integer.MAX_VALUE;
						}
						char[] buf = new char[newLength];
						System.arraycopy(lineBuf, 0, buf, 0, lineBuf.length);
						lineBuf = buf;
					}
					if (c == '\\') {
						precedingBackslash = !precedingBackslash;
					} else {
						precedingBackslash = false;
					}
				} else {
					if (isCommentLine || len == 0) {
						isCommentLine = false;
						isNewLine = true;
						skipWhiteSpace = true;
						len = 0;
						continue;
					}
					if (inOff >= inLimit) {
						inLimit = inStream.read(inBuf);
						inOff = 0;
						if (inLimit <= 0) {
							return len;
						}
					}
					if (precedingBackslash) {
						len -= 1;
						skipWhiteSpace = true;
						appendedLineBegin = true;
						precedingBackslash = false;
						if (c == '\r') {
							skipLF = true;
						}
					} else {
						return len;
					}
				}
			}
		}
	}

	private String loadConvert(char[] in, int off, int len, char[] convtBuf) {
		if (convtBuf.length < len) {
			int newLen = len * 2;
			if (newLen < 0) {
				newLen = Integer.MAX_VALUE;
			}
			convtBuf = new char[newLen];
		}
		char aChar;
		char[] out = convtBuf;
		int outLen = 0;
		int end = off + len;

		while (off < end) {
			aChar = in[off++];
			if (aChar == '\\') {
				aChar = in[off++];
				if (aChar == 'u') {
					int value = 0;
					for (int i = 0; i < 4; i++) {
						aChar = in[off++];
						switch (aChar) {
						case '0':
						case '1':
						case '2':
						case '3':
						case '4':
						case '5':
						case '6':
						case '7':
						case '8':
						case '9':
							value = (value << 4) + aChar - '0';
							break;
						case 'a':
						case 'b':
						case 'c':
						case 'd':
						case 'e':
						case 'f':
							value = (value << 4) + 10 + aChar - 'a';
							break;
						case 'A':
						case 'B':
						case 'C':
						case 'D':
						case 'E':
						case 'F':
							value = (value << 4) + 10 + aChar - 'A';
							break;
						default:
							throw new IllegalArgumentException(
									"Malformed \\uxxxx encoding.");
						}
					}
					out[outLen++] = (char) value;
				} else {
					if (aChar == 't')
						aChar = '\t';
					else if (aChar == 'r')
						aChar = '\r';
					else if (aChar == 'n')
						aChar = '\n';
					else if (aChar == 'f')
						aChar = '\f';
					out[outLen++] = aChar;
				}
			} else {
				out[outLen++] = (char) aChar;
			}
		}
		return new String(out, 0, outLen);
	}

	private String saveConvertComment(String theString) {
		
		String space = theString.replaceFirst("(\\s*).*", "$1");
		return space + saveConvert(theString.trim(), false);
	}
	
	private String saveConvert(String theString, boolean escapeSpace) {
		
		int len = theString.length();
		int bufLen = len * 2;
		if (bufLen < 0) {
			bufLen = Integer.MAX_VALUE;
		}
		StringBuffer outBuffer = new StringBuffer(bufLen);

		for (int x = 0; x < len; x++) {
			char aChar = theString.charAt(x);
			if ((aChar > 61) && (aChar < 127)) {
				if (aChar == '\\') {
					outBuffer.append('\\');
					outBuffer.append('\\');
					continue;
				}
				outBuffer.append(aChar);
				continue;
			}
			switch (aChar) {
			case ' ':
				if (x == 0 || escapeSpace)
					outBuffer.append('\\');
				outBuffer.append(' ');
				break;
			case '\t':
				outBuffer.append('\\');
				outBuffer.append('t');
				break;
			case '\n':
				outBuffer.append('\\');
				outBuffer.append('n');
				break;
			case '\r':
				outBuffer.append('\\');
				outBuffer.append('r');
				break;
			case '\f':
				outBuffer.append('\\');
				outBuffer.append('f');
				break;
			case '=': // Fall through
			case ':': // Fall through
			case '#': // Fall through
			case '!':
				outBuffer.append('\\');
				outBuffer.append(aChar);
				break;
			default:
				if ((aChar < 0x0020) || (aChar > 0x007e)) {
					outBuffer.append('\\');
					outBuffer.append('u');
					outBuffer.append(toHex((aChar >> 12) & 0xF));
					outBuffer.append(toHex((aChar >> 8) & 0xF));
					outBuffer.append(toHex((aChar >> 4) & 0xF));
					outBuffer.append(toHex(aChar & 0xF));
				} else {
					outBuffer.append(aChar);
				}
			}
		}
		return outBuffer.toString();
	}

	private static void writeln(BufferedWriter bw, String s) throws IOException {
		bw.write(s);
		bw.newLine();
	}

	private static char toHex(int nibble) {
		return hexDigit[(nibble & 0xF)];
	}

	private static final char[] hexDigit = { '0', '1', '2', '3', '4', '5', '6',
			'7', '8', '9', 'A', 'B', 'C', 'D', 'E', 'F' };
}
