/*
 * Copyright (c) 2005- Shinji Kashihara.
 * All rights reserved. This program are made available under
 * the terms of the Eclipse Public License v1.0 which accompanies
 * this distribution, and is available at epl-v10.html.
 */
package jp.sourceforge.mergedoc.pleiades.generator.xls;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;

import jp.sourceforge.mergedoc.pleiades.util.FastProperties;
import jp.sourceforge.mergedoc.pleiades.util.Files;
import jp.sourceforge.mergedoc.pleiades.util.UnMnemonicProperties;

import org.apache.poi.hssf.usermodel.HSSFCell;
import org.apache.poi.hssf.usermodel.HSSFRow;
import org.apache.poi.hssf.usermodel.HSSFSheet;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.poifs.filesystem.POIFSFileSystem;

/**
 * Excel ファイルを読み込み、プロパティー・ファイルを作成します。
 * <p>
 * このクラスは訳の提供を Excel ファイルで受けた時に、Pleiades
 * プロパティー形式に変換するためのものです。
 * <p>
 * @author cypher256
 */
public class ExcelToProperties {

	/** 入力 Excel ディレクトリー */
	private static final File IN_EXCEL_DIR = Files.getResourceFile("props/excel");
	
	/** 出力プロパティー・ディレクトリー */
	private static final File OUT_PROP_DIR = Files.getResourceFile("props/additions");
	
	/** Excel のヘッダー行数 */
	private static final int HEADER_ROWS = 1;

	/** 英語カラム位置 */
	private static final short EN_COLUMN_POS = 3;

	/** 日本語カラム位置 */
	private static final short JA_COLUMN_POS = 4;
	
	/**
	 * 開始するための main メソッドです。
	 * <p>
	 * @param args 起動引数
	 * @throws IOException 入出力例外が発生した場合
	 */
	public static void main(String... args) throws IOException {
		new ExcelToProperties().run();
	}

	/**
	 * 実行します。
	 * <p>
	 * @throws IOException 入出力例外が発生した場合
	 */
	private void run() throws IOException {
		
		for (File excelFile : IN_EXCEL_DIR.listFiles()) {
			writeProperties(excelFile);
		}
	}
	
	/**
	 * 指定された Excel ファイルをプロパティー・ファイルに書き出します。
	 * <p>
	 * @param excelFile Excel ファイル
	 * @throws IOException 入出力例外が発生した場合
	 */
	protected void writeProperties(File excelFile) throws IOException {
		
		// 入力 Excel ファイルの読み込み
		POIFSFileSystem fs = new POIFSFileSystem(new FileInputStream(excelFile));
		HSSFWorkbook wb = new HSSFWorkbook(fs);
		HSSFSheet sheet = wb.getSheetAt(0);
		
		// 出力プロパティー
		FastProperties prop = new UnMnemonicProperties();

		// 行ごとに、データを取得
		for (int rowIdx = HEADER_ROWS;; rowIdx++) {

			HSSFRow row = sheet.getRow(rowIdx);
			if (row == null) {
				break;
			}
			HSSFCell en = row.getCell(EN_COLUMN_POS);
			HSSFCell ja = row.getCell(JA_COLUMN_POS);
			if (en == null || ja == null) {
				break;
			}
			String enValue = en.getRichStringCellValue().getString();
			String jaValue = ja.getRichStringCellValue().getString();
			
			prop.put(enValue, jaValue);
		}
		
		// 出力結果プロパティー書き込み
		String propFileName = "_" + excelFile.getName().replace(".xls", ".properties");
		File propFile = new File(OUT_PROP_DIR, propFileName);
		prop.store(propFile,
			"[#99999] contributed by xxx\n" +
			"blancoNLpackGenerator: TranslationFragment to Pleiades.");
	}
}
