/*
 * Copyright (c) 2005- Shinji Kashihara.
 * All rights reserved. This program are made available under
 * the terms of the Eclipse Public License v1.0 which accompanies
 * this distribution, and is available at epl-v10.html.
 */
package jp.sourceforge.mergedoc.pleiades.aspect.resource;

import java.io.File;
import java.io.FileNotFoundException;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import jp.sourceforge.mergedoc.pleiades.aspect.Pleiades;
import jp.sourceforge.mergedoc.pleiades.log.Logger;
import jp.sourceforge.mergedoc.pleiades.util.FastProperties;
import jp.sourceforge.mergedoc.pleiades.util.Files;

/**
 * 翻訳用の正規表現辞書です。
 * <p>
 * @author cypher256
 */
public class RegexDictionary {

	/** ロガー */
	private static final Logger log = Logger.getLogger(RegexDictionary.class);

	/** 翻訳正規表現プロパティー・ファイル名 */
	public static final String PROP_FILE_NAME = "translation-regex.properties";

	/** このクラスのシングルトン・インスタンス */
	private static final RegexDictionary singleton = new RegexDictionary();

	/**
	 * 正規表現辞書オブジェクトを取得します。
	 * <p>
	 * @return 翻訳正規表現プロパティー・オブジェクト
	 */
	public static RegexDictionary getInstance() {
		return singleton;
	}

	/** 翻訳正規表現マップ */
	private final Map<Pattern, String> map = new HashMap<Pattern, String>();

	/**
	 * 正規表現辞書を構築します。
	 */
	private RegexDictionary() {

		File file = Files.getResourceFile(PROP_FILE_NAME);
		if (!file.exists()) {
			Exception e = new FileNotFoundException(file.getPath());
			Exception ise = new IllegalStateException("翻訳正規表現プロパティーが見つかりません。", e);
			Pleiades.abort(ise);
		}

		FastProperties prop = new FastProperties(file);
		for (Entry<String, String> entry : prop.entrySet()) {
			map.put(
				Pattern.compile((String) entry.getKey()),
				(String) entry.getValue());
		}
		log.info("翻訳正規表現プロパティーをロードしました。" + map.size());
	}

	/**
	 * 翻訳します。
	 * <p>
	 * @param enValue 英語文字列
	 * @return 翻訳後日本語文字列
	 */
	public String lookup(String enValue) {

		for (Entry<Pattern, String> entry : map.entrySet()) {

			Pattern pattern = entry.getKey();
			Matcher matcher = pattern.matcher(
					Matcher.quoteReplacement(enValue));
			try {
				if (matcher.matches()) {
					String replacement = entry.getValue();
					return matcher.replaceFirst(replacement);
				}
			} catch (RuntimeException e) {
				log.error("正規表現翻訳でエラーが発生しました。" + pattern, e);
				break;
			}
		}
		return null;
	}
}
