/*
 * Copyright (c) 2005- Shinji Kashihara.
 * All rights reserved. This program are made available under
 * the terms of the Eclipse Public License v1.0 which accompanies
 * this distribution, and is available at epl-v10.html.
 */
package jp.sourceforge.mergedoc.pleiades.aspect;

import java.io.File;
import java.io.IOException;
import java.security.ProtectionDomain;
import java.util.concurrent.ExecutorService;

import javassist.CannotCompileException;
import javassist.CtBehavior;
import javassist.CtClass;
import javassist.CtConstructor;
import javassist.NotFoundException;
import jp.sourceforge.mergedoc.pleiades.aspect.resource.DynamicTranslationDictionary;
import jp.sourceforge.mergedoc.pleiades.log.Logger;

/**
 * 翻訳処理を埋め込むためのバイトコード変換を行うトランスフォーマーです。
 * <p>
 * @author cypher256
 */
public class TranslationTransformer extends AbstractTransformer {

	/** ロガー */
	private static final Logger log = Logger.getLogger(TranslationTransformer.class);

	/** ロック・ファイル */
	private static final File lockFile = new File(Pleiades.getConfigurationPath(), ".lock");
	
	/**
	 * 翻訳トランスフォーマーを構築します。
	 */
	public TranslationTransformer() {
		load();
	}

	/**
	 * 翻訳に必要なプロパティーをロードします。
	 */
	protected void load() {
		
		// ロック・ファイルの終了時削除予約
		lockFile.deleteOnExit();
		Runtime.getRuntime().addShutdownHook(new Thread() {
			@Override
			public void run() {
				unlock();
			}
		});
		
		// 非同期タスク実行サービスを取得
		ExecutorService executorService = Pleiades.getExecutorService();

		// ロック
		lock();
		
		// 変換除外クラス名キャッシュをロード
		executorService.execute(new Runnable(){
			public void run() {
				ExcludesClassNameCache.getInstance();
			}
		});
		// 変換済みクラス・キャッシュをロード
		executorService.execute(new Runnable(){
			public void run() {
				TransformedClassCache.getInstance();
			}
		});
		// 翻訳辞書（キャッシュの場合あり）をロード
		executorService.execute(new Runnable(){
			public void run() {
				DynamicTranslationDictionary.getInstance();
				
				// ロック解除
				// 最も時間がかかるこのファイル・ロード後にロック解除。
				// 他のファイルのロードが遅かった場合でも解除される。
				unlock();
			}
		});
	}

	/**
	 * バイトコードを変換します。
	 * <p>
	 * {@link TranslationEditor 翻訳エディター} を使用し、バイトコードを変換します。
	 * ただし、{@link ExcludesClassNameCache 変換除外クラス名キャッシュ}
	 * に処理するクラス名が含まれる場合、何も行いません。
	 */
	@Override
	protected byte[] transform(
			ClassLoader loader,
			String className,
			ProtectionDomain protectionDomain,
			byte[] bytecode)
		throws CannotCompileException, NotFoundException, IOException {

		// 変換除外クラスの場合は何もしない
		ExcludesClassNameCache excludeList = ExcludesClassNameCache.getInstance();
		if (excludeList.contains(className)) {
			return null;
		}

		// 変換済みクラス・キャッシュがある場合はそれを返す
		TransformedClassCache classCache = TransformedClassCache.getInstance();
		byte[] cachedBytecode = classCache.get(className);
		if (cachedBytecode != null) {
			return cachedBytecode;
		}

		// バイトコードに翻訳アスペクトを埋め込み
		CtClass ctClass = createCtClass(bytecode, protectionDomain);
		byte[] transformedBytecode = transformClass(ctClass);

		// 次回起動用の情報を作成
		if (transformedBytecode == null) {
			// 変換対象外の場合は、変換除外リストに追加
			excludeList.add(className);
		} else {
			// 変換した場合は、変換済みクラス・キャッシュに追加
			classCache.put(className, transformedBytecode);
		}
		
		return transformedBytecode;
	}

	/**
	 * クラス・オブジェクトに翻訳アスペクトを埋め込みます。
	 * <p>
	 * @param ctClass 変換対象クラス・オブジェクト
	 * @return 埋め込み後のバイトコード。埋め込み対象でない場合は null。
	 */
	protected byte[] transformClass(CtClass ctClass)
		throws CannotCompileException, NotFoundException, IOException {

		TranslationEditor editor = new TranslationEditor(ctClass);

		// コンストラクター、メソッドの変換
		for (CtBehavior ctBehavior : ctClass.getDeclaredBehaviors()) {

			// コードを検査し、呼び出し部分を編集
			ctBehavior.instrument(editor);

			// メソッドを編集
			editor.editBehavior(ctBehavior);
		}

		// スタティック・イニシャライザーの変換
		CtConstructor ctInitializer = ctClass.getClassInitializer();
		if (ctInitializer != null) {
			ctInitializer.instrument(editor);
		}
		return editor.toBytecode();
	}

	// ------------------------------------------------------------------------
	// 以下、Eclipse から呼び出される public static メソッド

	/**
	 * 翻訳トランスフォーマーを破棄します。
	 */
	public static void destroy() {

		// ロック
		lock();
		try {
			// 翻訳プロパティーをキャッシュとして保存
			DynamicTranslationDictionary.getInstance().store();
			
			// 変換済みクラス・キャッシュを保存
			TransformedClassCache.getInstance().store();
			
			// 変換除外クラス名キャッシュを保存
			ExcludesClassNameCache.getInstance().store();
			
			// タスク実行サービスは停止済みのはずだが念のため強制停止
			Pleiades.getExecutorService().shutdownNow();
			
		} catch (Exception e) {
			log.error("トランスフォーマーの破棄でエラーが発生しました。", e);
			
		} finally {
			// ロック解除
			unlock();
		}
	}

	/**
	 * ロックします。
	 * 完全なロックをサポートするわけではありません。
	 */
	private static void lock() {
		
		final int MAX_WAIT = 30;
		try {
			for (int i = 1; i <= MAX_WAIT && lockFile.exists(); i++) {
				Thread.sleep(1000);
				log.debug("ロック解除待機中です。(" + i + "/" + MAX_WAIT + " 秒)");
			}
		} catch (Exception e) {
			log.error("ロック待ちで割り込みが発生しました。", e);
		}
		try {
			lockFile.createNewFile();
			log.debug("ロックしました。");
		} catch (Exception e) {
			log.error("ロック・ファイルの作成に失敗しました。", e);
		}
	}
	
	/**
	 * ロックを解除します。
	 * 完全なロックをサポートするわけではありません。
	 */
	private static void unlock() {
		try {
			lockFile.delete();
			log.debug("ロック解除しました。");
		} catch (Exception e) {
			log.error("ロック・ファイルの削除に失敗しました。", e);
		}
	}
}
