/*
 * Copyright (c) 2005- Shinji Kashihara.
 * All rights reserved. This program are made available under
 * the terms of the Eclipse Public License v1.0 which accompanies
 * this distribution, and is available at epl-v10.html.
 */
package jp.sourceforge.mergedoc.pleiades.aspect;

import java.io.File;
import java.io.IOException;
import java.lang.instrument.Instrumentation;
import java.lang.reflect.Method;
import java.security.ProtectionDomain;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.ExecutorService;

import javassist.CannotCompileException;
import javassist.CtClass;
import javassist.CtMethod;
import javassist.NotFoundException;
import jp.sourceforge.mergedoc.pleiades.aspect.advice.AspectMapping;
import jp.sourceforge.mergedoc.pleiades.aspect.resource.RegexDictionary;
import jp.sourceforge.mergedoc.pleiades.aspect.resource.TranslationExcludeProperties;
import jp.sourceforge.mergedoc.pleiades.log.Logger;
import jp.sourceforge.mergedoc.pleiades.util.FastProperties;
import jp.sourceforge.mergedoc.pleiades.util.Files;

/**
 * Eclipse 起動時のバイトコード変換を行うトランスフォーマーです。
 * <p>
 * @author cypher256
 */
public class LauncherTransformer extends AbstractTransformer {

	/** ロガー */
	private static final Logger log = Logger.getLogger(LauncherTransformer.class);

	/** このクラス名 (static 呼び出し用) */
	private final String transformerClass;
	
	/**
	 * 起動トランスフォーマーを構築します。
	 */
	public LauncherTransformer() {
		load();
		transformerClass = getClass().getName();
	}
	
	/**
	 * プロパティーをロードします。
	 */
	protected void load() {

		// アスペクト定義をロード
		// 注) 別スレッドで処理すると SAXParser でデッドロックする
		AspectMapping.getInstance();

		// -clean に依存しないものをロード (非同期実行)
		// -clean に依存するもは TranslationTransformer でロード
		Pleiades.getExecutorService().execute(new Runnable(){
			public void run() {

				// 翻訳除外プロパティーをロード
				TranslationExcludeProperties.getInstance();
				
				// 正規表現翻訳プロパティーをロード
				RegexDictionary.getInstance();
			}
		});
	}

	/**
	 * バイトコード変換を行います。
	 */
	@Override
	protected byte[] transform(
			ClassLoader loader,
			String className,
			ProtectionDomain protectionDomain,
			byte[] bytecode)
		throws CannotCompileException, NotFoundException, IOException {

		//-------------------------------------------------
		// Main クラスの変換
		// 3.2 以前： org.eclipse.core.launcher.Main
		// 3.3 以降： org.eclipse.equinox.launcher.Main
		//-------------------------------------------------
		if (className.endsWith(".launcher.Main")) {

			// 変換
			return transformMainClass(protectionDomain, bytecode);
		}
		//-------------------------------------------------
		// Workbench クラスの変換
		//-------------------------------------------------
		else if (className.equals("org.eclipse.ui.internal.Workbench")) {

			// metadata デフォルト初期化 (非同期実行)
			ExecutorService executorService = Pleiades.getExecutorService();
			executorService.execute(new Runnable(){
				public void run() {
					MetadataDefaultInitializer.init();
				}
			});
			
			// 非同期タスク実行サービスをシャットダウン予約
			executorService.shutdown();

			// 変換
			return transformWorkbenchClass(protectionDomain, bytecode);
		}
		//-------------------------------------------------
		// WorkbenchWindows クラスの変換
		//-------------------------------------------------
		else if (className.equals("org.eclipse.ui.internal.WorkbenchWindow")) {

			// このトランスフォーマーをエージェントから削除
			Pleiades.getInstrumentation().removeTransformer(this);

			// 変換
			return transformWorkbenchWindowClass(protectionDomain, bytecode);
		}

		return null;
	}

	/**
	 * Main クラスを変換します。
	 * <p>
	 * @param protectionDomain プロテクション・ドメイン
	 * @param bytecode バイトコード
	 * @return 変換後のバイトコード
	 */
	protected byte[] transformMainClass(ProtectionDomain protectionDomain, byte[] bytecode)
		throws CannotCompileException, NotFoundException, IOException {

		// Eclipse Main クラスに開始処理を追加
		CtClass clazz = createCtClass(bytecode, protectionDomain);
		CtMethod method = clazz.getMethod("basicRun", "([Ljava/lang/String;)V");
		method.insertBefore("$1 = " + transformerClass + ".startTranslationTransformer($$);");

		// スプラッシュ画像パスを変更
		PleiadesOption agentOption = Pleiades.getPleiadesOption();
		if (!agentOption.isDefaultSplash()) {
			String splashLocation = getSplashLocation();
			if (splashLocation != null) {
				System.setProperty("osgi.splashLocation", splashLocation);
			}
		}
		return clazz.toBytecode();
	}

	/**
	 * Workbench クラスを変換します。
	 * <p>
	 * @param protectionDomain プロテクション・ドメイン
	 * @param bytecode バイトコード
	 * @return 変換後のバイトコード
	 */
	protected byte[] transformWorkbenchClass(ProtectionDomain protectionDomain, byte[] bytecode)
		throws CannotCompileException, NotFoundException, IOException {

		CtClass clazz = createCtClass(bytecode, protectionDomain);
		
		// 終了時の destroy
		CtMethod shutdown = clazz.getMethod("shutdown", "()V");
		shutdown.insertBefore(transformerClass + ".destroy();");
		
		// イベントループ開始前の処理
		CtMethod runEventLoop = clazz.getMethod("runEventLoop",
			"(Lorg/eclipse/jface/window/Window$IExceptionHandler;Lorg/eclipse/swt/widgets/Display;)V");
		runEventLoop.insertBefore(transformerClass + ".beforeEventLoop($0);");
		
		return clazz.toBytecode();
	}

	/**
	 * WorkbenchWindow クラスを変換します。
	 * <p>
	 * @param protectionDomain プロテクション・ドメイン
	 * @param bytecode バイトコード
	 * @return 変換後のバイトコード
	 */
	protected byte[] transformWorkbenchWindowClass(ProtectionDomain protectionDomain, byte[] bytecode)
		throws CannotCompileException, NotFoundException, IOException {

		CtClass clazz = createCtClass(bytecode, protectionDomain);
		
		// ワークベンチ復元前の処理
		CtMethod restore = clazz.getMethod("restoreState",
			"(Lorg/eclipse/ui/IMemento;Lorg/eclipse/ui/IPerspectiveDescriptor;)Lorg/eclipse/core/runtime/IStatus;");
		restore.insertBefore(transformerClass + ".beforeRestoreState($1);");
		
		return clazz.toBytecode();
	}

	// ------------------------------------------------------------------------
	// 以下、Eclipse から呼び出される public static メソッド
	
	/**
	 * ワークベンチ復元前の処理です。
	 * <p>
	 * @param memento org.eclipse.ui.IMemento インスタンス
	 * @throws Exception 例外が発生した場合
	 */
	public static void beforeRestoreState(Object memento) throws Exception {
		
		if (isPreload()) {
			
			// プリロード起動時の表示位置に座標外の値をセット
			Method putInteger = memento.getClass().getMethod("putInteger", String.class, int.class);
			putInteger.invoke(memento, "width", 200);
			putInteger.invoke(memento, "x", -199);
		}
	}

	/**
	 * プリロード起動か判定します。
	 * @return プリロード起動の場合は true
	 */
	private static boolean isPreload() {
		return System.getProperty("pleiades.preload") != null;
	}
	
	/**
	 * イベントループ開始前の処理です。
	 * <p>
	 * @param workbench org.eclipse.ui.internal.Workbench インスタンス
	 * @throws Exception 例外が発生した場合
	 */
	public static void beforeEventLoop(final Object workbench) throws Exception {
		
		if (log.isDebugEnabled()) {
			
			// 始動完了ログ
			LauncherLoggingTransformer.logUpTime();
			
			// システム・プロパティーをログとして出力
			FastProperties systemProp = new FastProperties();
			systemProp.putProperties(System.getProperties());
			File file = new File(Pleiades.getConfigurationPath(), "system.properties");
			systemProp.store(file, "実行時システム・プロパティー");
		}
		
		if (isPreload()) {
			
			// プリロード用の自動終了
			try {
				Method close = workbench.getClass().getMethod("close");
				close.invoke(workbench);
			} catch (Throwable e) {
				log.error("プリロード自動終了処理でエラーが発生しました。", e);
			} finally {
				System.exit(0);
			}
		}
	}

	/**
	 * 翻訳トランスフォーマーを開始します。
	 * <p>
	 * @param args Eclipse 起動オプション配列
	 * @return 起動オプション配列
	 */
	public static String[] startTranslationTransformer(String... args) {

		try {
			List<String> argList = new LinkedList<String>(Arrays.asList(args));

			// 起動引数の -clean を取得
			PleiadesOption option = Pleiades.getPleiadesOption();
			option.setClean(argList.contains("-clean"));
			log.info("Eclipse の起動を開始しました。-clean:" + option.isClean());

			// キャッシュが無い場合は強制的に -claen を指定
			if (!option.isClean() && !ExcludesClassNameCache.file.exists()) {
				log.info("変換除外クラス名キャッシュが存在しないため、" +
					"強制的に -clean モードで起動します。");
				argList.add("-clean");
				option.setClean(true);
			}

			// 翻訳トランスフォーマーを開始
			Instrumentation inst = Pleiades.getInstrumentation();
			if (log.isDebugEnabled()) {
				inst.addTransformer(new TranslationLoggingTransformer());
			} else {
				inst.addTransformer(new TranslationTransformer());
			}

			return argList.toArray(new String[argList.size()]);

		} catch (Throwable e) {

			String msg = "翻訳トランスフォーマーの開始に失敗しました。";
			Exception ise = new IllegalStateException(msg, e);
			Pleiades.abort(ise);
			return null;
		}
	}
	
	/**
	 * トランスフォーマーを破棄します。
	 */
	public static void destroy() {

		TranslationTransformer.destroy();
	}

	/**
	 * スプラッシュ画像パスを取得します。
	 * @return スプラッシュ画像パス。RCP の場合は null。
	 */
	public static String getSplashLocation() {

		// RCP の場合
		File home = Pleiades.getEclipseHome();
		File eclipse = new File(home, "eclipse");
		File eclipseExe = new File(home, "eclipse.exe");
		if (!eclipse.exists() && !eclipseExe.exists()) {
			return null;
		}
		
		String version = "";
		File pluginsFolder = new File(home, "plugins");
		for (File folder : pluginsFolder.listFiles()) {
			String folderName = folder.getName();
			if (folderName.startsWith("org.eclipse.osgi_")) {
				version = folderName.replaceFirst(".*?_(\\d+\\.\\d+).*", "$1");
				break;
			}
		}
		String fileName = "splash" + version + ".bmp";

		File file = Files.getResourceFile(fileName);
		if (!file.exists()) {
			fileName = "splash.bmp";
		}
		File splashFile = Files.getResourceFile(fileName);
		return splashFile.getAbsolutePath().replace('\\', '/');
	}
}
