/*
 * Copyright (c) 2005- Shinji Kashihara.
 * All rights reserved. This program are made available under
 * the terms of the Eclipse Public License v1.0 which accompanies
 * this distribution, and is available at epl-v10.html.
 */
package jp.sourceforge.mergedoc.pleiades.aspect;

import java.io.IOException;
import java.lang.instrument.IllegalClassFormatException;
import java.lang.reflect.Field;
import java.security.ProtectionDomain;

import javassist.CannotCompileException;
import javassist.CtClass;
import javassist.NotFoundException;
import jp.sourceforge.mergedoc.pleiades.log.Logger;

/**
 * 時間計測ログ機能付きの LauncherTransformer です。
 * <p>
 * @author cypher256
 */
public class LauncherLoggingTransformer extends LauncherTransformer implements ITimingLogger {

	/** ロガー */
	private static final Logger log = Logger.getLogger(LauncherLoggingTransformer.class);

	/** 開始時間 */
	private static final long startTime = System.currentTimeMillis();

	/** load メソッドの所要時間 */
	private volatile int loadTime;

	/** load メソッドの呼び出し回数 */
	private volatile int loadCall;

	/** transform (抽象クラス) メソッドの所要時間 */
	private volatile int transformAbstractTime;

	/** transform (抽象クラス) メソッドの呼び出し回数 */
	private volatile int transformAbstractCall;

	/** createCtClass (抽象クラス) メソッドの所要時間 */
	private volatile int createCtClassTime;

	/** createCtClass (抽象クラス) メソッドの呼び出し回数 */
	private volatile int createCtClassCall;

	/** transform メソッドの所要時間 */
	private volatile int transformTime;

	/** transform メソッドの呼び出し回数 */
	private volatile int transformCall;

	/** weaveUptimeMethod メソッドの所要時間 */
	private volatile int transformWorkbenchClassTime;

	/** weaveUptimeMethod メソッドの呼び出し回数 */
	private volatile int transformWorkbenchClassCall;

	/** weaveStartTranslationAspect メソッドの所要時間 */
	private volatile int transformMainClassTime;

	/** weaveStartTranslationAspect メソッドの呼び出し回数 */
	private volatile int transformMainClassCall;

	/**
	 * 起動トランスフォーマーを構築します。
	 */
	public LauncherLoggingTransformer() {
		Pleiades.registTimingLogger(this);
	}

	/**
	 * 親クラスの同メソッドに時間計測ログ出力を追加したメソッドです。
	 */
	@Override
	protected void load() {

		long startTime = System.currentTimeMillis();
		super.load();
		synchronized (this) {
			loadTime += System.currentTimeMillis() - startTime;
			loadCall++;
		}
	}

	/**
	 * 親クラスの同メソッドに時間計測ログ出力を追加したメソッドです。
	 */
	@Override
	public byte[] transform(
			ClassLoader loader,
			String internalName,
			Class<?> classBeingRedefined,
			ProtectionDomain protectionDomain,
			byte[] bytecode) throws IllegalClassFormatException {

		long startTime = System.currentTimeMillis();
		byte[] result = super.transform(loader, internalName, classBeingRedefined, protectionDomain, bytecode);
		synchronized (this) {
			transformAbstractTime += System.currentTimeMillis() - startTime;
			transformAbstractCall++;
		}
		return result;
	}

	/**
	 * バイトコード変換を行います。
	 */
	@Override
	protected byte[] transform(
			ClassLoader loader,
			String className,
			ProtectionDomain protectionDomain,
			byte[] bytecode)
		throws CannotCompileException, NotFoundException, IOException {

		long startTime = System.currentTimeMillis();
		byte[] result = super.transform(loader, className, protectionDomain, bytecode);
		synchronized (this) {
			transformTime += System.currentTimeMillis() - startTime;
			transformCall++;
		}
		return result;
	}

	/**
	 * 親クラスの同メソッドに時間計測ログ出力を追加したメソッドです。
	 */
	@Override
	protected byte[] transformMainClass(ProtectionDomain protectionDomain, byte[] bytecode)
		throws CannotCompileException, NotFoundException, IOException {

		long startTime = System.currentTimeMillis();
		byte[] result = super.transformMainClass(protectionDomain, bytecode);
		synchronized (this) {
			transformMainClassTime += System.currentTimeMillis() - startTime;
			transformMainClassCall++;
		}
		return result;
	}

	/**
	 * 親クラスの処理に加え、終了時の時間計測結果のログ出力処理を追加します。
	 */
	@Override
	protected byte[] transformWorkbenchClass(ProtectionDomain protectionDomain, byte[] bytecode)
		throws CannotCompileException, NotFoundException, IOException {

		long startTime = System.currentTimeMillis();
		byte[] result = super.transformWorkbenchClass(protectionDomain, bytecode);
		synchronized (this) {
			transformWorkbenchClassTime += System.currentTimeMillis() - startTime;
			transformWorkbenchClassCall++;
		}
		return result;
	}

	/**
	 * 親クラスの同メソッドに時間計測ログ出力を追加したメソッドです。
	 */
	@Override
	protected CtClass createCtClass(
			byte[] bytecode,
			ProtectionDomain protectionDomain) throws IOException, NotFoundException {

		long startTime = System.currentTimeMillis();
		CtClass result = super.createCtClass(bytecode, protectionDomain);
		synchronized (this) {
			createCtClassTime += System.currentTimeMillis() - startTime;
			createCtClassCall++;
		}
		return result;
	}

	/**
	 * トランスフォーマーを破棄します。
	 */
	public static void destroy() {

		// 終了時の時間計測
		logLapTime();

		// 親クラスの処理 (注：static)
		LauncherTransformer.destroy();
	}

	/**
	 * すべての登録済みトランスフォーマーの起動時間計測ログ出力します。
	 */
	public static void logUpTime() {

		// 時間計測ログを出力
		logLapTime();

		// 起動時間の出力
		// Eclipse 開始からではなく、Pleiades 開始から計測
		//long startTime = Long.valueOf(System.getProperty("eclipse.startTime"));
		long curTime = System.currentTimeMillis();
		double time = (curTime - startTime) / 1000d;
		log.debug(String.format(
				"Eclipse の起動が完了しました。起動時間: %.3f 秒", time));
	}

	/**
	 * すべての登録済みトランスフォーマーの時間計測ログ出力します。
	 */
	public static void logLapTime() {

		long summuryTime = 0;
		for (ITimingLogger logger : Pleiades.getTimeingLoggers()) {

			// 計測ログの出力
			summuryTime += logger.logTime();

			// 計測値をリセット
			logger.reset();
		}
		log.debug(String.format("Pleiades 処理時間計 %6.3f 秒", summuryTime / 1000d));
	}

	/**
	 * 時間計測ログを出力します。
	 * @return 計測時間合計 (秒)
	 */
	public long logTime() {

		String msg = "%-31s %8d 回呼出計 %6.3f 秒";

		log.debug(String.format(msg, "load",
				loadCall, loadTime / 1000d));
		log.debug(String.format(msg, "transform (Abstract)",
				transformAbstractCall, transformAbstractTime / 1000d));
		log.debug(String.format(msg, "  transform",
				transformCall, transformTime / 1000d));
		log.debug(String.format(msg, "    createCtClass (Abstract)",
				createCtClassCall, createCtClassTime / 1000d));
		log.debug(String.format(msg, "    transformMainClass",
				transformMainClassCall, transformMainClassTime / 1000d));
		log.debug(String.format(msg, "    transformWorkbenchClass",
				transformWorkbenchClassCall, transformWorkbenchClassTime / 1000d));

		long summuryTime = transformAbstractTime + transformTime + createCtClassTime +
			transformMainClassTime + transformWorkbenchClassTime;

		log.debug(String.format("クラス計 %6.3f 秒", summuryTime / 1000d));

		return summuryTime;
	}

	/**
	 * 計測値をリセットします。
	 */
	public void reset() {

		synchronized (this) {
			Field[] fields = getClass().getDeclaredFields();
			for (int i = 0; i < fields.length; i++) {
				Field field = fields[i];
				if (field.getType() == Integer.TYPE) {
					try {
						field.setAccessible(true);
						field.set(this, 0);
					} catch (Exception e) {
						log.error("フィールドの初期化に失敗しました。", e);
					}
				}
			}
		}
	}
}
