/*
 * Copyright (c) 2005- Shinji Kashihara.
 * All rights reserved. This program are made available under
 * the terms of the Eclipse Public License v1.0 which accompanies
 * this distribution, and is available at epl-v10.html.
 */
package jp.sourceforge.mergedoc.pleiades.aspect;

import java.io.File;
import java.io.IOException;
import java.util.HashSet;
import java.util.Set;

import jp.sourceforge.mergedoc.pleiades.log.Logger;

import org.apache.commons.io.FileUtils;

/**
 * 変換除外クラス名キャッシュです。
 * <p>
 * @author cypher256
 */
public class ExcludesClassNameCache {

	/** ロガー */
	private static final Logger log = Logger.getLogger(ExcludesClassNameCache.class);

	/** 変換除外クラス名キャッシュ・ファイル */
	public static final File file = new File(Pleiades.getConfigurationPath(), "excludes-class.list");

	/** このクラスのシングルトン・インスタンス */
	private static final ExcludesClassNameCache singleton = new ExcludesClassNameCache();

	/**
	 * 変換除外クラス名キャッシュ・オブジェクトを取得します。
	 * <p>
	 * @return 変換除外クラス名キャッシュ・オブジェクト
	 */
	public static ExcludesClassNameCache getInstance() {
		return singleton;
	}

	/** 変換除外クラス・セット */
	private Set<String> excludeSet = new HashSet<String>();

	/** ロード時の辞書サイズ */
	private int loadedSize;

	/**
	 * 変換除外クラス名キャッシュを構築します。
	 */
	@SuppressWarnings("unchecked")
	private ExcludesClassNameCache() {

		if (Pleiades.getPleiadesOption().isClean()) {
			log.info("-clean により変換除外クラス名キャッシュはロードされません。");

		} else  if (!file.exists()) {
			log.info("変換除外クラス名キャッシュが存在しません。");

		} else {
			try {
				excludeSet.addAll(FileUtils.readLines(file, null));
				log.info("変換除外クラス名キャッシュをロードしました。" + excludeSet.size());
			} catch (IOException e) {
				String msg = "変換除外クラス名キャッシュのロードに失敗しました。";
				Exception ise = new IllegalStateException(msg, e);
				Pleiades.abort(ise);
			}
		}
		// ロード時の辞書サイズを保存
		loadedSize = excludeSet.size();
	}

	/**
	 * 追加されたクラス名をキャッシュとして永続化します。
	 */
	public void store() {

		if (excludeSet.size() > loadedSize) {
			try {
				synchronized (ExcludesClassNameCache.class) {
					FileUtils.writeLines(file, null, excludeSet);
				}
				log.info("変換除外クラス名キャッシュを保管しました。" +
						loadedSize + " -> " + excludeSet.size());
			} catch (IOException e) {
				log.error("変換除外クラス名キャッシュの保管に失敗しました。", e);
			}
		}
	}

	/**
	 * アスペクトから除外するクラス名を追加します。
	 * <p>
	 * @param className クラス名
	 */
	public void add(String className) {
		synchronized (ExcludesClassNameCache.class) {
			excludeSet.add(className);
		}
	}

	/**
	 * アスペクトから除外するクラス名が存在するか判定します。
	 * <p>
	 * @param className クラス名
	 * @return 指定されたクラス名が存在する場合は true
	 */
	public boolean contains(String className) {
		return excludeSet.contains(className);
	}
}
