package jp.kirikiri.tjs2;

import java.io.FileDescriptor;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.ByteBuffer;

/**
 * バイナリストリーム読み書きクラス
 */
public abstract class BinaryStream {

	/** 読み込みモード */
	public static final int READ = 0;
	/** 書き込みモード */
	public static final int WRITE = 1;
	/** 追記モード */
	public static final int APPEND = 2;
	/** 更新モード */
	public static final int UPDATE = 3;

	/** アクセスモードマスク */
	public static final int ACCESS_MASK = 0x0f;

	/** 先頭からのシーク */
	public static final int SEEK_SET = 0;
	/** 現在位置からのシーク */
	public static final int SEEK_CUR = 1;
	/** 終端位置からのシーク */
	public static final int SEEK_END = 2;

	/**
	 * シークする
	 * エラー時、位置は変更されない
	 * @param offset 基準位置からのオフセット
	 * @param whence 基準位置、SEEK_SET, SEEK_CUR, SEEK_END のいずれかを指定
	 * @return シーク後の現在位置
	 * @throws TJSException
	 */
	public abstract long seek( long offset, int whence ) throws TJSException;

	/**
	 * returns actually read size
	 * @throws TJSException
	 */
	public abstract int read( ByteBuffer buffer ) throws TJSException;

	/**
	 * returns actually read size
	 * @throws TJSException
	 */
	public abstract int read( byte[] buffer ) throws TJSException;

	/**
	 * ストリームからの読み込み
	 * @param b 読み込み先byte配列
	 * @param off 配列オフセット
	 * @param len 読み込みサイズ
	 * @return 実際に読み込んだ長さ。-1 の時ファイル終端
	 * @throws TJSException
	 */
	public abstract int read(byte[] b, int off, int len) throws TJSException;

	/**
	 * returns actually written size
	 */
	public abstract int write( final ByteBuffer buffer );

	/**
	 * returns actually written size
	 */
	public abstract int write( final byte[] buffer );

	public abstract void write(byte[] b, int off, int len);

	/**
	 * 1 バイトが出力ストリームに書き込まれます。
	 * 書き込まれるバイトは、引数 b の下位 8 ビットです。
	 * b の上位 24 ビットは無視されます。
	 * @param b
	 */
	public abstract void write(int b);

	/**
	 * close stream
	 */
	public abstract void close();

	/**
	 * the default behavior is raising a exception
	 * if error, raises exception
	 */
	public void setEndOfStorage() throws TJSException {
		throw new TJSException( Error.WriteError );
	}

	/**
	 * should re-implement for higher performance
	 * @throws TJSException
	 */
	public long getSize() throws TJSException {
		long orgpos = getPosition();
		long size = seek(0, SEEK_END );
		seek( orgpos, SEEK_SET );
		return size;
	}

	public long getPosition() throws TJSException {
		return seek( 0, SEEK_CUR );
	}

	public void setPosition( long pos ) throws TJSException {
		if( pos != seek( pos, SEEK_SET ) ) {
			throw new TJSException( Error.SeekError );
		}
	}

	public void readBuffer( ByteBuffer buffer ) throws TJSException {
		if( read( buffer ) != -1 ) {
			throw new TJSException( Error.ReadError );
		}
		buffer.flip();
	}
	public void writeBuffer( final ByteBuffer buffer ) throws TJSException {
		if( write( buffer ) != -1 ) {
			throw new TJSException( Error.WriteError );
		}
	}

	public abstract InputStream getInputStream();
	public abstract OutputStream getOutputStream();

	/**
	 * アーカイブ内のファイルかどうか判定する
	 * @return
	 */
	public boolean isArchive() {
		return false;
	}

	public abstract String getFilePath();

	public FileDescriptor getFileDescriptor() {
		return null;
	}
	public long getFileOffset() { return 0; }

	/*
	// reads little-endian integers
	public long readI64LE() throws TJSException {
		ByteBuffer buffer = ByteBuffer.allocate(8);
		buffer.order(ByteOrder.LITTLE_ENDIAN);
		readBuffer(buffer);
		return buffer.getLong();
	}
	public int readI32LE() throws TJSException {
		ByteBuffer buffer = ByteBuffer.allocate(4);
		buffer.order(ByteOrder.LITTLE_ENDIAN);
		readBuffer(buffer);
		return buffer.getInt();
	}
	public short readI16LE() throws TJSException {
		ByteBuffer buffer = ByteBuffer.allocate(4);
		buffer.order(ByteOrder.LITTLE_ENDIAN);
		readBuffer(buffer);
		return buffer.getShort();
	}
	*/
}

