/**
 * ============================================================================================
 * Easing Equations v2.0
 * September 1, 2003
 * (c) 2003 Robert Penner, all rights reserved.
 * This work is subject to the terms in http://www.robertpenner.com/easing_terms_of_use.html.
 * ============================================================================================
 */

/**
 * イージング関数ライブラリ
 * enchant.jsプロジェクトで
 * 「ActionScript で広く使われている Robert Penner による Easing Equations を JavaScript に移植した」
 * ライブラリを、さらにType script用に移植したもの。
 *
 * @see http://www.robertpenner.com/easing/
 * @see http://www.robertpenner.com/easing/penner_chapter7_tweening.pdf
 *
 * Easing function library, from "Easing Equations" by Robert Penner.
 */
class Easing {
	/**
	 * @param t
	 * @param b
	 * @param c
	 * @param d
	 * @return {Number}
	 */
	static LINEAR(t, b, c, d) {
		return c * t / d + b;
	}

	/**
	 * @param t
	 * @param b
	 * @param c
	 * @param d
	 * @return {Number}
	 */
	static SWING(t, b, c, d) {
		return c * (0.5 - Math.cos(((t / d) * Math.PI)) / 2) + b;
	}

	// quad
	/**
	 * @param t
	 * @param b
	 * @param c
	 * @param d
	 * @return {Number}
	 */
	static QUAD_EASEIN(t, b, c, d) {
		return c * (t /= d) * t + b;
	}

	/**
	 * @param t
	 * @param b
	 * @param c
	 * @param d
	 * @return {Number}
	 */
	static QUAD_EASEOUT(t, b, c, d) {
		return -c * (t /= d) * (t - 2) + b;
	}

	/**
	 * @param t
	 * @param b
	 * @param c
	 * @param d
	 * @return {Number}
	 */
	static QUAD_EASEINOUT(t, b, c, d) {
		if ((t /= d / 2) < 1) {
			return c / 2 * t * t + b;
		}
		return -c / 2 * ((--t) * (t - 2) - 1) + b;
	}

	// cubic
	/**
	 * @param t
	 * @param b
	 * @param c
	 * @param d
	 * @return {Number}
	 */
	static CUBIC_EASEIN(t, b, c, d) {
		return c * (t /= d) * t * t + b;
	}

	/**
	 * @param t
	 * @param b
	 * @param c
	 * @param d
	 * @return {Number}
	 */
	static CUBIC_EASEOUT(t, b, c, d) {
		return c * ((t = t / d - 1) * t * t + 1) + b;
	}

	/**
	 * @param t
	 * @param b
	 * @param c
	 * @param d
	 * @return {Number}
	 */
	static CUBIC_EASEINOUT(t, b, c, d) {
		if ((t /= d / 2) < 1) {
			return c / 2 * t * t * t + b;
		}
		return c / 2 * ((t -= 2) * t * t + 2) + b;
	}

	// quart
	/**
	 * @param t
	 * @param b
	 * @param c
	 * @param d
	 * @return {Number}
	 */
	static QUART_EASEIN(t, b, c, d) {
		return c * (t /= d) * t * t * t + b;
	}

	/**
	 * @param t
	 * @param b
	 * @param c
	 * @param d
	 * @return {Number}
	 */
	static QUART_EASEOUT(t, b, c, d) {
		return -c * ((t = t / d - 1) * t * t * t - 1) + b;
	}

	/**
	 * @param t
	 * @param b
	 * @param c
	 * @param d
	 * @return {Number}
	 */
	static QUART_EASEINOUT(t, b, c, d) {
		if ((t /= d / 2) < 1) {
			return c / 2 * t * t * t * t + b;
		}
		return -c / 2 * ((t -= 2) * t * t * t - 2) + b;
	}

	// quint
	/**
	 * @param t
	 * @param b
	 * @param c
	 * @param d
	 * @return {Number}
	 */
	static QUINT_EASEIN(t, b, c, d) {
		return c * (t /= d) * t * t * t * t + b;
	}

	/**
	 * @param t
	 * @param b
	 * @param c
	 * @param d
	 * @return {Number}
	 */
	static QUINT_EASEOUT(t, b, c, d) {
		return c * ((t = t / d - 1) * t * t * t * t + 1) + b;
	}

	/**
	 * @param t
	 * @param b
	 * @param c
	 * @param d
	 * @return {Number}
	 */
	static QUINT_EASEINOUT(t, b, c, d) {
		if ((t /= d / 2) < 1) {
			return c / 2 * t * t * t * t * t + b;
		}
		return c / 2 * ((t -= 2) * t * t * t * t + 2) + b;
	}

	//sin
	/**
	 * @param t
	 * @param b
	 * @param c
	 * @param d
	 * @return {Number}
	 */
	static SIN_EASEIN(t, b, c, d) {
		return -c * Math.cos(t / d * (Math.PI / 2)) + c + b;
	}

	/**
	 * @param t
	 * @param b
	 * @param c
	 * @param d
	 * @return {Number}
	 */
	static SIN_EASEOUT(t, b, c, d) {
		return c * Math.sin(t / d * (Math.PI / 2)) + b;
	}

	/**
	 * @param t
	 * @param b
	 * @param c
	 * @param d
	 * @return {Number}
	 */
	static SIN_EASEINOUT(t, b, c, d) {
		return -c / 2 * (Math.cos(Math.PI * t / d) - 1) + b;
	}

	// circ
	/**
	 * @param t
	 * @param b
	 * @param c
	 * @param d
	 * @return {Number}
	 */
	static CIRC_EASEIN(t, b, c, d) {
		return -c * (Math.sqrt(1 - (t /= d) * t) - 1) + b;
	}

	/**
	 * @param t
	 * @param b
	 * @param c
	 * @param d
	 * @return {Number}
	 */
	static CIRC_EASEOUT(t, b, c, d) {
		return c * Math.sqrt(1 - (t = t / d - 1) * t) + b;
	}

	/**
	 * @param t
	 * @param b
	 * @param c
	 * @param d
	 * @return {Number}
	 */
	static CIRC_EASEINOUT(t, b, c, d) {
		if ((t /= d / 2) < 1) {
			return -c / 2 * (Math.sqrt(1 - t * t) - 1) + b;
		}
		return c / 2 * (Math.sqrt(1 - (t -= 2) * t) + 1) + b;
	}

	// elastic
	/**
	 * @param t
	 * @param b
	 * @param c
	 * @param d
	 * @return {Number}
	 */
	static ELASTIC_EASEIN(t, b, c, d, a, p) {
		if (t === 0) {
			return b;
		}
		if ((t /= d) === 1) {
			return b + c;
		}

		if (!p) {
			p = d * 0.3;
		}

		var s;
		if (!a || a < Math.abs(c)) {
			a = c;
			s = p / 4;
		} else {
			s = p / (2 * Math.PI) * Math.asin(c / a);
		}
		return -(a * Math.pow(2, 10 * (t -= 1)) * Math.sin((t * d - s) * (2 * Math.PI) / p)) + b;
	}

	/**
	 * @param t
	 * @param b
	 * @param c
	 * @param d
	 * @return {Number}
	 */
	static ELASTIC_EASEOUT(t, b, c, d, a, p) {
		if (t === 0) {
			return b;
		}
		if ((t /= d) === 1) {
			return b + c;
		}
		if (!p) {
			p = d * 0.3;
		}
		var s;
		if (!a || a < Math.abs(c)) {
			a = c;
			s = p / 4;
		} else {
			s = p / (2 * Math.PI) * Math.asin(c / a);
		}
		return (a * Math.pow(2, -10 * t) * Math.sin((t * d - s) * (2 * Math.PI) / p) + c + b);
	}

	/**
	 * @param t
	 * @param b
	 * @param c
	 * @param d
	 * @return {Number}
	 */
	static ELASTIC_EASEINOUT(t, b, c, d, a, p) {
		if (t === 0) {
			return b;
		}
		if ((t /= d / 2) === 2) {
			return b + c;
		}
		if (!p) {
			p = d * (0.3 * 1.5);
		}
		var s;
		if (!a || a < Math.abs(c)) {
			a = c;
			s = p / 4;
		} else {
			s = p / (2 * Math.PI) * Math.asin(c / a);
		}
		if (t < 1) {
			return -0.5 * (a * Math.pow(2, 10 * (t -= 1)) * Math.sin((t * d - s) * (2 * Math.PI) / p)) + b;
		}
		return a * Math.pow(2, -10 * (t -= 1)) * Math.sin((t * d - s) * (2 * Math.PI) / p) * 0.5 + c + b;
	}

	// bounce
	/**
	 * @param t
	 * @param b
	 * @param c
	 * @param d
	 * @return {Number}
	 */
	static BOUNCE_EASEOUT(t, b, c, d) {
		if ((t /= d) < (1 / 2.75)) {
			return c * (7.5625 * t * t) + b;
		} else if (t < (2 / 2.75)) {
			return c * (7.5625 * (t -= (1.5 / 2.75)) * t + 0.75) + b;
		} else if (t < (2.5 / 2.75)) {
			return c * (7.5625 * (t -= (2.25 / 2.75)) * t + 0.9375) + b;
		} else {
			return c * (7.5625 * (t -= (2.625 / 2.75)) * t + 0.984375) + b;
		}
	}

	/**
	 * @param t
	 * @param b
	 * @param c
	 * @param d
	 * @return {Number}
	 */
	static BOUNCE_EASEIN(t, b, c, d) {
		return c - Easing.BOUNCE_EASEOUT(d - t, 0, c, d) + b;
	}

	/**
	 * @param t
	 * @param b
	 * @param c
	 * @param d
	 * @return {Number}
	 */
	static BOUNCE_EASEINOUT(t, b, c, d) {
		if (t < d / 2) {
			return Easing.BOUNCE_EASEIN(t * 2, 0, c, d) * 0.5 + b;
		} else {
			return Easing.BOUNCE_EASEOUT(t * 2 - d, 0, c, d) * 0.5 + c * 0.5 + b;
		}
	}

	// back
	/**
	 * @param t
	 * @param b
	 * @param c
	 * @param d
	 * @return {Number}
	 */
	static BACK_EASEIN(t, b, c, d, s) {
		if (s === undefined) {
			s = 1.70158;
		}
		return c * (t /= d) * t * ((s + 1) * t - s) + b;
	}

	/**
	 * @param t
	 * @param b
	 * @param c
	 * @param d
	 * @return {Number}
	 */
	static BACK_EASEOUT(t, b, c, d, s) {
		if (s === undefined) {
			s = 1.70158;
		}
		return c * ((t = t / d - 1) * t * ((s + 1) * t + s) + 1) + b;
	}

	/**
	 * @param t
	 * @param b
	 * @param c
	 * @param d
	 * @return {Number}
	 */
	static BACK_EASEINOUT(t, b, c, d, s) {
		if (s === undefined) {
			s = 1.70158;
		}
		if ((t /= d / 2) < 1) {
			return c / 2 * (t * t * (((s *= (1.525)) + 1) * t - s)) + b;
		}
		return c / 2 * ((t -= 2) * t * (((s *= (1.525)) + 1) * t + s) + 2) + b;
	}

	// expo
	/**
	 * @param t
	 * @param b
	 * @param c
	 * @param d
	 * @return {Number}
	 */
	static EXPO_EASEIN(t, b, c, d) {
		return (t === 0) ? b : c * Math.pow(2, 10 * (t / d - 1)) + b;
	}

	/**
	 * @param t
	 * @param b
	 * @param c
	 * @param d
	 * @return {Number}
	 */
	static EXPO_EASEOUT(t, b, c, d) {
		return (t === d) ? b + c : c * (-Math.pow(2, -10 * t / d) + 1) + b;
	}

	/**
	 * @param t
	 * @param b
	 * @param c
	 * @param d
	 * @return {Number}
	 */
	static EXPO_EASEINOUT(t, b, c, d) {
		if (t === 0) {
			return b;
		}
		if (t === d) {
			return b + c;
		}
		if ((t /= d / 2) < 1) {
			return c / 2 * Math.pow(2, 10 * (t - 1)) + b;
		}
		return c / 2 * (-Math.pow(2, -10 * --t) + 2) + b;
	}

	static RANDOM() {
		var ary = [];
		for (var i in Easing) {
			if (i != "RANDOM")
				ary.push(i);
		}
		return Easing[ary[Math.floor(Math.random() * ary.length)]];
	}
}