//======================================================================
//-----------------------------------------------------------------------
/**
 * @file		WXMMMP3File.cpp
 * @brief		MP3t@CNX
 *
 * @author		t.sirayanagi
 * @version		1.0
 *
 * @par			copyright
 * Copyright (C) 2009-2010 Takazumi Shirayanagi\n
 * The new BSD License is applied to this software.
 * see iris_LICENSE.txt
*/
//-----------------------------------------------------------------------
//======================================================================
#define _IRIS_WXMMMP3File_CPP_

//======================================================================
// include
#include "WXMMMP3File.h"
#include "WXMMSysError.h"
#include "fnd/utility/FndRIFF.h"
#include "fnd/memory/FndMemory.h"
#include "audio/format/AXAudioBuffer.h"

#include "../debug/WXDebugLeakCheckMacro.h"

//======================================================================
// link
#pragma comment ( lib, "msacm32.lib" )

namespace iris {
namespace wx
{

//======================================================================
// class

/**********************************************************************//**
 *
 * RXgN^
 *
*//***********************************************************************/
CMMMP3File::CMMMP3File(void)
: m_hAcmstream( nullptr )
, m_FrameSize(0)
, m_Offset(0)
, m_Size(0)
, m_DecodeRestSize(0)
{
}

/**********************************************************************//**
 *
 * fXgN^
 *
*//***********************************************************************/
CMMMP3File::~CMMMP3File(void)
{
	Close();
}

/**********************************************************************//**
 *
 * obt@f[^̉
 *
*//***********************************************************************/
void CMMMP3File::Release(void)
{
	m_File.Close();
	// ACMXg[
	if( m_hAcmstream != nullptr )
	{
		acmStreamClose( m_hAcmstream, 0 );
		m_hAcmstream = nullptr;
	}
}

/**********************************************************************//**
 *
 * J
 *
 ----------------------------------------------------------------------
 * @param [in]	filename	= t@C
 * @return	MMRESULTl
*//***********************************************************************/
bool CMMMP3File::OpenA(LPSTR	filename)
{
	if( !m_File.OpenA( filename, "rb" ) ) 
	{
		printf( "CMMMP3File OpenW failed. <%s>\n", filename);
		return false;
	}
	return _Open();
}
/// CMMMP3File::OpenA Q
bool CMMMP3File::OpenW(LPWSTR	filename)
{
	if( !m_File.OpenW( filename, L"rb" ) ) 
	{
		wprintf( L"CMMMP3File OpenW failed. <%s>\n", filename);
		return false;
	}
	return _Open();
}

/**********************************************************************//**
 *
 * t@C
 * 
 ----------------------------------------------------------------------
 * @return	MMRESULTl
*//***********************************************************************/
void CMMMP3File::Close(void)
{
	Release();
}

/**********************************************************************//**
 *
 * PCMf[^̓ǂݎ
 *
 ----------------------------------------------------------------------
 * @param [out]	lpBuffer	= o̓obt@
 * @param [in]	dstSize		= o̓obt@
 * @param [in]	srcSize		= ϊTCY
 * @return	WJobt@TCY
*//***********************************************************************/
s32 CMMMP3File::Decode(void* lpBuffer, u32 dstSize, u32 srcSize)
{
	if( lpBuffer == nullptr )	return 0;
	MMRESULT mmr = MMSYSERR_NOERROR;

	s32 ret = 0;
	ACMSTREAMHEADER		ash		= {0};
	LPBYTE				buf		= nullptr;
	DWORD read;
	if( dstSize < srcSize ) srcSize = dstSize;
	u32 seek = m_File.Tell();
	if( seek + srcSize > (u32)(m_Offset + m_FrameSize) )
	{
		srcSize = m_Offset + m_FrameSize - seek;
	}
	buf = reinterpret_cast<LPBYTE>(irisAlloc(srcSize, 0));
	m_File.Read(buf, srcSize, &read);
	if( seek + srcSize >= (u32)(m_Offset + m_FrameSize) )
	{
		m_File.Seek(m_Offset, nullptr, SEEK_SET);
	}

	memset( &ash, 0, sizeof(ACMSTREAMHEADER) );
	ash.cbStruct	= sizeof(ACMSTREAMHEADER);
	ash.pbSrc		= buf;
	ash.cbSrcLength = srcSize;
	ash.pbDst		= reinterpret_cast<LPBYTE>(lpBuffer);
	ash.cbDstLength = dstSize;
	// o͗p\̂̏
	mmr = acmStreamPrepareHeader(m_hAcmstream, &ash, 0);
	if( mmr != MMSYSERR_NOERROR ) goto exit;
	// ϊ
	mmr = acmStreamConvert(m_hAcmstream, &ash, 0);
	if( mmr != MMSYSERR_NOERROR ) goto exit;
	// o͗p\̂̔j
	mmr = acmStreamUnprepareHeader(m_hAcmstream, &ash, 0);
	if( mmr != MMSYSERR_NOERROR ) goto exit;

	irisFree(buf);
	ret += ash.cbDstLengthUsed;
	return ret;
exit:
	irisFree(buf);
	WX_MM_ERROR(mmr);
	return -(s32)mmr;
}

/**********************************************************************//**
 *
 * PCMf[^̓ǂݎ
 *
 ----------------------------------------------------------------------
 * @param [out]	lpBuffer	= o̓obt@
 * @param [in]	length		= o̓obt@
 * @return	WJobt@TCY
*//***********************************************************************/
s32 CMMMP3File::ReadPCM(void* lpBuffer, u32 length)
{
	if( lpBuffer == nullptr )	return 0;
	u32 total = 0;
	while(1)
	{
		s32 srcSize = GetStreamSize(length-total, ACM_STREAMSIZEF_DESTINATION);
		s32 size = Decode(static_cast<char*>(lpBuffer) + total, length-total, srcSize);
		if( size <= 0 )
		{
			break;
		}
		total += size;
		if( total >= length )
		{
			break;
		}
	}
	return total;
}

/**********************************************************************//**
 *
 * WAVEFORMATEX̎擾
 *
 ----------------------------------------------------------------------
 * @param [out]	pwfex	= WAVEFORMATEX
 * @return	
*//***********************************************************************/
bool CMMMP3File::GetWaveFormatEx(LPWAVEFORMATEX pwfex)
{
	*pwfex = m_WaveFmt;
	return true;
}

/**********************************************************************//**
 *
 * tell
 *
 ----------------------------------------------------------------------
 * @return	V[Nʒu(TvP)
*//***********************************************************************/
s32 CMMMP3File::Tell(void)
{
	u32 pos = m_File.Tell();
	if( pos < (u32)m_Offset ) return -1;
	if( pos >= m_Offset + m_FrameSize ) return -1;
	u32 size = pos - m_Offset;
	return GetStreamSize(size, ACM_STREAMSIZEF_SOURCE) / (m_WaveFmt.wBitsPerSample * m_WaveFmt.nChannels);
}

/**********************************************************************//**
 *
 * tell time
 *
 ----------------------------------------------------------------------
 * @return	V[Nʒu(bP)
*//***********************************************************************/
f64 CMMMP3File::TellTime(void)
{
	s32 samples = Tell();
	if( samples == -1 ) return 0.0f;
	return (f64)samples / m_WaveFmt.nSamplesPerSec;
}

/**********************************************************************//**
 *
 * seek
 *
 ----------------------------------------------------------------------
 * @param [in]	pos	= TvP
 * @return	
*//***********************************************************************/
bool CMMMP3File::Seek(s32 samples)
{
	s32 size = samples * m_WaveFmt.wBitsPerSample * m_WaveFmt.nChannels;
	m_File.Seek(GetStreamSize(size, ACM_STREAMSIZEF_DESTINATION) + m_Offset, nullptr, SEEK_SET);
	return true;
}

/**********************************************************************//**
 *
 * seek time
 *
 ----------------------------------------------------------------------
 * @param [in]	time	= bP
 * @return	
*//***********************************************************************/
bool CMMMP3File::SeekTime(f64 time)
{
	s32 samples = static_cast<s32>(time * m_WaveFmt.nSamplesPerSec);
	Seek(samples);
	return true;
}

/**********************************************************************//**
 *
 * ʃI[v
 * 
 ----------------------------------------------------------------------
 * @return	MMRESULTl
*//***********************************************************************/
bool CMMMP3File::_Open(void)
{
	// 4 byte read
	u4cc head;
	DWORD read;
	u32 file_end = m_File.Seek(0, nullptr, SEEK_END);
	m_File.Seek(0, nullptr, SEEK_SET);
	//u32 file_end = m_File.GetSize();
	c8* ph = reinterpret_cast<c8*>(&head);
	m_File.Read(&head, sizeof(head), &read);
	
	if( head == fnd::RIFF_ID_FILE_CHUNK )
	{
		// TODO : not supported
		m_File.Seek(4, nullptr, SEEK_CUR);
		m_File.Read(&head, sizeof(head), &read);
		if( head != fnd::RIFF_FORM_WAVE ) return false;
		
		// data`N܂Œʉ
		fnd::RIFF_CHUNK_HEADER chunk;
		while(1)
		{
			m_File.Read(&chunk, sizeof(chunk), &read);
			if( chunk.uID == fnd::RIFF_ID_DATA_CHUNK )
			{
				file_end = m_File.Tell() + chunk.Size;
				m_File.Read(&head, sizeof(head), &read);
				break;
			}
			else
			{
				m_File.Seek(chunk.Size, nullptr, SEEK_CUR);
			}
		}
	}
	
	// ID3
	if( ph[0] == 'I' && ph[1] == 'D' && ph[2] == '3' )
	{
		// ǂݎ̂
		u16 ver;
		u8 flag;
		u8 szb[4];
		u32 size = 0;
		m_File.Read(&ver , sizeof(ver) , &read);
		m_File.Read(&flag, sizeof(flag), &read);
		m_File.Read(&szb , sizeof(szb) , &read);
		
		if( flag & 0x10 )
		{
			size += 10;
		}
		
		size += (szb[0] << 21) + (szb[1] << 14) + (szb[2] << 7) + (szb[3] & 0x7F);
		m_File.Seek(size, nullptr, SEEK_SET);
		m_File.Read(&head, sizeof(head), &read);
	}
	
	// TAG
	{
		u32 temp = head;
		u32 seek = m_File.Tell();
		m_File.Seek(file_end-128, nullptr, SEEK_SET);
		m_File.Read(&head, sizeof(head), &read);
		if( ph[0] == 'T' && ph[1] == 'A' && ph[2] == 'G' )
		{
			file_end = m_File.Seek(128, nullptr, SEEK_END);
		}
		head = temp;
		m_File.Seek(seek, nullptr, SEEK_SET);
	}

	// MP3
	u8	MPVerTable[4]		= {3, 0, 2, 1};// MEPG̃o[We[u
	u8	LayerTable[4]		= {0, 3, 2, 1};// Layere[u
	if( (ph[0] & 0xFF) != 0xFF 
		|| (ph[1] & 0xD0) != 0xD0)
		return false;
	
	u8 version	= MPVerTable[(ph[1] >> 3) & 0x3];
	u8 layer	= LayerTable[(ph[1] >> 1) & 0x3];
	u32	nBitRateTable[][16] =	// rbg[ge[u
	{
		{0, 32, 64, 96, 128, 160, 192, 224, 256, 288, 320, 352, 384, 416, 448, 0},
		{0, 32, 48, 56, 64,	 80,  96,  112, 128, 160, 192, 224, 256, 320, 384, 0},
		{0, 32, 40, 48, 56,	 64,  80,  96,	112, 128, 160, 192, 224, 256, 320, 0},// MPEG1
		{0, 32, 48, 56, 64,	 80,  96,  112, 128, 144, 160, 176, 192, 224, 256, 0},
		{0, 8,	16, 24, 32,	 40,  48,  56,	64,	 80,  96,  112, 128, 144, 160, 0},// MPEG2
		{0, 8,	16, 24, 32,	 40,  48,  56,	64,	 80,  96,  112, 128, 144, 160, 0},// MPEG2
		{0, 8,	16, 24, 32,	 40,  48,  56,	64,	 80,  96,  112, 128, 144, 160, 0},// MPEG2
	};
	u32	nSampleTable[][4]	=// TvO[ge[u
	{
		{44100, 48000, 32000, 0},// MPEG1
		{22050, 24000, 16000, 0},// MPEG2
		{11025, 12000, 8000,  0},
		{0,		0,	   0,	  0},// \
	};
	
	if( version < 1 || layer < 1 ) goto exit;
	
	u8 index = (ph[2] >> 4) & 0xF;
	u32 bitrate = 0;
	switch( version )
	{
	case 1:
		bitrate = nBitRateTable[layer - 1][index];
		break;
	case 2:
		bitrate = nBitRateTable[layer - 1 + 3][index];
		break;
	default:
		bitrate = nBitRateTable[6][index];
		break;
	}
	// ςї\ ΏۊO
	if( bitrate == 0 ) goto exit;
	index = (ph[2] >> 2) & 0x3;
	u32 samplerate	= nSampleTable[version - 1][index];
	
	if( samplerate == 0 ) goto exit;
	
	u32 padding		= (ph[2] >> 1) & 0x1;
	u16 channels	= ((ph[3] >> 6) & 0x3) == 0x3 ? 1 : 2;

	u16 framesize = 0;
	switch(version)
	{
	case 1:
		if( layer ) framesize = (u16)((12 * 1000 * bitrate / samplerate + padding) * 4);
		else		framesize = (u16)((144 * 1000 * bitrate / samplerate + padding));
		break;
	case 2:
		if( layer ) framesize = (u16)((12 * 1000 * bitrate / samplerate + padding) * 4);
		else		framesize = (u16)((72 * 1000 * bitrate / samplerate + padding));
		break;
	default:
		if( layer ) framesize = (u16)((72 * 1000 * bitrate / samplerate + padding));
		else		goto exit;
		break;
	}
	
	m_Mp3Fmt.wfx.wFormatTag			= WAVE_FORMAT_MPEGLAYER3;
	m_Mp3Fmt.wfx.nChannels			= channels;
	m_Mp3Fmt.wfx.nSamplesPerSec		= samplerate;
	m_Mp3Fmt.wfx.nAvgBytesPerSec	= ( bitrate * 1000 ) / 8;
	m_Mp3Fmt.wfx.nBlockAlign		= 1;
	m_Mp3Fmt.wfx.wBitsPerSample		= 0;
	m_Mp3Fmt.wfx.cbSize				= MPEGLAYER3_WFX_EXTRA_BYTES;

	m_Mp3Fmt.wID				= MPEGLAYER3_ID_MPEG;
	m_Mp3Fmt.fdwFlags			= ( padding ) ? MPEGLAYER3_FLAG_PADDING_ON : MPEGLAYER3_FLAG_PADDING_OFF;
	m_Mp3Fmt.nBlockSize			= framesize;
	m_Mp3Fmt.nFramesPerBlock	= 1;
	m_Mp3Fmt.nCodecDelay		= 0x571;
	
	m_Offset = m_File.Tell();
	m_FrameSize = file_end - m_Offset;
	
	ReadWaveFormatEx(&m_WaveFmt);
	
	// Xg[̏
	if( InitStream() != MMSYSERR_NOERROR )
		goto exit;

	m_Size = GetStreamSize(m_FrameSize, ACM_STREAMSIZEF_SOURCE);

	return true;

exit:
	Close();
	return false;
}

/**********************************************************************//**
 *
 * Xg[̏
 * 
 ----------------------------------------------------------------------
 * @return	MMRESULTl
*//***********************************************************************/
MMRESULT CMMMP3File::InitStream(void)
{
	MMRESULT				mmr = MMSYSERR_NOERROR;	
	MPEGLAYER3WAVEFORMAT	mwf = {0};

	// ACMXg[J
	mmr = acmStreamOpen(&m_hAcmstream
		, nullptr
		, &m_Mp3Fmt.wfx
		, &m_WaveFmt
		, nullptr
		, 0
		, 0
		, ACM_STREAMOPENF_NONREALTIME );
	if( mmr != MMSYSERR_NOERROR )
	{
		WX_MM_ERROR(mmr);
		return mmr;
	}
	return mmr;
}

/**********************************************************************//**
 *
 * WJ̃tH[}bg擾
 * 
 *----------------------------------------------------------------------
 * @param [out]	pWfx	WAVEFORMATEX\̂̃AhX
 * @return	MMRESULTl
*//***********************************************************************/
MMRESULT CMMMP3File::ReadWaveFormatEx(LPWAVEFORMATEX pWfx)
{
	MMRESULT	mmr = MMSYSERR_NOERROR;	// G[擾p
	// ϊݒ
	m_WaveFmt.cbSize		= 0;
	m_WaveFmt.wFormatTag	= WAVE_FORMAT_PCM;

	// o͌`̐ݒ
	mmr = acmFormatSuggest(
		nullptr,
		&m_Mp3Fmt.wfx,
		pWfx,
		sizeof( WAVEFORMATEX ),
		ACM_FORMATSUGGESTF_WFORMATTAG );

	// G[
	if( mmr != MMSYSERR_NOERROR )
	{
		WX_MM_ERROR(mmr);
	}
	return mmr;
}

/**********************************************************************//**
 *
 * WJEÕTCY̎擾
 * 
 ----------------------------------------------------------------------
 * @param [in]	dwBufferSize	= ϊTCY
 * @param [in]	fdwSize			= ACM_STREAMSIZEF_***
 * @return	WJEÕTCY
*//***********************************************************************/
DWORD CMMMP3File::GetStreamSize(DWORD dwBufferSize, DWORD fdwSize)
{
	MMRESULT	mmr		= MMSYSERR_NOERROR;
	DWORD		dwSize	= 0;	// WJ̃TCY

	if( m_hAcmstream == nullptr )
		return 0xFFFFFFFF;

	// o̓TCY̎擾
	mmr = acmStreamSize(m_hAcmstream
		, dwBufferSize
		, &dwSize
		, fdwSize);
	//G[
	if( mmr != MMSYSERR_NOERROR )
	{
		WX_MM_ERROR(mmr);
		return 0xFFFFFFFF;
	}
	return dwSize;
}

}	// end of namespace wx
}	// end of namespace iris

#if (defined(_IRIS_UNITTEST) || defined(_IRIS_MULTI_UNITTEST))
#include "../debug/unittest/WXDebugUnitTest.h"
#include "WXMMWaveOut.h"
#include "iris.h"

//======================================================================
// test
IRIS_UNITTEST(CWXMMMP3FileFileUnitTest,WXMMMP3FileFileUnitTest)
{
	MMRESULT	mmr = MMSYSERR_NOERROR;//G[擾p
	CMMMP3File	mp3;
	ax::CAXBuffer buffer(&mp3); 
	CMMWaveOut	waveOut;

	if( !mp3.Open( TEXT("../../data/snd/sample.mp3") ) )
	{
		puts("mp3 file open failed.");
		return;
	}
	if( !buffer.Read() )
	{
		return;
	}

	// svȃobt@
	mp3.Release();

	// WAVEt@CI[v
	mmr = waveOut.Open(	mp3.GetWaveFormatEx() );
	// G[
	if( mmr != MMSYSERR_NOERROR )
	{
		return;
	}

	// Đ
	waveOut.PrepareHeader( (LPSTR)buffer.GetAddr(), buffer.GetSize(), 0, 0 );
	waveOut.Write();

	WAVEFORMATEX *pWfx = mp3.GetWaveFormatEx();

	{
		printf( "sample.mp3̏\n" );

		printf( "FormatTag[%d]\n"		, pWfx->wFormatTag );
		printf( "Channels[%d]\n"		, pWfx->nChannels );
		printf( "SamplesPerSec[%d]\n"	, pWfx->nSamplesPerSec );
		printf( "AvgBytesPerSec[%d]\n"	, pWfx->nAvgBytesPerSec );
		printf( "BlockAlign[%d]\n"		, pWfx->nBlockAlign );
		printf( "BitsPerSample[%d]\n"	, pWfx->wBitsPerSample );
		printf( "Size[%d]\n"			, pWfx->cbSize );

		printf( "\n" );
		printf( "--------------------" );
		printf( "[ESCAPE] Close\n" );
		printf( "[Z]      Play\n" );
		printf( "[X]      Pause\n" );
		printf( "[C]      Loop Change\n" );
	}

	bool	bLoop		= true;		// [vtO
	bool	bPause		= false;	// |[YtO
	bool	bWaveLoop	= true;		// [vtO
	MMTIME	mmt			= {0};		// MMTIME\
	DWORD	dwSecond	= 0;

	mmt.wType = TIME_BYTES;

	while( bLoop )
	{
		waveOut.GetPosition(&mmt);
		// ĐԂ߂
		dwSecond = mmt.u.cb / pWfx->nAvgBytesPerSec;

		if( bWaveLoop == true )
		{
			if( mmt.u.cb >= mp3.GetSize() )
			{
				waveOut.Reset();
				waveOut.UnprepareHeader();
				waveOut.PrepareHeader( (LPSTR)buffer.GetAddr(), buffer.GetSize(), 0, 0 );
				waveOut.Write();
			}
		}
		// I
		if( GetAsyncKeyState( VK_ESCAPE ) & 0x8000 )
			bLoop = false;
		// Đ
		if( GetAsyncKeyState( 'Z' ) & 0x8000 )
		{
			waveOut.Reset();
			waveOut.UnprepareHeader();
			waveOut.PrepareHeader( (LPSTR)buffer.GetAddr(), buffer.GetSize(), 0, 0 );
			waveOut.Write();
			Sleep( 100 );
		}

		// |[Y
		if( GetAsyncKeyState( 'X' ) & 0x8000 )
		{
			if( bPause == false )
			{
				waveOut.Pause();
				bPause = true;
				Sleep( 100 );
			}
			else
			{
				waveOut.Restart();
				bPause = false;
				Sleep( 100 );
			}
		}

		// [v؂ւ
		if( GetAsyncKeyState( 'C' ) & 0x8000 )
		{
			if( bWaveLoop == false )
			{
				bWaveLoop = true;
				Sleep( 100 );
			}
			else
			{
				bWaveLoop = false;
				Sleep( 100 );
			}
		}
	}

	// WAVEt@C
	waveOut.Reset();
	waveOut.UnprepareHeader();
	waveOut.Close();
	mp3.Close();
}

#endif	// #if (defined(_IRIS_UNITTEST) || defined(_IRIS_MULTI_UNITTEST))
