/*
 * Galatea Dialog Manager:
 * (c)2003 Takuya NISHIMOTO (nishimoto [atmark] m.ieice.org)
 * Based on Phoenix By Takuya NISHIMOTO and Mitsuhiro KIZU
 *
 * $Id: OutItemQueueManager.java,v 1.4 2009/01/28 13:51:17 nishimoto Exp $
 */
package galatea.io;

import galatea.logger.Logger;
import galatea.outitem.OutItem;

import java.util.ArrayList;

public class OutItemQueueManager            
{
	private Logger logger_ = new Logger(this.getClass());
	private DeviceManager deviceManager_;
	private int outputState_ = 0;
	private int outputCount_ = 0;
	private int prevOutputCount_ = 0;
	private OutItem currOutItem_ = null;
	private ArrayList<OutItem> queue_; 
	private boolean currOutItemIsValid_ = false;
	private boolean currOutQueueIsValid_ = false;
	
	private final int STATE_NOT_STARTED = 0;
	private final int STATE_WAIT_FOR_OUTPUT_BUSY = 1;
	private final int STATE_WAIT_FOR_OUTPUT_READY = 2;
	private final int STATE_WAIT_FOR_OUTPUT_STOPPED = 3;
	
	public OutItemQueueManager(DeviceManager d) {
		deviceManager_ = d;
		queue_ = new ArrayList<OutItem>();
	}
	
	public synchronized void setCurrOutQueueIsValid(boolean b) {
		currOutQueueIsValid_ = b;
		logger_.print("currOutQueueIsValid: "+b);
	}
	
	public synchronized boolean getCurrOutQueueIsValid() {
		return currOutQueueIsValid_;
	}
	
	public synchronized void setCurrOutItemIsValid(boolean b) {
		currOutItemIsValid_ = b;
	}
	
	public synchronized void processDeviceEvent(DeviceEvent evt) {
		logger_.print("processDeviceEvent " + evt.toString());
		if (evt.isOutputBusyEvent()) {
			incrementOutputCount();	    
		} else if (evt.isOutputReadyEvent()) {
			decrementOutputCount();	    
		}
	}

	private void incrementOutputCount() {
		outputCount_ ++;
		logger_.print("(increment) outputCount=" + outputCount_);
	}
	
	private void decrementOutputCount() {
		outputCount_ --;
		logger_.print("(decrement) outputCount=" + outputCount_);
	}
	
	public synchronized void discard() {
		logger_.print("queue discard");
		queue_.clear();
	}
	
	public synchronized void enqueue(OutItem o) {
		queue_.add(o);
		logger_.print("queue added (" + o.getClass().getName() 
				+ ") total:" + queue_.size() );
	}
	
	private OutItem dequeue() {
		OutItem ret;
		if (queue_.isEmpty()) {
			return null;
		} else {
			ret = (OutItem)queue_.remove(0);
			logger_.print("dequeue() removed (" + ret.getClass().getName() 
					+ ") total:" + queue_.size() );
			return ret;
		}
	}
	
	
	public synchronized void removeDelayedOutputItems() {
		ArrayList<OutItem> newvec = new ArrayList<OutItem>();
		for (int i = 0; i < queue_.size(); i++ ) {
			OutItem o = queue_.get(i);
			if (o.isInstant() || o.isImportant() ) {
				newvec.add(o);
			}
		}
		queue_ = newvec;
		logger_.print("removeDelayedOutputItems() total:" + queue_.size() );
	}
	

	// returns false if dequeue item is null
	public synchronized boolean doOutputIteration() {
		OutItem item;
		switch (getOutputState()) {
		case STATE_NOT_STARTED:
			// dbg.print("[0] starting output");
			if (queue_.isEmpty()) {
				return false;
			}
			item = dequeue();
			if ( item == null ) {
				logger_.print("doOutputIteration() dequeue item is null");
				return false;
			}
			if ( item.isInstant() ) {
				logger_.print("output instant: " + item.toString());
				boolean success = deviceManager_.doOutput(item);
				if (!success) {
					return false;
				}
			} else {
				logger_.print("output non-instant: " + item.toString());
				boolean success = deviceManager_.doOutput(item);
				if (!success) {
					return false;
				}
				setCurrOutItemIsValid(true);
				setOutputState(STATE_WAIT_FOR_OUTPUT_BUSY);
				setCurrOutItem(item);
				setPrevOutputCount(outputCount_);
			}
			break;

		case STATE_WAIT_FOR_OUTPUT_BUSY:
			// dbg.print("[1] waiting for output busy");
			if ( ! currOutItemIsValid_ ) {
				deviceManager_.doStop(getCurrOutItem());
				setOutputState(STATE_WAIT_FOR_OUTPUT_STOPPED);
				break;
			}
			if (outputCount_ == getPrevOutputCount() + 1) {
				setOutputState(STATE_WAIT_FOR_OUTPUT_READY);
				break;
			}
			break;
		
		case STATE_WAIT_FOR_OUTPUT_READY:
			// dbg.print("[2] waiting for output ready");
			if ( ! currOutItemIsValid_ ) {
				deviceManager_.doStop(getCurrOutItem());
				setOutputState(STATE_WAIT_FOR_OUTPUT_STOPPED);
				break;
			}
			if (outputCount_ == getPrevOutputCount()) {
				setOutputState(STATE_NOT_STARTED);
				break;
			}
			break;
		
		case STATE_WAIT_FOR_OUTPUT_STOPPED:
			// dbg.print("[3] waiting for output stop done");
			if (outputCount_ == getPrevOutputCount()) {
				setOutputState(STATE_NOT_STARTED);
				break;
			}
			break;
		
		default:
			break;
		}
		
		return true;
	}
	
	private void setOutputState(int outputState) {
		this.outputState_ = outputState;
		switch (this.outputState_) {
		case STATE_NOT_STARTED:
			logger_.print("[outputState 0] starting output");
			break;
		case STATE_WAIT_FOR_OUTPUT_BUSY:
			logger_.print("[outputState 1] waiting for output busy");
			break;
		case STATE_WAIT_FOR_OUTPUT_READY:
			logger_.print("[outputState 2] waiting for output ready");
			break;
		case STATE_WAIT_FOR_OUTPUT_STOPPED:
			logger_.print("[outputState 3] waiting for output stop done");
			break;
		default:
			break;
		}
	}
	
	public synchronized OutItem getCurrOutItem() {
		return currOutItem_;
	}

	private void setCurrOutItem(OutItem currOutItem) {
		this.currOutItem_ = currOutItem;
	}
	
	private synchronized int getOutputState() {
		return outputState_;
	}
	
	private int getPrevOutputCount() {
		return prevOutputCount_;
	}
	
	private void setPrevOutputCount(int prevOutputCount) {
		this.prevOutputCount_ = prevOutputCount;
	}
	
	public synchronized void stopCurrentOutput() {
		if (getCurrOutItem() != null) {
			deviceManager_.doStop(getCurrOutItem());
			setOutputState(STATE_WAIT_FOR_OUTPUT_STOPPED);
		}
	}
	
	public synchronized String toString() {
		String result = "";
		result += "count: " + queue_.size() + "\n";
		for (int i=0; i<queue_.size(); i++) {
			result += "[item " + i + "] ";
			result += queue_.get(i).toString() + "\n";
		}
		return result;
	}

}
