﻿using System;
using System.IO;
using System.Threading.Tasks;
using Windows.Storage;
using System.Xml.Serialization;
using System.Xml;
using System.Xml.Schema;

namespace FooEditor
{
    public class FileModel
    {
        public StorageFile File
        {
            get;
            private set;
        }

        public FileModel(StorageFile file)
        {
            this.File = file;
        }

        public string Path
        {
            get
            {
                return this.File.Path;
            }
        }

        public string Name
        {
            get
            {
                return this.File.Name;
            }
        }

        public static string TrimFullPath(string filepath)
        {
            if (filepath == null || filepath == "")
                return string.Empty;
            string DirectoryPart = System.IO.Path.GetDirectoryName(filepath);
            string FilenamePart = System.IO.Path.GetFileName(filepath);
            string[] slice = DirectoryPart.Split('\\');
            if (slice.Length > 3)
            {
                DirectoryPart = slice[0] + "\\..\\" + slice[slice.Length - 1];
                return DirectoryPart + "\\" + FilenamePart;
            }
            else
                return filepath;
        }

        public static async Task<FileModel> GetFileModelAsync(string file_name,bool overwrite = false)
        {
            var file = await ApplicationData.Current.LocalFolder.GetFileAsync(file_name);
            return new FileModel(file);
        }

        public static async Task<FileModel> CreateFileModel(string file_name,bool overwrite = false)
        {
            CreationCollisionOption coll_opt = overwrite ? CreationCollisionOption.ReplaceExisting : CreationCollisionOption.FailIfExists;
            return new FileModel(await ApplicationData.Current.LocalFolder.CreateFileAsync(file_name,coll_opt));
        }

        public static async Task<FileModel> CreateFileModelFromAppSetting(string file_name)
        {
            var file = await ApplicationData.Current.LocalFolder.CreateFileAsync("DocumentCollection.xml", CreationCollisionOption.ReplaceExisting);
            return new FileModel(file);
        }

        public static async Task<FileModel> GetFileModelAsync(Uri uri)
        {
            return new FileModel(await Windows.Storage.StorageFile.GetFileFromApplicationUriAsync(uri));
        }

        public static async Task<FileModel> GetFileModelFromPath(string path)
        {
            return new FileModel(await StorageFile.GetFileFromPathAsync(path));
        }

        public static async Task<FileModel> TryGetFileModelFromAppSetting(string file_name)
        {
            var file = await ApplicationData.Current.LocalFolder.TryGetItemAsync(file_name) as StorageFile;
            return file == null ? null : new FileModel(file);
        }

        public async Task<Stream> GetReadStreamAsync()
        {
            return await this.File.OpenStreamForReadAsync();
        }

        public async Task<Stream> GetWriteStreamAsync()
        {
            return await this.File.OpenStreamForWriteAsync();
        }

        public async Task DeleteAsync()
        {
            await this.File.DeleteAsync();
        }

        public bool IsNeedUserActionToSave()
        {
            //ドラッグアンドドロップされたファイルはリードオンリーだが、ファイルピッカーで選ばせれば次からは保存できる
            return (this.File.Attributes & Windows.Storage.FileAttributes.ReadOnly) == Windows.Storage.FileAttributes.ReadOnly;
        }

    }
}
