﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using FooEditEngine;
using SharpDX;
using D2D = SharpDX.Direct2D1;
using DW = SharpDX.DirectWrite;
using DXGI = SharpDX.DXGI;
using System.Runtime.InteropServices;

namespace FooEditEngine
{
    delegate void PreDrawOneLineHandler(MyTextLayout layout);

    class D2DRenderCommon : IDisposable
    {
        CacheManager<string, MyTextLayout> LayoutCache = new CacheManager<string, MyTextLayout>();
        ResourceManager<TokenType, D2D.SolidColorBrush> SyntaxResources = new ResourceManager<TokenType, D2D.SolidColorBrush>();
        ResourceManager<HilightType, DrawingEffect> MarkerEffects = new ResourceManager<HilightType, DrawingEffect>();
        InlineManager HiddenChars;
        D2D.SolidColorBrush InsertCaretBrush, OverwriteCaretBrush, LineMarkerBrush;
        DW.Factory DWFactory;
        D2D.Factory D2DFactory;
        DW.TextFormat format;
        D2D.Bitmap bitmap;
        D2D.RenderTarget render;
        CustomTextRenderer textRender;
        float tabWidth = 64.0f;
        int tabLength = 8;
        bool hasCache;
        Size renderSize = new Size();

#if WINFORM
        public D2DRenderCommon(FooEditEngine.Windows.FooTextBox textbox, double width, double height)
#endif
#if WPF
        public D2DRenderCommon(FooEditEngine.WPF.FooTextBox textbox, double width, double height)
#endif
        {
            textbox.Document.Markers.Updated += new EventHandler(Markers_Updated);

            this.DWFactory = new DW.Factory(DW.FactoryType.Shared);
            this.D2DFactory = new D2D.Factory(D2D.FactoryType.MultiThreaded);
        }

        public void InitTextFormat(string fontName, float fontSize)
        {
            if(this.format != null)
                this.format.Dispose();
            
            this.format = new DW.TextFormat(this.DWFactory, fontName, fontSize / 72.0f * 96.0f);
            this.format.WordWrapping = DW.WordWrapping.NoWrap;

            if (this.HiddenChars == null)
            {
                this.HiddenChars = new InlineManager(this.DWFactory, this.format);
                this.HiddenChars.TabLength = this.tabLength;
            }
            else
                this.HiddenChars.Format = this.format;
                        
            this.LayoutCache.Clear();
        }

        public bool ShowFullSpace
        {
            get
            {
                if (this.HiddenChars == null)
                    return false;
                else
                    return this.HiddenChars.ContainsSymbol('　');
            }
            set
            {
                if (this.HiddenChars == null)
                    throw new InvalidOperationException();
                if (value)
                    this.HiddenChars.AddSymbol('　', '□');
                else
                    this.HiddenChars.RemoveSymbol('　');
            }
        }

        public bool ShowHalfSpace
        {
            get
            {
                if (this.HiddenChars == null)
                    return false;
                else
                    return this.HiddenChars.ContainsSymbol(' ');
            }
            set
            {
                if (this.HiddenChars == null)
                    throw new InvalidOperationException();
                if (value)
                    this.HiddenChars.AddSymbol(' ', 'ﾛ');
                else
                    this.HiddenChars.RemoveSymbol(' ');
            }
        }

        public bool ShowTab
        {
            get
            {
                if (this.HiddenChars == null)
                    return false;
                else
                    return this.HiddenChars.ContainsSymbol('\t');
            }
            set
            {
                if (this.HiddenChars == null)
                    throw new InvalidOperationException();
                if (value)
                    this.HiddenChars.AddSymbol('\t', '>');
                else
                    this.HiddenChars.RemoveSymbol('\t');
            }
        }

        public Color4 Foreground
        {
            get
            {
                return this.MarkerEffects[HilightType.Sold].ForeBrush.Color;
            }
            set
            {
                if (this.render == null)
                    return;
                D2D.StrokeStyleProperties prop = new D2D.StrokeStyleProperties();
                prop.DashCap = D2D.CapStyle.Flat;
                prop.DashOffset = 0;
                prop.DashStyle = D2D.DashStyle.Solid;
                prop.EndCap = D2D.CapStyle.Flat;
                prop.LineJoin = D2D.LineJoin.Miter;
                prop.MiterLimit = 0;
                prop.StartCap = D2D.CapStyle.Flat;
                this.MarkerEffects[HilightType.Sold] = new DrawingEffect(
                    new D2D.StrokeStyle(this.D2DFactory, prop),
                    new D2D.SolidColorBrush(this.render, value));

                prop.DashStyle = D2D.DashStyle.Dash;
                this.MarkerEffects[HilightType.Dash] = new DrawingEffect(
                    new D2D.StrokeStyle(this.D2DFactory, prop),
                    new D2D.SolidColorBrush(this.render, value));

                prop.DashStyle = D2D.DashStyle.DashDot;
                this.MarkerEffects[HilightType.DashDot] = new DrawingEffect(
                    new D2D.StrokeStyle(this.D2DFactory, prop),
                    new D2D.SolidColorBrush(this.render, value));

                prop.DashStyle = D2D.DashStyle.DashDotDot;
                this.MarkerEffects[HilightType.DashDotDot] = new DrawingEffect(
                    new D2D.StrokeStyle(this.D2DFactory, prop),
                    new D2D.SolidColorBrush(this.render, value));

                prop.DashStyle = D2D.DashStyle.Dot;
                this.MarkerEffects[HilightType.Dot] = new DrawingEffect(
                    new D2D.StrokeStyle(this.D2DFactory, prop),
                    new D2D.SolidColorBrush(this.render, value));

                this.LayoutCache.Clear();
            }
        }

        public Color4 Background
        {
            get;
            set;
        }

        public Color4 InsertCaret
        {
            get
            {
                return this.InsertCaretBrush.Color;
            }
            set
            {
                if (this.InsertCaretBrush != null)
                    this.InsertCaretBrush.Dispose();
                if (this.render == null)
                    return;
                this.InsertCaretBrush = new D2D.SolidColorBrush(this.render, value);
            }
        }

        public Color4 OverwriteCaret
        {
            get
            {
                return this.OverwriteCaretBrush.Color;
            }
            set
            {
                if (this.OverwriteCaretBrush != null)
                    this.OverwriteCaretBrush.Dispose();
                if (this.render == null)
                    return;
                this.OverwriteCaretBrush = new D2D.SolidColorBrush(this.render, value);
            }
        }

        public Color4 LineMarker
        {
            get
            {
                return this.LineMarkerBrush.Color;
            }
            set
            {
                if (this.LineMarkerBrush != null)
                    this.LineMarkerBrush.Dispose();
                if (this.render == null)
                    return;
                this.LineMarkerBrush = new D2D.SolidColorBrush(this.render, value);
            }
        }

        public Color4 ControlChar
        {
            get
            {
                return this.SyntaxResources[TokenType.Control].Color;
            }
            set
            {
                if (this.render == null)
                    return;
                this.SyntaxResources[TokenType.Control] = new D2D.SolidColorBrush(this.render, value);
                this.LayoutCache.Clear();
            }
        }

        public Color4 Url
        {
            get
            {
                return this.MarkerEffects[HilightType.Url].ForeBrush.Color;
            }
            set
            {
                if (this.render == null)
                    return;

                D2D.StrokeStyleProperties prop = new D2D.StrokeStyleProperties();
                prop.DashCap = D2D.CapStyle.Flat;
                prop.DashOffset = 0;
                prop.DashStyle = D2D.DashStyle.Solid;
                prop.EndCap = D2D.CapStyle.Flat;
                prop.LineJoin = D2D.LineJoin.Miter;
                prop.MiterLimit = 0;
                prop.StartCap = D2D.CapStyle.Flat;
                this.MarkerEffects[HilightType.Url] = new DrawingEffect(
                    new D2D.StrokeStyle(this.D2DFactory, prop),
                    new D2D.SolidColorBrush(this.render, value));
                this.LayoutCache.Clear();
            }
        }

        public Color4 Hilight
        {
            get
            {
                return this.MarkerEffects[HilightType.Select].BackBrush.Color;
            }
            set
            {
                if (this.render == null)
                    return;
                Color4 back = value;
                Color4 fore = this.MarkerEffects[HilightType.Sold].ForeBrush.Color;
                this.MarkerEffects[HilightType.Select] =
                    new DrawingEffect(null,
                    new D2D.SolidColorBrush(this.render, fore),
                    new D2D.SolidColorBrush(this.render, back));
                this.LayoutCache.Clear();
            }
        }

        public Color4 Comment
        {
            get
            {
                return this.SyntaxResources[TokenType.Comment].Color;
            }
            set
            {
                if (this.render == null)
                    return;
                this.SyntaxResources[TokenType.Comment] = new D2D.SolidColorBrush(this.render, value);
                this.LayoutCache.Clear();
            }
        }

        public Color4 Literal
        {
            get
            {
                return this.SyntaxResources[TokenType.Literal].Color;
            }
            set
            {
                if (this.render == null)
                    return;
                this.SyntaxResources[TokenType.Literal] = new D2D.SolidColorBrush(this.render, value);
                this.LayoutCache.Clear();
            }
        }

        public Color4 Keyword1
        {
            get
            {
                return this.SyntaxResources[TokenType.Keyword1].Color;
            }
            set
            {
                if (this.render == null)
                    return;
                this.SyntaxResources[TokenType.Keyword1] = new D2D.SolidColorBrush(this.render, value);
                this.LayoutCache.Clear();
            }
        }

        public Color4 Keyword2
        {
            get
            {
                return this.SyntaxResources[TokenType.Keyword2].Color;
            }
            set
            {
                if (this.render == null)
                    return;
                this.SyntaxResources[TokenType.Keyword2] = new D2D.SolidColorBrush(this.render, value);
                this.LayoutCache.Clear();
            }
        }

        public Rectangle ClipRect
        {
            get;
            set;
        }

        const int LineNumberLength = 6;

        public double LineNemberWidth
        {
            get
            {
                double width = this.GetWidth("0");
                return width * (LineNumberLength + 1);
            }
        }

        public int TabWidthChar
        {
            get { return this.tabLength; }
            set
            {
                this.tabLength = value;
                this.tabWidth = (float)this.GetWidth("a") * value;
                this.format.IncrementalTabStop = this.tabWidth;
                this.HiddenChars.TabLength = value;
                this.LayoutCache.Clear();
            }
        }

        public void ReConstructDeviceResource()
        {
            this.ReConstructDeviceResource(this.renderSize.Width, this.renderSize.Height);
        }

        public void ReConstructDeviceResource(double width, double height)
        {
            this.DestructDeviceResource();
            this.ConstructDeviceResource(width, height);
        }

        public void ClearLayoutCache()
        {
            this.LayoutCache.Clear();
        }

        public void DrawCachedBitmap(Rectangle rect)
        {
            if (this.render == null)
                return;
            render.DrawBitmap(this.bitmap, rect, 1.0f, D2D.BitmapInterpolationMode.Linear, rect);
        }

        public void CacheContent()
        {
            if (this.render == null || this.bitmap == null)
                return;
            render.Flush();
            this.bitmap.FromRenderTarget(this.render, new DrawingPoint(), new SharpDX.Rectangle(0, 0, (int)this.renderSize.Width, (int)this.renderSize.Height));
            this.hasCache = true;
        }

        public bool IsVaildCache()
        {
            return this.hasCache;
        }

        public void BegineDraw()
        {
            if (this.render == null)
                return;
            this.render.BeginDraw();
        }

        public System.Action ReCreateTarget;

        public void EndDraw()
        {
            if (this.render == null)
                return;
            Result result = this.render.EndDraw();
            if (result.Code == -2003238900) //D2DERR_RECREATE_TARGET
            {
                if (this.ReCreateTarget != null)
                    this.ReCreateTarget();
            }
        }

        public void DrawLineNumber(int lineNumber, double x, double y)
        {
            if (this.render == null)
                return;
            string lineNumberFormat = "{0," + LineNumberLength + "}";
            DW.TextLayout layout = new DW.TextLayout(this.DWFactory, string.Format(lineNumberFormat, lineNumber), this.format, float.MaxValue, float.MaxValue);
            this.render.DrawTextLayout(new DrawingPointF((float)x, (float)y), layout, this.MarkerEffects[HilightType.Sold].ForeBrush);
            layout.Dispose();
        }

        public void DrawLineMarker(Rectangle rect)
        {
            if (this.render == null)
                return;
            this.render.AntialiasMode = D2D.AntialiasMode.Aliased;
            this.render.FillRectangle(rect, this.LineMarkerBrush);
            this.render.AntialiasMode = D2D.AntialiasMode.PerPrimitive;
        }

        public void DrawCaret(Rectangle rect, bool transparent)
        {
            if (this.render == null)
                return;
            this.render.AntialiasMode = D2D.AntialiasMode.Aliased;
            if (transparent == false)
                this.render.FillRectangle(rect, this.InsertCaretBrush);
            else
                this.render.FillRectangle(rect, this.OverwriteCaretBrush);
            this.render.AntialiasMode = D2D.AntialiasMode.PerPrimitive;
        }

        public void FillBackground(Rectangle rect)
        {
            if (this.render == null)
                return;
            this.render.Clear(this.Background);
        }

        public void DrawOneLine(LineToIndexTable lti, int row, double x, double y, IEnumerable<Selection> SelectRanges, IEnumerable<Marker> MarkerRanges,PreDrawOneLineHandler PreDrawOneLine)
        {
            string str = lti[row];

            if (str == string.Empty || str == null || this.render == null)
                return;

            MyTextLayout layout;
            this.CreateTextLayout(this.format, str, this.ClipRect.Size, out layout);
            if (layout.Drew == false)
            {
                LineToIndexTableData lineData = lti.GetData(row);
                if (lineData.Syntax != null)
                {
                    foreach (SyntaxInfo s in lineData.Syntax)
                    {
                        layout.SetDrawingEffect(this.SyntaxResources[s.type], new DW.TextRange(s.index, s.length));
                    }
                }

                foreach (Marker m in MarkerRanges)
                {
                    if (m.start == -1 || m.length == 0)
                        continue;
                    if (m.hilight != HilightType.None)
                        layout.SetDrawingEffect(this.MarkerEffects[m.hilight], new DW.TextRange(m.start, m.length));
                    if (m.hilight != HilightType.Select && m.hilight != HilightType.None)
                        layout.SetUnderline(true, new DW.TextRange(m.start, m.length));
                }
                layout.Drew = true;
            }

            this.render.PushAxisAlignedClip(this.ClipRect, D2D.AntialiasMode.Aliased);

            IMarkerEffecter effecter = new HilightMarker(this.MarkerEffects[HilightType.Select].BackBrush);
            foreach (Selection sel in SelectRanges)
            {
                if (sel.length == 0 || sel.start == -1)
                    continue;

                effecter.Apply(this.render, layout, sel.start, sel.length, x, y);
            }

            if(PreDrawOneLine != null)
                PreDrawOneLine(layout);

            layout.Draw(textRender, (float)x, (float)y);

            this.render.PopAxisAlignedClip();
        }

        public void ApplyEffect(MyTextLayout layout, HilightType type, int start, int length, double x, double y, bool isBold)
        {
            IMarkerEffecter effecter = null;

            float thickness = isBold ? 2 : 1;

            if(type == HilightType.Squiggle)
                effecter = new SquilleLineMarker(this.MarkerEffects[HilightType.Sold], thickness);
            else if(type == HilightType.Select)
                effecter = new HilightMarker(this.MarkerEffects[HilightType.Select].BackBrush);
            else
                effecter = new LineMarker(this.MarkerEffects[type], thickness);

            if (effecter != null)
                effecter.Apply(this.render, layout, start, length, x, y);
        }

        public int GetIndexFromX(string str, double x)
        {
            if (str == string.Empty || str == null || str[0] == Document.EndOfFile)
                return 0;

            MyTextLayout layout;
            this.CreateTextLayout(this.format, str, this.ClipRect.Size, out layout);
            bool isTrailing, isInsed;
            DW.HitTestMetrics metrics;
            metrics = layout.HitTestPoint((float)x, 0, out isTrailing, out isInsed);
            if (isTrailing)
                return Util.RoundUp(metrics.TextPosition + metrics.Length);
            else
                return Util.RoundUp(metrics.TextPosition);
        }

        public double GetWidthFromIndex(string str, int index)
        {
            MyTextLayout layout;
            this.CreateTextLayout(this.format, str, this.ClipRect.Size, out layout);
            float x, y;
            DW.HitTestMetrics metrics;
            metrics = layout.HitTestTextPosition(index, false, out x, out y);
            float x2;
            layout.HitTestTextPosition(index, true, out x2, out y);

            return x2 - x;
        }

        public double GetWidth(string str)
        {
            MyTextLayout layout;
            this.CreateTextLayout(this.format, str, this.ClipRect.Size, out layout);
            return Util.RoundUp(layout.Metrics.WidthIncludingTrailingWhitespace);
        }

        public double GetXFromIndex(string str, int index)
        {
            MyTextLayout layout;
            this.CreateTextLayout(this.format, str, this.ClipRect.Size, out layout);
            float x, y;
            DW.HitTestMetrics metrics;
            metrics = layout.HitTestTextPosition(index, false, out x, out y);
            return x;
        }

        public double GetHeight(string str)
        {
            MyTextLayout layout;
            this.CreateTextLayout(this.format, str, this.ClipRect.Size, out layout);
            double height = 0;
            DW.LineMetrics[] metrics = layout.GetLineMetrics();
            if (metrics != null && metrics.Length > 0)
                height = metrics[0].Height;
            return height;
        }

        public int AlignIndexToNearestCluster(string str, int index, AlignDirection flow)
        {
            MyTextLayout layout;
            this.CreateTextLayout(this.format, str, this.ClipRect.Size, out layout);
            float x, y;
            DW.HitTestMetrics metrics;
            metrics = layout.HitTestTextPosition(index, false, out x, out y);

            if (flow == AlignDirection.Forward)
                return Util.RoundUp(metrics.TextPosition + metrics.Length);
            else if (flow == AlignDirection.Back)
                return Util.RoundUp(metrics.TextPosition);
            throw new ArgumentOutOfRangeException();
        }

        public List<LineToIndexTableData> BreakLine(Document doc, int startIndex, int endIndex, double wrapwidth)
        {
            List<LineToIndexTableData> output = new List<LineToIndexTableData>();

            this.format.WordWrapping = DW.WordWrapping.Wrap;

            foreach (string str in Util.GetLines(doc, startIndex, endIndex))
            {
                DW.TextLayout layout = new DW.TextLayout(this.DWFactory, str, this.format, (float)wrapwidth, float.MaxValue);

                int i = startIndex;
                foreach (DW.LineMetrics metrics in layout.GetLineMetrics())
                {
                    if (metrics.Length == 0 && metrics.NewlineLength == 0)
                        continue;

                    bool lineend = false;
                    if (metrics.NewlineLength > 0)
                        lineend = true;

                    output.Add(new LineToIndexTableData(i, (int)metrics.Length, lineend, null));
                    i += Util.RoundUp(metrics.Length);
                }

                layout.Dispose();

                startIndex += str.Length;
            }

            this.format.WordWrapping = DW.WordWrapping.NoWrap;

            if (output.Count > 0)
                output.Last().LineEnd = true;

            return output;
        }

        public void Dispose()
        {
            this.DestructDeviceResource();
            this.LayoutCache.Clear();
            this.HiddenChars.Clear();
            if (this.format != null)
                this.format.Dispose();
            if(this.DWFactory != null)
                this.DWFactory.Dispose();
            if(this.D2DFactory != null)
                this.D2DFactory.Dispose();
        }

        public void ConstructDeviceResource(double width, double height)
        {
            int dpiX, dpiY;
            this.GetDpi(out dpiX, out dpiY);
            D2D.RenderTargetProperties prop = new D2D.RenderTargetProperties(
                D2D.RenderTargetType.Default,
                new D2D.PixelFormat(DXGI.Format.B8G8R8A8_UNorm, D2D.AlphaMode.Ignore),
                dpiX,
                dpiY,
                D2D.RenderTargetUsage.None,
                D2D.FeatureLevel.Level_DEFAULT);

            this.render = this.ConstructRender(this.D2DFactory,prop,width,height);

            D2D.BitmapProperties bmpProp = new D2D.BitmapProperties();
            bmpProp.DpiX = dpiX;
            bmpProp.DpiY = dpiY;
            bmpProp.PixelFormat = new D2D.PixelFormat(DXGI.Format.B8G8R8A8_UNorm, D2D.AlphaMode.Ignore);
            this.bitmap = new D2D.Bitmap(this.render, new DrawingSize((int)width, (int)height), bmpProp);
            this.hasCache = false;

            this.ConstrctedResource();

            this.textRender = new CustomTextRenderer(this.render, this.MarkerEffects[HilightType.Sold].ForeBrush);

            this.renderSize.Width = width;
            this.renderSize.Height = height;
        }


        public System.Func<D2D.Factory,D2D.RenderTargetProperties,double,double,D2D.RenderTarget> ConstructRender;
        public System.Action ConstrctedResource;

        [DllImport("gdi32.dll")]
        public static extern int GetDeviceCaps(IntPtr hDc, int nIndex);

        [DllImport("user32.dll")]
        public static extern IntPtr GetDC(IntPtr hWnd);

        [DllImport("user32.dll")]
        public static extern int ReleaseDC(IntPtr hWnd, IntPtr hDc);

        public const int LOGPIXELSX = 88;
        public const int LOGPIXELSY = 90;

        void GetDpi(out int dpiX, out int dpiY)
        {
            IntPtr hDc = GetDC(IntPtr.Zero);
            dpiX = GetDeviceCaps(hDc, LOGPIXELSX);
            dpiY = GetDeviceCaps(hDc, LOGPIXELSY);

            ReleaseDC(IntPtr.Zero, hDc);
        }

        public System.Action DestructRender;

        public void DestructDeviceResource()
        {
            this.hasCache = false;
            if (this.bitmap != null)
                this.bitmap.Dispose();
            this.MarkerEffects.Clear();
            this.SyntaxResources.Clear();
            if (this.LineMarkerBrush != null)
                this.LineMarkerBrush.Dispose();
            if (this.InsertCaretBrush != null)
                this.InsertCaretBrush.Dispose();
            this.DestructRender();
        }

        bool CreateTextLayout(DW.TextFormat format, string str, Size size, out MyTextLayout layout)
        {
            bool hasLayout = this.LayoutCache.TryGetValue(str, out layout);
            if (hasLayout == false)
            {
                layout = new MyTextLayout(this.DWFactory, str, format, size);
                this.LayoutCache.Add(str, layout);
                this.ParseLayout(layout, str);
            }
            return hasLayout;
        }

        void ParseLayout(MyTextLayout layout, string str)
        {
            for (int i = 0; i < str.Length; i++)
            {
                DW.InlineObject inlineObject = this.HiddenChars.Get(i, str);
                if (inlineObject != null)
                {
                    layout.SetInlineObject(inlineObject, new DW.TextRange(i, 1));
                    layout.SetDrawingEffect(this.SyntaxResources[TokenType.Control], new DW.TextRange(i, 1));
                }
            }
            return;
        }

        void Markers_Updated(object sender, EventArgs e)
        {
            this.ClearLayoutCache();
        }
    }
}
