﻿using System;
using System.Linq;
using System.Collections.Generic;

namespace FooEditEngine
{
    enum AdjustFlow
    {
        Row,
        Col,
        Both,
    }

    /// <summary>
    /// キャレットとドキュメントの表示を担当します。レイアウト関連もこちらで行います
    /// </summary>
    class View : IDisposable
    {
        const int SpiltCharCount = 1024;

        protected Document Document;

        Point _CaretLocation = new Point();
        Point2 _Src = new Point2();
        TextPoint _CaretPostion = new TextPoint();
        LineToIndexTable _LayoutLines;
        double MarginLeft;
        double _LongestWidth;
        long tickCount;
        Rectangle _Rect;
        bool _DrawLineNumber,_CaretBlink,_isLineBreak;


        /// <summary>
        /// コンストラクター
        /// </summary>
        /// <remarks>Graphicsプロパティをセットした後で必ずDocument.Clear(false)を呼び出してください。呼び出さない場合、例外が発生します</remarks>
        public View(Document doc, ITextRender r,int MarginLeft = 5)
        {
            this.Document = doc;
            this.Document.Update += new DocumentUpdateEventHandler(doc_Update);
            this._LayoutLines = new LineToIndexTable(this.Document);
            this._LayoutLines.SpilitString = new SpilitStringEventHandler(LayoutLines_SpilitStringByChar);
            this.render = r;
            this.render.ChangedRenderResource += new ChangedRenderResourceEventHandler(render_ChangedRenderResource);
            this._CaretLocation.X = this.MarginLeft = MarginLeft;
            this.CaretBlinkTime = 500;
            this.CaretWidthOnInsertMode = 1;
        }

        /// <summary>
        /// テキストレンダラ―
        /// </summary>
        public ITextRender render
        {
            get;
            set;
        }

        /// <summary>
        /// URLをハイパーリンクとして表示するなら真。そうでないなら偽
        /// </summary>
        public bool UrlMark
        {
            get { return this.LayoutLines.UrlMark; }
            set { this.LayoutLines.UrlMark = value; }
        }

        /// <summary>
        /// ラインマーカーを描くなら偽。そうでなければ真
        /// </summary>
        public bool HideLineMarker
        {
            get;
            set;
        }

        /// <summary>
        /// キャレットを描くなら偽。そうでなければ真
        /// </summary>
        public bool HideCaret
        {
            get;
            set;
        }

        /// <summary>
        /// 挿入モードなら真を返し、上書きモードなら偽を返す
        /// </summary>
        public bool InsertMode
        {
            get;
            set;
        }

        /// <summary>
        /// キャレットの点滅間隔
        /// </summary>
        public int CaretBlinkTime
        {
            get;
            set;
        }

        /// <summary>
        /// 挿入モード時のキャレットの幅
        /// </summary>
        public double CaretWidthOnInsertMode
        {
            get;
            set;
        }

        /// <summary>
        /// キャレットを点滅させるなら真。そうでないなら偽
        /// </summary>
        /// <remarks>キャレット点滅タイマーもリセットされます</remarks>
        public bool CaretBlink
        {
            get { return this._CaretBlink; }
            set
            {
                this._CaretBlink = value;
                if (value)
                    this.tickCount = DateTime.Now.Ticks + this.To100nsTime(this.CaretBlinkTime);
            }
        }

        /// <summary>
        /// 一ページの高さに収まる行数を返す
        /// </summary>
        public int LineCountOnScreen
        {
            get;
            protected set;
        }

        /// <summary>
        /// 折り返し時の右マージン
        /// </summary>
        public double LineBreakingMarginWidth
        {
            get { return this.PageBound.Width * 5 / 100; }
        }

        /// <summary>
        /// 保持しているレイアウト行
        /// </summary>
        public LineToIndexTable LayoutLines
        {
            get { return this._LayoutLines; }
        }

        /// <summary>
        /// 行番号を表示するかどうか
        /// </summary>
        public bool DrawLineNumber
        {
            get { return this._DrawLineNumber; }
            set
            {
                this._DrawLineNumber = value;
                CalculateClipRect();
            }
        }

        /// <summary>
        /// ページ全体を表す領域
        /// </summary>
        public Rectangle PageBound
        {
            get { return this._Rect; }
            set
            {
                if (value.Width < 0 || value.Height < 0)
                    throw new ArgumentOutOfRangeException("");
                this._Rect = value;
                CalculateClipRect();
                CalculateLineCountOnScreen();
            }
        }

        /// <summary>
        /// Draw()の対象となる領域の左上を表す
        /// </summary>
        public Point2 Src
        {
            get { return this._Src; }
            set { this._Src = value; }
        }

        /// <summary>
        /// 最も長い行の幅
        /// </summary>
        public double LongestWidth
        {
            get { return this._LongestWidth; }
        }

        /// <summary>
        /// キャレットがある領域を示す
        /// </summary>
        public Point CaretLocation
        {
            get { return this._CaretLocation; }
        }

        /// <summary>
        /// レイアウト行のどこにキャレットがあるかを表す
        /// </summary>
        public TextPoint CaretPostion
        {
            get { return this._CaretPostion; }
        }

        /// <summary>
        /// 桁折り処理を行うかどうか
        /// </summary>
        /// <remarks>
        /// 変更した場合、呼び出し側で再描写を行う必要があります
        /// </remarks>
        public bool isLineBreak
        {
            get
            {
                return this._isLineBreak;
            }
            set
            {
                this._isLineBreak = value;
                if (value)
                    this._LayoutLines.SpilitString = new SpilitStringEventHandler(LayoutLines_SpilitStringByPixelbase);
                else
                    this._LayoutLines.SpilitString = new SpilitStringEventHandler(LayoutLines_SpilitStringByChar);
                this.PerfomLayouts();
            }
        }

        /// <summary>
        /// シンタックスハイライター
        /// </summary>
        public IHilighter Hilighter
        {
            get { return this.LayoutLines.Hilighter; }
            set { this.LayoutLines.Hilighter = value; }
        }

        /// <summary>
        /// タブの幅
        /// </summary>
        /// <remarks>変更した場合、呼び出し側で再描写する必要があります</remarks>
        public int TabStops
        {
            get { return this.render.TabWidthChar; }
            set { this.render.TabWidthChar = value; }
        }

        /// <summary>
        /// すべてのレイアウト行を破棄し、再度レイアウトをやり直す
        /// </summary>
        public virtual void PerfomLayouts()
        {
            this.doc_Update(this.Document,new DocumentUpdateEventArgs(UpdateType.Clear,-1,-1,-1));
            this.doc_Update(this.Document, new DocumentUpdateEventArgs(UpdateType.Replace, 0, 0, this.Document.Length));
        }

        /// <summary>
        /// キャレット位置を再調整する
        /// </summary>
        public virtual void AdjustCaret()
        {
            int row = this.CaretPostion.row;
            if (this.CaretPostion.row > this.LayoutLines.Count - 1)
                row = this.LayoutLines.Count - 1;
            this.JumpCaret(row, 0);
            this.AdjustCaretAndSrc();
        }

        /// <summary>
        /// Rectで指定された範囲にドキュメントを描く
        /// </summary>
        /// <param name="updateRect">描写する範囲</param>
        /// <remarks>キャレットを点滅させる場合、定期的のこのメソッドを呼び出してください</remarks>
        public virtual void Draw(Rectangle updateRect)
        {
            if (this.LayoutLines.Count == 0)
                return;

            if ((updateRect.Height < this.PageBound.Height ||
                updateRect.Width < this.PageBound.Width) && 
                this.render.IsVaildCache())
            {
                this.render.DrawCachedBitmap(updateRect);
            }
            else
            {
                Rectangle background = this.PageBound;
                this.render.FillBackground(background);

                if (this.HideLineMarker == false)
                    this.DrawLineMarker(this.CaretPostion.row);

                Point pos = new Point(this.PageBound.X, this.PageBound.Y);
                pos.X -= this.Src.X;
                double endposy = pos.Y + this.PageBound.Height;
                for (int i = this.Src.Row; pos.Y < endposy && i < this.LayoutLines.Count; i++)
                {
                    var selectRange = GetIMarkerAtLine<Selection>(i, this.Document.Selections);

                    var markerRange = GetIMarkerAtLine<Marker>(i, this.Document.Markers);

                    this.render.DrawOneLine(this.LayoutLines, i, pos.X + this.render.ClipRect.X, pos.Y, selectRange, markerRange);

                    if (this.DrawLineNumber)
                        this.render.DrawLineNumber(i + 1, this.PageBound.X, pos.Y); //行番号は１からはじまる

                    pos.Y += this.render.GetHeight(this.LayoutLines[i]);
                }

                this.render.CacheContent();
            }

            if (this.HideCaret == false)
                this.DrawCaret();
        }

        void DrawCaret()
        {
            long diff = DateTime.Now.Ticks - this.tickCount;
            long blinkTime = this.To100nsTime(this.CaretBlinkTime);

            if (this._CaretBlink && diff % blinkTime >= blinkTime / 2)
                return;

            Rectangle CaretRect = new Rectangle();

            int row = this.CaretPostion.row;
            double lineHeight = this.render.GetHeight(this.LayoutLines[row]);
            double charWidth = this.render.GetWidthFromIndex(this.LayoutLines[row], this.CaretPostion.col);
            bool transparent = false;

            if (this.InsertMode || charWidth <= 0)
            {
                CaretRect.Size = new Size(CaretWidthOnInsertMode, lineHeight);
                CaretRect.Location = new Point(this.CaretLocation.X, this.CaretLocation.Y + this.render.ClipRect.Y);
            }
            else
            {
                double height = lineHeight / 3;
                CaretRect.Size = new Size(charWidth, height);
                CaretRect.Location = new Point(this.CaretLocation.X, this.CaretLocation.Y + lineHeight - height + this.render.ClipRect.Y);
                transparent = true;
            }
            render.DrawCaret(CaretRect,transparent);
        }

        long To100nsTime(int ms)
        {
            return ms * 10000;
        }

        public virtual void DrawLineMarker(int row)
        {
            Point p = this.CaretLocation;
            double height = this.render.GetHeight(this.LayoutLines[this.CaretPostion.row]);
            double width = this.render.ClipRect.Width;
            this.render.DrawLineMarker(new Rectangle(this.render.ClipRect.X,this.CaretLocation.Y,width,height));
        }

        /// <summary>
        /// 指定した座標の一番近くにあるTextPointを取得する
        /// </summary>
        /// <param name="p">(PageBound.X,PageBound.Y)を左上とする相対位置</param>
        /// <returns>レイアウトラインを指し示すTextPoint</returns>
        public virtual TextPoint GetTextPointFromPostion(Point p)
        {
            TextPoint tp = new TextPoint();

            if (this.LayoutLines.Count == 0)
                return tp;

            double y = 0;
            tp.row = this.LayoutLines.Count - 1;
            for (int i = this._Src.Row; i < this.LayoutLines.Count; i++)
            {
                double height = this.render.GetHeight(this.LayoutLines[i]);
                LineToIndexTableData lineData = this.LayoutLines.GetData(i);
                if (y + height > p.Y)
                {
                    tp.row = i;
                    break;
                }
                y += height;
            }

            if (p.X < this.render.ClipRect.X)
                return tp;

            tp.col = GetIndexFromX(this.LayoutLines[tp.row], p.X);

            int lineLength = this.LayoutLines[tp.row].Length;
            if (tp.col > lineLength)
                tp.col = lineLength;

            return tp;
        }

        /// <summary>
        /// X座標に対応するインデックスを取得する
        /// </summary>
        /// <param name="line">対象となる文字列</param>
        /// <param name="x">PageBound.Xからの相対位置</param>
        /// <returns></returns>
        public virtual int GetIndexFromX(string line, double x)
        {
            x -= this.render.ClipRect.X - this.PageBound.X;
            int index = this.render.GetIndexFromX(line, this.Src.X + x);
            return index;
        }

        /// <summary>
        /// インデックスに対応するＸ座標を得る
        /// </summary>
        /// <param name="line">対象となる文字列</param>
        /// <param name="index">インデックス</param>
        /// <returns>PageBound.Xからの相対位置を返す</returns>
        public virtual double GetXFromIndex(string line, int index)
        {
            double x = this.render.GetXFromIndex(line, index);
            return x - Src.X + this.render.ClipRect.X;
        }

        /// <summary>
        /// TextPointに対応する座標を得る
        /// </summary>
        /// <param name="tp">レイアウトライン上の位置</param>
        /// <returns>(PageBound.X,PageBound.Y)を左上とする相対位置</returns>
        public virtual Point GetPostionFromTextPoint(TextPoint tp)
        {
            Point p = new Point();
            for (int i = this.Src.Row; i < tp.row; i++)
                p.Y += this.render.GetHeight(this.LayoutLines[i]);
            p.X = this.GetXFromIndex(this.LayoutLines[tp.row], tp.col);
            p.Y += this.render.ClipRect.Y;
            return p;
        }

        /// <summary>
        /// キャレットを指定した位置に移動させる
        /// </summary>
        /// <param name="row"></param>
        /// <param name="col"></param>
        public virtual void JumpCaret(int row, int col)
        {
            this._CaretPostion.row = row;
            this._CaretPostion.col = col;
        }

        /// <summary>
        /// キャレットがあるところまでスクロールする
        /// </summary>
        /// <return>再描写する必要があるなら真を返す</return>
        /// <remarks>Document.Update(type == UpdateType.Clear)イベント時に呼び出した場合、例外が発生します</remarks>
        public virtual bool AdjustCaretAndSrc(AdjustFlow flow = AdjustFlow.Both)
        {
            if (this.PageBound.Width == 0 || this.PageBound.Height == 0)
            {
                this.SetCaretPostion(this.MarginLeft, 0);
                return false;
            }

            bool result = false;
            TextPoint tp = this.CaretPostion;
            double x = this.CaretLocation.X;
            double y = this.CaretLocation.Y;

            if (flow == AdjustFlow.Col || flow == AdjustFlow.Both)
            {
                x = this.render.GetXFromIndex(this.LayoutLines[tp.row], tp.col);

                double left = this.Src.X;
                double right = this.Src.X + this.render.ClipRect.Width - this.LineBreakingMarginWidth;

                if (x >= left && x < right)    //xは表示領域にないにある
                    x -= left;
                else if (x >= right) //xは表示領域の右側にある
                {
                    this._Src.X = x - this.render.ClipRect.Width + this.LineBreakingMarginWidth;
                    x = this.render.ClipRect.Width - this.LineBreakingMarginWidth;
                    result = true;
                }
                else if (x < left)    //xは表示領域の左側にある
                {
                    this._Src.X = x;
                    x = this.PageBound.X;
                    result = true;
                }
            }

            if (flow == AdjustFlow.Row || flow == AdjustFlow.Both)
            {
                int caretRow = 0;
                int lineCount = this.LineCountOnScreen;
                if (tp.row >= this.Src.Row && tp.row < this.Src.Row + lineCount)
                    caretRow = tp.row - this.Src.Row;
                else if (tp.row >= this.Src.Row + lineCount)
                {
                    caretRow = lineCount;
                    this._Src.Row = tp.row - lineCount;
                    result = true;
                    CalculateLineCountOnScreen();
                }
                else if (tp.row < this.Src.Row)
                {
                    this._Src.Row = tp.row;
                    result = true;
                    CalculateLineCountOnScreen();
                }

                y = 0;

                if (caretRow > 0)
                {
                    for (int i = 0; i < caretRow; i++)
                        y += this.render.GetHeight(this.LayoutLines[this.Src.Row + i]);
                }
            }

            this.SetCaretPostion(x, y);

            return result;
        }

        /// <summary>
        /// キャレットを指定した座標に移動させる
        /// </summary>
        /// <param name="x">render.ClipRect.Xからの相対座標X</param>
        /// <param name="y">render.ClipRect.Yからの相対座標X</param>
        public virtual void SetCaretPostion(double x, double y)
        {
            this._CaretLocation = new Point(x + this.render.ClipRect.X - this.PageBound.X, y);
        }

        /// <summary>
        /// レイアウト行をテキストポイントからインデックスに変換する
        /// </summary>
        /// <param name="tp">テキストポイント表す</param>
        /// <returns>インデックスを返す</returns>
        public virtual int GetIndexFromLayoutLine(TextPoint tp)
        {
            return this.LayoutLines.GetIndexFromTextPoint(tp);
        }

        /// <summary>
        /// インデックスからレイアウト行を指し示すテキストポイントに変換する
        /// </summary>
        /// <param name="index">インデックスを表す</param>
        /// <returns>テキストポイント返す</returns>
        public virtual TextPoint GetLayoutLineFromIndex(int index)
        {
            return this.LayoutLines.GetTextPointFromIndex(index);
        }

        /// <summary>
        /// 指定した座標までスクロールする
        /// </summary>
        /// <param name="x"></param>
        /// <param name="row"></param>
        /// <returns>trueの場合、スクロールに失敗したことを表す</returns>
        public virtual bool TryScroll(double x, int row)
        {
            if (x < 0 || row < 0)
                return true;
            if (row > this.LayoutLines.Count - 1)
                return true;
            this._Src.X = x;
            this._Src.Row = row;
            CalculateLineCountOnScreen();
            return false;
        }

        public virtual void Dispose()
        {
            this.Document.Update -= new DocumentUpdateEventHandler(this.doc_Update);    //これをしないと複数のビューを作成した時に妙なエラーが発生する
            GC.SuppressFinalize(this);
        }

        protected virtual void CalculateClipRect()
        {
            if (this.DrawLineNumber)
                this.render.ClipRect = Util.OffsetAndDeflate(this.PageBound, new Size(this.render.LineNemberWidth, 0));
            else
                this.render.ClipRect = Util.OffsetAndDeflate(this.PageBound, new Size(MarginLeft, 0));
            if (this.render.ClipRect.Width < 0)
                this.render.ClipRect = this.PageBound;
        }

        protected virtual void CalculateLineCountOnScreen()
        {
            if (this.LayoutLines.Count == 0)
                return;

            double y = 0;
            int i = this.Src.Row;
            do
            {
                string str = i < this.LayoutLines.Count ?
                    this.LayoutLines[i] :
                    this.LayoutLines[this.LayoutLines.Count - 1];

                double width = this.render.GetWidth(str);

                if (width > this._LongestWidth)
                    this._LongestWidth = width;

                double lineHeight = this.render.GetHeight(str);
                
                y += lineHeight;
                
                if (y >= this.PageBound.Height)
                    break;
                
                i++;
            } while (true);
            this.LineCountOnScreen = Math.Max(i - this.Src.Row - 1, 0);
        }

        void render_ChangedRenderResource(object sender, EventArgs e)
        {
            CalculateClipRect();
            CalculateLineCountOnScreen();
        }

        void doc_Update(object sender, DocumentUpdateEventArgs e)
        {
            switch (e.type)
            {
                case UpdateType.Replace:
                    this._LayoutLines.UpdateAsReplace(e.startIndex, e.removeLength, e.insertLength);
                    this.CalculateLineCountOnScreen();
                    break;
                case UpdateType.Clear:
                    this.LayoutLines.Clear();
                    this._LongestWidth = 0;
                    break;
            }
        }

        IList<LineToIndexTableData> LayoutLines_SpilitStringByPixelbase(object sender, SpilitStringEventArgs e)
        {
            double WrapWidth;
            WrapWidth = this.render.ClipRect.Width - LineBreakingMarginWidth;  //余白を残さないと欠ける

            if (WrapWidth < 0 && this.isLineBreak)
                throw new InvalidOperationException();

            int startIndex = e.index;
            int endIndex = e.index + e.length - 1;

            return this.render.BreakLine(e.buffer, startIndex, endIndex, WrapWidth);
        }

        IList<LineToIndexTableData> LayoutLines_SpilitStringByChar(object sender, SpilitStringEventArgs e)
        {
            int startIndex = e.index;
            int endIndex = e.index + e.length - 1;
            List<LineToIndexTableData> output = new List<LineToIndexTableData>();

            foreach (string str in Util.GetLines(e.buffer, startIndex, endIndex, SpiltCharCount))
            {
                char c = str.Last();
                bool hasNewLine = c == Document.NewLine;
                output.Add(new LineToIndexTableData(startIndex, str.Length, hasNewLine, null));
                startIndex += str.Length;
            }

            if(output.Count > 0)
                output.Last().LineEnd = true;

            return output;
        }

        protected IEnumerable<T> GetIMarkerAtLine<T>(int line, IEnumerable<T> collection) where T : IMarker
        {
            int StartIndex = this.LayoutLines.GetIndexFromLineNumber(line);
            int EndIndex = StartIndex + this.LayoutLines.GetData(line).Length - 1;
            var selectRange = from s in collection
                              let n = ConvertAbsIndexToRelIndex(s, StartIndex,EndIndex)
                              select n;
            return selectRange;
        }

        T ConvertAbsIndexToRelIndex<T>(T n, int StartIndex,int EndIndex) where T : IMarker
        {
            n = Util.NormalizeIMaker<T>(n);

            int markerEnd = n.start + n.length - 1;

            if (n.start >= StartIndex && markerEnd <= EndIndex)
                n.start -= StartIndex;
            else if (n.start >= StartIndex && n.start <= EndIndex)
            {
                n.start -= StartIndex;
                n.length = EndIndex - StartIndex + 1;
            }
            else if (markerEnd >= StartIndex && markerEnd <= EndIndex)
            {
                n.start = 0;
                n.length = markerEnd - StartIndex + 1;
            }
            else if (n.start >= StartIndex && markerEnd <= EndIndex)
                n.start -= StartIndex;
            else if (n.start <= StartIndex && markerEnd > EndIndex)
            {
                n.start = 0;
                n.length = EndIndex - StartIndex + 1;
            }
            else
            {
                n.start = -1;
                n.length = 0;
            }
            return n;
        }

    }
}
