﻿// TSF のデバッグ表示を行うかどうか？
//#define TSF_DEBUG_OUTPUT
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Diagnostics;
using System.Reflection;
using System.Runtime.InteropServices;
using System.Globalization;

using DotNetTextStore.UnmanagedAPI.TSF;
using DotNetTextStore.UnmanagedAPI.TSF.TextStore;
using DotNetTextStore.UnmanagedAPI.WinDef;
using DotNetTextStore.UnmanagedAPI.WinError;


namespace DotNetTextStore 
{
    #if TSF_DEBUG_OUTPUT
    /// <summary>コールスタックの階層にあわせてインデントしてデバッグ表示するクラス。</summary>
    public class DebugOut : IDisposable
    {
        public DebugOut(string i_string, params object[] i_params)
        {
            _text = string.Format(i_string, i_params);

            s_callCount++;
            Debug.Indent();
            Debug.WriteLine("");
            Debug.WriteLine(string.Format("{0, 4} : ↓↓↓ ", s_callCount) + _text);
        }


        public void Dispose()
        {
            s_callCount++;
            Debug.WriteLine(string.Format("{0, 4} : ↑↑↑ ", s_callCount) + _text);
            Debug.Unindent();
        }


        string      _text;
        static int  s_callCount = 0;
    }
    #endif

    //=============================================================================================


    public struct TextDisplayAttribute
    {
        public int startIndex;
        public int endIndex;
        public TF_DISPLAYATTRIBUTE attribute;
    }

    //========================================================================================


    /// <summary>Dispose() で TextStore のロック解除を行うクラス。</summary>
    public class Unlocker : IDisposable
    {
        /// <summary>コンストラクタ</summary>
        public Unlocker(TextStore io_textStore)
        {
            _textStore = io_textStore;
        }
        /// <summary>ロックが成功したかどうか調べる。</summary>
        public bool IsLocked
        {
            get { return _textStore != null; }
        }
        /// <summary>アンロックを行う。</summary>
        void IDisposable.Dispose()
        {
            if (_textStore != null)
            {
                _textStore.UnlockDocument();
                _textStore = null;
            }
        }

        /// <summary>アンロックを行うテキストストア</summary>
        TextStore _textStore;
    }


    //========================================================================================

    /// <summary>
    ///  テキストストアの実装を担うクラス。
    /// <pre>
    ///   各ドキュメント毎に実装すべき部分をイベントとして切り離して、テキストストアの実装を楽に
    ///   させる。
    /// </pre>
    /// <pre>
    ///   使用側としては各イベントハンドラの実装、フォーカスを取得した時に SetFocus() メソッドの
    ///   呼び出し、選択やテキストが変更された時に NotifySelectionChanged() メソッドや
    ///   NotifyTextChange() メソッドの呼び出しを行う必要がある。
    /// </pre>
    public class TextStore : IDisposable, ITextStoreACP, ITfContextOwnerCompositionSink
    {
        public delegate bool IsReadOnlyHandler();
        public event IsReadOnlyHandler IsReadOnly;

        public delegate int GetStringLengthHandler();
        public event GetStringLengthHandler GetStringLength;

        public delegate void GetSelectionIndexHandler(out int o_start, out int o_end);
        public event GetSelectionIndexHandler GetSelectionIndex;

        public delegate void SetSelectionIndexHandler(int i_start, int i_end);
        public event SetSelectionIndexHandler SetSelectionIndex;

        public delegate string GetStringHandler(int start,int length);
        public event GetStringHandler GetString;

        public delegate void InsertAtSelectionHandler(string i_value);
        public event InsertAtSelectionHandler InsertAtSelection;

        public delegate IntPtr GetHWndHandler();
        public event GetHWndHandler GetHWnd;

        public delegate void GetScreenExtentHandler(
            out POINT o_pointTopLeft,
            out POINT o_pointBottomRight
        );
        public event GetScreenExtentHandler GetScreenExtent;

        public delegate void GetStringExtentHandler(
            int         i_startIndex,
            int         i_endIndex,
            out POINT   o_pointTopLeft,
            out POINT   o_pointBottomRight
        );
        public event GetStringExtentHandler GetStringExtent;

        public delegate bool CompositionStartedHandler();
        public event CompositionStartedHandler CompositionStarted;

        public delegate void CompostionUpdateHandler(int start,int end);
        public event CompostionUpdateHandler CompositionUpdated;

        public delegate void CompositionEndedHandler();
        public event CompositionEndedHandler CompositionEnded;


        //=========================================================================================
        // 生成と破棄
        //=========================================================================================
        #region "生成と破棄"
        /// <summary>コンストラクタ</summary>
        public TextStore()
        {
            #if TSF_DEBUG_OUTPUT
            using(var dbgout = new DebugOut("{0}()", MethodBase.GetCurrentMethod().Name) )
            #endif
            {
                try
                {
                    // スレッドマネージャ－の生成
                    CreateThreadMgr();
                    // カテゴリマネージャーの生成
                    CreateCategoryMgr();
                    // 表示属性マネージャーの生成
                    CreateDisplayAttributeMgr();

                    // ドキュメントマネージャーの生成
                    _threadMgr.CreateDocumentMgr(out _documentMgr);

                    // スレッドマネージャのアクティブ化
                    int clientId = 0;
                    _threadMgr.Activate(out clientId);

                    // コンテキストの生成
                    _documentMgr.CreateContext(clientId, 0, this, out _context, out _editCookie);

                    // コンテキストの push
                    _documentMgr.Push(_context);

                    // ファンクションプロバイダーを取得する。
                    Guid guid = TfDeclarations.GUID_SYSTEM_FUNCTIONPROVIDER;
                    _threadMgr.GetFunctionProvider(ref guid, out _functionProvider);

                    // ITfReconversion オブジェクトを取得する。
                    var guidNull         = new Guid();
                    var guidReconversion = typeof(ITfFnReconversion).GUID;
                    object reconversion  = null;
                    _functionProvider.GetFunction(
                        ref guidNull,
                        ref guidReconversion,
                        out reconversion
                    );
                    _reconversion = reconversion as ITfFnReconversion;

                    // MODEBIAS の初期化
                    uint guidAtom = 0;
                    Guid guidModebiasNone = TfDeclarations.GUID_MODEBIAS_NONE;
                    _categoryMgr.RegisterGUID(ref guidModebiasNone, out guidAtom);
                    _attributeInfo[0].attrID             = TfDeclarations.GUID_PROP_MODEBIAS;
                    _attributeInfo[0].flags              = AttributeInfoFlags.None;
                    _attributeInfo[0].currentValue.vt    = (short)VarEnum.VT_EMPTY;
                    _attributeInfo[0].defaultValue.vt    = (short)VarEnum.VT_I4;
                    _attributeInfo[0].defaultValue.data1 = (IntPtr)guidAtom;
                }
                catch(Exception exception)
                {
                    Debug.WriteLine(exception.Message);
                    Dispose();
                }
            }
        }


        //=========================================================================================
        

        /// <summary>デストラクタ</summary>
        ~TextStore()
        {
            // デストラクタが呼ばれたときには、COMオブジェクトはすでに開放されている模様・・・
            //Dispose();
        }

        
        //=========================================================================================


        /// <summary>
        /// 後処理
        /// </summary>
        public void Dispose()
        {
            #if TSF_DEBUG_OUTPUT
            using(var dbgout = new DebugOut("{0}()", MethodBase.GetCurrentMethod().Name) )
            #endif
            {
                // ITfReconversion の解放
                ReleaseComObject("_reconversion", ref _reconversion);
                // ファンクションプロバイダの解放
                ReleaseComObject("_functionProvider", ref _functionProvider);
                // コンテキストの解放
                ReleaseComObject("_context", ref _context);
                // ドキュメントマネージャの解放
                DestroyDocumentMgr();
                // 表示属性マネージャーの解放
                DestroyDisplayAttributeMgr();
                // カテゴリマネージャーの解放
                DestroyCategoryMgr();
                // スレッドマネージャーの解放
                DestroyThreadMgr();
            }
        }
        #endregion "生成と破棄"

        
        //=========================================================================================
        // コントロール側が状況に応じて呼び出さなければいけない TSF に通知を送るメソッド
        //=========================================================================================
        #region "コントロール側が状況に応じて呼び出さなければいけない TSF に通知を送るメソッド"
        /// <summary>
        /// 選択領域が変更されたことをTSFに伝える。各種ハンドラ内からコールしてはいけない。
        /// </summary>
        public void NotifySelectionChanged()
        {
            if( _sink != null )
            {
                if( (_adviseFlags & AdviseFlags.TS_AS_SEL_CHANGE) != 0 )
                    _sink.OnSelectionChange();
            }
        }

        
        //=========================================================================================


        /// <summary>
        /// テキストが変更されたことをTSFに伝える。各種ハンドラ内からコールしてはいけない。
        /// </summary>
        public void NotifyTextChanged(TS_TEXTCHANGE textChange)
        {
            if( _sink != null )
            {
                if( (_adviseFlags & AdviseFlags.TS_AS_TEXT_CHANGE) != 0 )
                    _sink.OnTextChange(0, ref textChange);
                _sink.OnLayoutChange(TsLayoutCode.TS_LC_CHANGE, 1);
            }
        }

        
        //=========================================================================================


        /// <summary>
        /// テキストが変更されたことをTSFに伝える。各種ハンドラ内からコールしてはいけない。
        /// </summary>
        /// <param name="i_oldLength"></param>
        /// <param name="i_newLength"></param>
        public void NotifyTextChanged(int i_oldLength, int i_newLength)
        {
            if( _sink != null )
            {
                if( (_adviseFlags & AdviseFlags.TS_AS_TEXT_CHANGE) != 0 )
                {
                    var textChange = new TS_TEXTCHANGE();
                    textChange.start = 0;
                    textChange.oldEnd = i_oldLength;
                    textChange.newEnd = i_newLength;

                    _sink.OnTextChange(0, ref textChange);
                }
                _sink.OnLayoutChange(TsLayoutCode.TS_LC_CHANGE, 1);
            }
        }

        
        
        //=========================================================================================


        /// <summary>コントロールがフォーカスを取得した時に呼び出さなければいけない。</summary>
        public void SetFocus()
        {
            if( _threadMgr != null )
                _threadMgr.SetFocus(_documentMgr);
        }
        #endregion "コントロール側が状況に応じて呼び出さなければいけない TSF に通知を送るメソッド"
        

        //=========================================================================================

        /// <summary>表示属性の取得</summary>
        public List<TextDisplayAttribute> EnumAttributes()
        {
            return this.EnumAttributes(0, this.GetStringLength());
        }

        /// <summary>表示属性の取得</summary>
        public List<TextDisplayAttribute> EnumAttributes(int start,int end)
        {
            ITfRangeACP allRange;
            _services.CreateRange(start, end, out allRange);

            List<TextDisplayAttribute> attrs = this.EnumAttributes((ITfRange)allRange);
            
            ReleaseComObject("allRange", ref allRange);
            
            return attrs;
        }

        List<TextDisplayAttribute> EnumAttributes(ITfRange range)
        {
            var retval = new List<TextDisplayAttribute>();

#if TSF_DEBUG_OUTPUT
            using(var dbgout = new DebugOut("{0}()", MethodBase.GetCurrentMethod().Name))
#endif
            {
                ITfProperty property = null;    // プロパティインターフェイス
                IEnumTfRanges enumRanges;         // 範囲の列挙子
                Guid guidPropAttribute = TfDeclarations.GUID_PROP_ATTRIBUTE;

                if (_context == null || _services == null)
                    return retval;

                // GUID_PROP_ATTRIBUTE プロパティを取得。
                _context.GetProperty(ref guidPropAttribute, out property);
                if (property == null)
                    return retval;

                // 全範囲の中で表示属性プロパティをもつ範囲を列挙する。
                property.EnumRanges((int)_editCookie, out enumRanges, range);

                ITfRange[] ranges = new ITfRange[1];
                int fetchedLength = 0;
                while (HRESULT.Succeeded(enumRanges.Next(1, ranges, out fetchedLength))
                    && fetchedLength > 0)
                {
                    // ItfRange から ItfRangeACP を取得。
                    ITfRangeACP rangeACP = ranges[0] as ITfRangeACP;
                    if (rangeACP == null)
                        continue;   // 普通はあり得ない

                    // 範囲の開始位置と文字数を取得。
                    int start, count;
                    rangeACP.GetExtent(out start, out count);

                    // VARIANT 値としてプロパティ値を取得。VT_I4 の GUID ATOM がとれる。
                    VARIANT value = new VARIANT();
                    property.GetValue((int)_editCookie, ranges[0], ref value);
                    if (value.vt == (short)VarEnum.VT_I4)
                    {
                        Guid guid, clsid;
                        ITfDisplayAttributeInfo info;
                        TF_DISPLAYATTRIBUTE attribute;

                        // GUID ATOM から GUID を取得する。
                        _categoryMgr.GetGUID((int)value.data1, out guid);
                        // その GUID から IDisplayAttributeInfo インターフェイスを取得。
                        _displayAttributeMgr.GetDisplayAttributeInfo(
                            ref guid,
                            out info,
                            out clsid
                        );
                        // さらに IDisplayAttributeInfo インターフェイスから表示属性を取得する。
                        info.GetAttributeInfo(out attribute);
                        ReleaseComObject("info", ref info);

                        retval.Add(new TextDisplayAttribute
                        {
                            startIndex = start,
                            endIndex = start + count,
                            attribute = attribute
                        });

#if TSF_DEBUG_OUTPUT
                            Debug.WriteLine(
                                "*******:::: DisplayAttribute: {0} ~ {1} :::::: *********",
                                start, start + count
                            );
                            Debug.WriteLine(attribute.bAttr);
                            Debug.WriteLine(
                                "LineColorType: {0}, {1}",
                                attribute.crLine.type, attribute.crLine.indexOrColorRef
                            );
                            Debug.WriteLine(
                                "TextColorType: {0}, {1}",
                                attribute.crText.type, attribute.crText.indexOrColorRef
                            );
                            Debug.WriteLine(
                                "BackColorType: {0}, {1}",
                                attribute.crBk.type, attribute.crBk.indexOrColorRef
                            );
                            Debug.WriteLine(
                                "Bold, Style  : {0}, {1}",
                                attribute.fBoldLine, attribute.lsStyle
                            );
#endif
                    }

                    ReleaseComObject("rangeACP", ref rangeACP);
                }

                ReleaseComObject("ranges[0]", ref ranges[0]);
                ReleaseComObject("enumRanges", ref enumRanges);
                ReleaseComObject("property", ref property);
            }

            return retval;
        }

        /// <summary>
        /// ドキュメントのロックを行う。
        /// </summary>
        /// <param name="i_writable">読み書き両用ロックか？false の場合、読み取り専用。</param>
        /// <returns>Unlocker のインスタンスを返す。失敗した場合 null を返す。</returns>
        public Unlocker LockDocument(bool i_writable)
        {
            #if TSF_DEBUG_OUTPUT
            using(var dbgout = new DebugOut("{0}({1})", MethodBase.GetCurrentMethod().Name,
                                                        i_writable) )
            #endif
            {
                lock(this)
                {
                    if( this._lockFlags == 0 )
                    {
                        if( i_writable )
                            this._lockFlags = LockFlags.TS_LF_READWRITE;
                        else
                            this._lockFlags = LockFlags.TS_LF_READ;

                        #if TSF_DEBUG_OUTPUT
                            Debug.WriteLine("LockDocument is succeeded.");
                        #endif

                        return new Unlocker(this);
                    }
                    else
                    {
                        #if TSF_DEBUG_OUTPUT
                            Debug.WriteLine("LockDocument is failed. {0}", _lockFlags);
                        #endif

                        return null;
                    }
                }
            }
        }


        //========================================================================================

        
        /// <summary>
        /// ドキュメントのロックを行う。
        /// </summary>
        /// <param name="i_writable">読み書き両用ロックか？false の場合、読み取り専用。</param>
        /// <returns>Unlocker のインスタンスを返す。失敗した場合 null を返す。</returns>
        public Unlocker LockDocument(LockFlags i_flags)
        {
            #if TSF_DEBUG_OUTPUT
            using(var dbgout = new DebugOut("{0}({1})", MethodBase.GetCurrentMethod().Name,
                                                        i_flags) )
            #endif
            {
                lock(this)
                {
                    if( this._lockFlags == 0 )
                    {
                        this._lockFlags = i_flags;

                        #if TSF_DEBUG_OUTPUT
                            Debug.WriteLine("LockDocument is succeeded.");
                        #endif

                        return new Unlocker(this);
                    }
                    else
                    {
                        #if TSF_DEBUG_OUTPUT
                            Debug.WriteLine("LockDocument is failed. {0}", _lockFlags);
                        #endif

                        return null;
                    }
                }
            }
        }


        //========================================================================================

        
        /// <summary>
        /// ドキュメントのアンロックを行う。
        /// </summary>
        public void UnlockDocument()
        {
            #if TSF_DEBUG_OUTPUT
            using(var dbgout = new DebugOut("{0}()", MethodBase.GetCurrentMethod().Name) )
            #endif
            {
                lock(this)
                {
                    _lockFlags = 0;
                }

                if( _pendingLockUpgrade )
                {
                    _pendingLockUpgrade = false;
                    int sessionResult;
                    RequestLock(LockFlags.TS_LF_READWRITE, out sessionResult);
                }
            }
        }


        //========================================================================================

        
        /// <summary>
        /// 指定されたフラグでロックしている状態かどうか調べる。
        /// </summary>
        /// <param name="i_lockFlags"></param>
        /// <returns>ロックされている場合は true, されていない場合は false を返す。</returns>
        public bool IsLocked(LockFlags i_lockFlags)
        {
            #if TSF_DEBUG_OUTPUT
            using(var dbgout = new DebugOut("{0}({1})",
                                            MethodBase.GetCurrentMethod().Name, i_lockFlags) )
            #endif
            {
                #if TSF_DEBUG_OUTPUT
                    Debug.WriteLine(
                        "IsLocked() returns " + ((this._lockFlags & i_lockFlags) == i_lockFlags)
                    );
                #endif
                return (this._lockFlags & i_lockFlags) == i_lockFlags;
            }
        }

        /// <summary>
        /// ロックされているか調べる
        /// </summary>
        /// <returns>ロックされているなら真、そうでなければ偽を返す</returns>
        public bool IsLocked()
        {
#if TSF_DEBUG_OUTPUT
            using(var dbgout = new DebugOut("{0}({1})",
                                            MethodBase.GetCurrentMethod().Name, i_lockFlags) )
#endif
            {
#if TSF_DEBUG_OUTPUT
                    Debug.WriteLine(
                        "IsLocked() returns " + ((this._lockFlags & i_lockFlags) == i_lockFlags)
                    );
#endif
                return this._lockFlags != 0;
            }
        }


        //========================================================================================

        
        /// <summary>
        /// 文字列を挿入する。
        /// </summary>
        public void InsertTextAtSelection(string s)
        {
            TS_TEXTCHANGE   textChange = new TS_TEXTCHANGE();

            using(var unlocker = LockDocument(true))
            {
                if( unlocker != null )
                {
                    int             startIndex, endIndex;

                    InsertTextAtSelection(
                        UnmanagedAPI.TSF.TextStore.InsertAtSelectionFlags.TF_IAS_NOQUERY,
                        s.ToCharArray(),
                        s.Length,
                        out startIndex,
                        out endIndex,
                        out textChange
                    );
                }
            }

            // シンクの OnSelectionChange() をコール。
            NotifySelectionChanged();
            NotifyTextChanged(textChange);
        }


        //========================================================================================
        //  ITextStoreACP インターフェイスの実装
        //========================================================================================
        #region ITextStoreACP インターフェイスの実装


        /// <summary>
        /// ITextStoreACP::AdviseSink() の実装。
        /// <para>
        /// TSF マネージャのシンクインターフェイスを識別する。
        /// </para>
        /// <para>
        /// ITextStoreACP::AdviseSink() メソッドは ITextStoreACPSink インターフェイス
        /// から新しいアドバイズシンクをインストール、または既存のアドバイズシンクの
        /// 修正を行う。シンクインターフェイスは io_unknown_cp パラメータによって指定
        /// される。
        /// </para>
        /// </summary>
        ///
        /// <param name="i_riid">
        /// シンクインターフェイスを指定する。
        /// </param>
        ///
        /// <param name="i_unknown">
        /// シンクインターフェイスへのポインタ。NULL 不可。
        /// </param>
        ///
        /// <param name="i_mask">
        /// アドバイズシンクを通知するイベントを指定する。
        ///   <list type="table">
        ///     <listheader>
        ///       <term>フラグ(値)</term>
        ///       <description>コメント</description>
        ///     </listheader>
        ///     <item>
        ///       <term>TS_AS_TEXT_CHANGE(0x1)</term>
        ///       <description>テキストはドキュメント内で変更された。</description>
        ///     </item>
        ///     <item>
        ///       <term>TS_AS_SEL_CHANGE(0x2)</term>
        ///       <description>テキストはドキュメント内で選択された。</description>
        ///     </item>
        ///     <item>
        ///       <term>TS_AS_LAYOUT_CHANGE(0x04)</term>
        ///       <description>ドキュメントのレイアウトが変更された。</description>
        ///     </item>
        ///     <item>
        ///       <term>TS_AS_ATTR_CHANGE(0x08)</term>
        ///       <description>ドキュメントの属性が変更された。</description>
        ///     </item>
        ///     <item>
        ///       <term>TS_AS_STATUS_CHANGE(0x10)</term>
        ///       <description>ドキュメントのステータスが変更された。</description>
        ///     </item>
        ///     <item>
        ///       <term>TS_AS_ALL_SINKS</term>
        ///       <description>上記全て</description>
        ///     </item>
        ///   </list>
        /// </param>
        public void AdviseSink(ref Guid i_riid, object i_unknown, AdviseFlags i_mask)
        {
            #if TSF_DEBUG_OUTPUT
            using(var dbgout = new DebugOut("{0}()", MethodBase.GetCurrentMethod().Name) )
            #endif
            {
                if( i_riid != typeof(ITextStoreACPSink).GUID )
                {
                    throw new COMException(
                        "ITextStoreACPSink 以外のIIDが渡されました。",
                        UnmanagedAPI.WinError.HRESULT.E_INVALIDARG
                    );
                }

                // 既存のシンクのマスクのみを更新
                if( _sink == i_unknown )
                {
                    _adviseFlags = i_mask;
                }
                // シンクを複数登録しようとした
                else if( _sink != null )
                {
                    throw new COMException(
                        "既にシンクを登録済みです。",
                        UnmanagedAPI.TSF.TextStore.TsResult.CONNECT_E_ADVISELIMIT
                    );
                }
                else
                {
                    // 各種値を保存
                    _services    = (ITextStoreACPServices)i_unknown;
                    _sink        = (ITextStoreACPSink)i_unknown;
                    _adviseFlags = i_mask;
                }
            }
        }


        //========================================================================================

        
        /// <summary>
        /// ITextStoreACP::UnadviseSink() の実装。
        /// <para>
        ///   アプリケーションはもはやTSFマネージャから通知を必要としないことを示す
        ///   ためにアプリケーションによって呼ばれる。TSFマネージャはシンクインター
        ///   フェイスの解放と通知の停止を行う。
        /// </para>
        /// </summary>
        ///
        /// <param name="i_unknown">
        ///   シンクオブジェクトへのポインタ。NULL 不可。
        /// </param>
        ///
        /// <remark>
        ///   新しいシンクオブジェクトを登録する ITextStoreACP::AdviseSink メソッド
        ///   の全ての呼び出しは、このメソッドの呼び出しと対応していなければならない。
        ///   以前に登録されたシンクの dwMask パラメータを更新するだけの
        ///   ITextStoreACP::AdviseSink() メソッドの呼び出しは
        ///   ITextStoreACP::UnadviseSink() メソッドの呼び出しを必要としない。
        ///
        ///   <para>
        ///     io_unknown_cp パラメータは ITextStoreACP::AdviseSink メソッドに渡さ
        ///     れたオリジナルのポインタとして同じ COM 識別子でなければいけない。
        ///   </para>
        /// </remark>
        ///
        /// <returns>
        ///   <list type="table">
        ///     <listheader>
        ///       <term>戻り値</term>
        ///       <description>意味</description>
        ///     </listheader>
        ///     <item>
        ///       <term>S_OK</term>
        ///       <description>メソッドは成功した。</description>
        ///     </item>
        ///     <item>
        ///       <term>CONNECT_E_NOCONNECTION</term>
        ///       <description>
        ///         アクティブなシンクオブジェクトは存在しない。
        ///       </description>
        ///     </item>
        ///   </list>
        /// </returns>
        public void UnadviseSink(object i_unknown)
        {
            #if TSF_DEBUG_OUTPUT
            using(var dbgout = new DebugOut("{0}()", MethodBase.GetCurrentMethod().Name) )
            #endif
            {
                if( _sink == null || _sink != i_unknown )
                {
                    throw new COMException(
                        "シンクは登録されていません。",
                        UnmanagedAPI.TSF.TextStore.TsResult.CONNECT_E_NOCONNECTION
                    );
                }

                _services    = null;
                _sink        = null;
                _adviseFlags = 0;
            }
        }


        //========================================================================================

        
        /// <summary>
        /// ITextStoreACP::RequestLock() の実装。
        /// <para>
        ///   TSFマネージャがドキュメントを修正するためにドキュメントロックを提供す
        ///   るためにTSFマネージャによって呼び出される。このメソッドはドキュメント
        ///   ロックを作成するために ITextStoreACPSink::OnLockGranted() メソッドを呼
        ///   び出さなければいけない。
        /// </para>
        /// </summary>
        ///
        /// <param name="i_lockFlags">ロック要求のタイプを指定する。
        ///   <list type="table">
        ///     <listheader>
        ///       <term>フラグ</term>
        ///       <description>意味</description>
        ///     </listheader>
        ///     <item>
        ///       <term>TS_LF_READ</term>
        ///       <description>
        ///         ドキュメントは読取専用ロックで、修正はできない。
        ///       </description>
        ///     </item>
        ///     <item>
        ///       <term>TS_LF_READWRITE</term>
        ///       <description>
        ///         ドキュメントは読み書き両用で、修正できる。
        ///       </description>
        ///     </item>
        ///     <item>
        ///       <term>TS_LF_SYNC</term>
        ///       <description>
        ///         他のフラグと一緒に指定された場合は、ドキュメントは同期ロックである。
        ///       </description>
        ///     </item>
        ///   </list>
        /// </param>
        ///
        /// <param name="o_sessionResult">
        ///   ロックリクエストが同期ならば、ロック要求の結果である
        ///   ITextStoreACP::OnLockGranted() メソッドからの HRESULT を受け取る。
        ///
        ///   <para>
        ///     ロック要求が非同期で結果が TS_S_ASYNC の場合、ドキュメントは非同期ロッ
        ///     クを受け取る。ロック要求が非同期で結果が TS_E_SYNCHRONOUS の場合は、ド
        ///     キュメントは非同期でロックできない。
        ///   </para>
        /// </param>
        ///
        /// <remark>
        ///   このメソッドはドキュメントをロックするために
        ///   ITextStoreACPSink::OnLockGranted() メソッドを使用する。アプリケーショ
        ///   ンは ITextStoreACP::RequestLock() メソッド内でドキュメントを修正したり
        ///   ITextStoreACPSink::OnTextChange() メソッドを使用して変更通知を送ったり
        ///   してはいけない。もし、アプリケーションがレポートするための変更をペンディ
        ///   ングしているなら、アプリケーションは非同期ロック要求のみを返さなければ
        ///   いけない。
        ///
        ///   <para>
        ///     アプリケーションは複数の RequestLock() メソッド呼び出しをキューに入
        ///     れることを試みてはいけない、なぜなら、アプリケーションは一つのコール
        ///     バックのみを要求されるから。もし、呼び出し元がいくつかの読取要求や一
        ///     つ以上の書き込み要求を作ったとしても、コールバックは書き込みアクセス
        ///     でなければいけない。
        ///   </para>
        ///
        ///   <para>
        ///     成功は、非同期ロックの要求にとって代わって同期ロックを要求する。失敗は
        ///     (複数の)非同期ロックの要求を同期ロックに代えることができない。実装は、
        ///     要求が存在すれば発行済みの非同期要求をまだ満たしていなければいけない。
        ///   </para>
        ///
        ///   <para>
        ///     もしロックが ITextStoreACP::RequestLock() メソッドが返る前に認められ
        ///     たなら、o_sessionResult パラメータは
        ///     ITextStoreACPSink::OnLockGranted() メソッドから返された HRESULT を受
        ///     け取る。もし、呼び出しが成功したが、ロックは後で認められるなら、
        ///     o_sessionResult パラメータは TS_S_ASYNC フラグを受け取る。もし、
        ///     RequestLock() が S_OK 以外を返した場合は o_sessionResult パラメー
        ///     ターは無視すべきである。
        ///   </para>
        ///
        ///   <para>
        ///     ドキュメントが既にロックされている状態で、同期ロック要求の場合は
        ///     o_sessionResult に TS_E_SYNCHRONOUS をセットして S_OK を返す。
        ///     これは同期要求は認められなかったことを示す。
        ///     ドキュメントが既にロックされている状態で、非同期ロック要求の場合は、
        ///     アプリケーションはリクエストをキューに追加し、 o_sessionResult に
        ///     TS_S_ASYNC をセットし、S_OK を返す。ドキュメントが有効になったとき、
        ///     アプリケーションはリクエストをキューから削除して、 OnLockGranted() 
        ///     を呼び出す。このロックリクエストのキューは任意である。アプリケーション
        ///     がサポートしなければいけないシナリオが一つある。ドキュメントが読み取
        ///     り専用ロックで、アプリケーションが OnLockGranted の内部で新しい読み
        ///     書き両用の非同期ロック要求を出した場合、RequestLock は再帰呼び出しを
        ///     引き起こす。アプリケーションは OnLockGranted() を TS_LF_READWRITE と
        ///     共に呼び出してリクエストをアップグレードすべきである。
        ///   </para>
        ///
        ///   <para>
        ///     呼び出し元が読み取り専用ロックを維持している場合を除いて、呼び出し元
        ///     はこのメソッドを再帰的に呼び出してはいけない。この場合、メソッドは非
        ///     同期に書き込み要求を尋ねるために再帰的に呼び出すことができる。書き込
        ///     みロックは読取専用ロックが終わった後に認められるだろう。
        ///   </para>
        ///
        ///   <para>
        ///     ロックの強要：アプリケーションは適切なロックのタイプが存在するかどうか
        ///     ドキュメントへのアクセスを許す前に確かめなければいけない。例えば、
        ///     GetText() を処理することを許可する前に少なくとも読み取り専用ロックが
        ///     行われているかどうか確かめなければいけない。もし、適切なロックがされ
        ///     ていなければ、アプリケーションは TF_E_NOLOCK を返さなければいけない。
        ///   </para>
        /// </remark>
        public void RequestLock(LockFlags i_lockFlags, out int o_sessionResult)
        {
            #if TSF_DEBUG_OUTPUT
            using(var dbgout = new DebugOut("{0}()", MethodBase.GetCurrentMethod().Name) )
            #endif
            {
                o_sessionResult = UnmanagedAPI.WinError.HRESULT.E_FAIL;

                if( _sink == null )
                {
                    throw new COMException(
                        "シンクが登録されていません。",
                        UnmanagedAPI.TSF.TextStore.TsResult.CONNECT_E_NOCONNECTION
                    );
                }

                if( _lockFlags != 0 )   // すでにロックされた状態の場合。
                {
                    if( (i_lockFlags & LockFlags.TS_LF_SYNC) != 0 )
                    {
                        o_sessionResult = TsResult.TS_E_SYNCHRONOUS;
                        return;
                    }
                    else
                    if( (_lockFlags & LockFlags.TS_LF_READWRITE) == LockFlags.TS_LF_READ
                    &&  (i_lockFlags & LockFlags.TS_LF_READWRITE) == LockFlags.TS_LF_READWRITE )
                    {
                        _pendingLockUpgrade = true;
                        o_sessionResult = TsResult.TS_S_ASYNC;
                        return;
                    }

                    throw new COMException();
                }

                using(var unlocker = LockDocument(i_lockFlags))
                {
                    // ロックに失敗した場合は TS_E_SYNCHRONOUS をセットして S_OK を返す。
                    if( unlocker == null )
                    {
                        o_sessionResult = TsResult.TS_E_SYNCHRONOUS;
                    }
                    // ロックに成功した場合は OnLockGranted() を呼び出す。
                    else
                    {
                        try
                        {
                            o_sessionResult = _sink.OnLockGranted(i_lockFlags);
                        }
                        catch(COMException comException)
                        {
                            Debug.WriteLine("OnLockGranted() 呼び出し中に例外が発生。");
                            Debug.WriteLine("  " + comException.Message);
                            o_sessionResult = comException.ErrorCode;
                        }
                        catch(Exception exception)
                        {
                            Debug.WriteLine("OnLockGranted() 呼び出し中に例外が発生。");
                            Debug.WriteLine("  " + exception.Message);
                        }
                    }
                }
            }
        }


        //========================================================================================

        
        /// <summary>
        /// ITextStoreACP::GetStatus() の実装。
        /// <para>
        ///   ドキュメントのステータスを取得するために使用される。ドキュメントのステー
        ///   タスは TS_STATUS 構造体を通して返される。
        /// </para>
        /// </summary>
        ///
        /// <param name="o_documentStatus">
        ///   ドキュメントのステータスを含む TS_STATUS 構造体を受け取る。NULL 不可。
        /// </param>
        public void GetStatus(out TS_STATUS o_documentStatus)
        {
            #if TSF_DEBUG_OUTPUT
            using(var dbgout = new DebugOut("{0}()", MethodBase.GetCurrentMethod().Name) )
            #endif
            {
                if( IsReadOnly != null && IsReadOnly() )
                    o_documentStatus.dynamicFlags = DynamicStatusFlags.TS_SD_READONLY;
                else
                    o_documentStatus.dynamicFlags = 0;
                o_documentStatus.staticFlags  = StaticStatusFlags.TS_SS_REGIONS;
            }
        }


        //========================================================================================

        
        /// <summary>
        /// ITextStoreACP::QueryInsert() の実装。
        ///   <para>
        ///   ドキュメントが選択や挿入を許可することができるかどうかを決めるためにア
        ///   プリケーションから呼ばれる。
        ///   </para>
        ///
        ///   <para>
        ///     QueryInsert() メソッドは指定した開始位置と終了位置が有効かどうかを決
        ///     める。編集を実行する前にドキュメントの編集を調整するために使用される。
        ///     メソッドはドキュメントの範囲外の値を返してはいけない。
        ///   </para>
        /// </summary>
        ///
        /// <param name="i_startIndex">
        ///   テキストを挿入する開始位置。
        /// </param>
        ///
        /// <param name="i_endIndex">
        ///   テキストを挿入する終了位置。選択中のテキストを置換する代わりに指定した
        ///   位置へテキストを挿入する場合は、この値は i_startIndex と同じになる。
        /// </param>
        ///
        /// <param name="i_length">
        ///   置換するテキストの長さ。
        /// </param>
        ///
        /// <param name="o_startIndex">
        ///   挿入されるテキストの新しい開始位置を返す。このパラメータが NULL の場合、
        ///   テキストは指定された位置に挿入できない。この値はドキュメントの範囲外を
        ///   指定できない。
        /// </param>
        ///
        /// <param name="o_endIndex">
        ///   挿入されるテキストの新しい終了位置を返す。このパラメータが NULL の場合、
        ///   テキストは指定された位置に挿入できない。この値はドキュメントの範囲外を
        ///   指定できない
        /// </param>
        ///
        /// <remark>
        ///   o_startIndex と o_endIndex の値は、アプリケーションがどのように
        ///   ドキュメントにテキストを挿入するかに依存している。もし、o_startIndex 
        ///   と o_endIndex が i_startIndex と同じならば、カーソルは挿入後のテキ
        ///   ストの始まりにある。
        ///   もし、o_startIndex と o_endIndex が i_endIndex と同じならば、
        ///   カーソルは挿入後のテキストの終わりにある。o_startIndex と
        ///   o_endIndex の差が挿入されたテキストの長さと同じなら、挿入後、挿入さ
        ///   れたテキストはハイライトされている。
        /// </remark>
        public void QueryInsert(
            int         i_startIndex,
            int         i_endIndex,
            int         i_length,
            out int     o_startIndex,
            out int     o_endIndex
        )
        {
            #if TSF_DEBUG_OUTPUT
            using(var dbgout = new DebugOut("{0}()", MethodBase.GetCurrentMethod().Name) )
            #endif
            {
                if( GetStringLength == null )
                    throw new NotImplementedException();

                if( i_startIndex < 0
                ||  i_startIndex > i_endIndex
                ||  i_endIndex > GetStringLength() )
                {
                    throw new COMException(
                        "インデックスが無効です。",
                        UnmanagedAPI.WinError.HRESULT.E_INVALIDARG
                    );
                }

                o_startIndex = i_startIndex;
                o_endIndex   = i_endIndex;
            }
        }


        //========================================================================================

        
        /// <summary>
        /// ITextStoreACP::GetSelection() の実装。
        ///   <para>
        ///   ドキュメント内のテキスト選択の位置を返す。このメソッドは複数テキスト選
        ///   択をサポートする。呼び出し元はこのメソッドを呼び出す前にドキュメントに
        ///   読取専用ロックをかけておかなければいけない。
        ///   </para>
        /// </summary>
        ///
        /// <param name="i_index">
        ///   処理を開始するテキスト選択を指定する。もし、TS_DEFAULT_SELECTION(-1)
        ///   定数が指定された場合、入力選択は処理を開始する。
        /// </param>
        ///
        /// <param name="i_selectionBufferLength">
        ///   返す選択の最大数を指定する。
        /// </param>
        ///
        /// <param name="o_selections">
        ///   選択されたテキストのスタイル、開始位置、終了位置を受け取る。これらの値
        ///   は TS_SELECTION_ACP 構造体に出力される。
        /// </param>
        ///
        /// <param name="o_fetchedLength">
        ///   o_selections に返された構造体の数を受け取る。
        /// </param>
        public void GetSelection(
            int                 i_index,
            int                 i_selectionBufferLength,
            TS_SELECTION_ACP[]  o_selections,
            out int             o_fetchedLength
        )
        {
            #if TSF_DEBUG_OUTPUT
            using(var dbgout = new DebugOut("{0}()", MethodBase.GetCurrentMethod().Name) )
            #endif
            {
                if( GetSelectionIndex == null )
                    throw new NotImplementedException();

                o_fetchedLength = 0;

                if( IsLocked(LockFlags.TS_LF_READ) == false )
                {
                    throw new COMException(
                        "読取用ロックがされていません。",
                        TsResult.TS_E_NOLOCK
                    );
                }
                
                // -1 は TF_DEFAULT_SELECTION。選択は1つだけしかサポートしないので、
                // TF_DEFAULT_SELECTION でもなく、0 を超える数値が指定された場合はエラー。
                if( i_index != -1 && i_index > 0 )
                {
                    throw new COMException(
                        "選択は1つだけしかサポートしていません。",
                        UnmanagedAPI.WinError.HRESULT.E_INVALIDARG
                    );
                }

                if( i_selectionBufferLength > 0 )
                {
                    int start = 0, end = 0;
                    GetSelectionIndex(out start, out end);
                    if( start <= end )
                    {
                        o_selections[0].start = start;
                        o_selections[0].end   = end;
                        o_selections[0].style.ase = TsActiveSelEnd.TS_AE_END;
                        o_selections[0].style.interimChar = false;
                    }
                    else
                    {
                        o_selections[0].start = end;
                        o_selections[0].end   = start;
                        o_selections[0].style.ase = TsActiveSelEnd.TS_AE_START;
                        o_selections[0].style.interimChar = false;
                    }
            
                    o_fetchedLength = 1;
                }
            }
        }


        //========================================================================================

        
        /// <summary>
        /// ITextStoreACP::SetSelection() の実装。
        ///   <para>
        ///   ドキュメント内のテキストを選択する。アプリケーションはこのメソッドを呼
        ///   ぶ前に読み書き両用ロックをかけなればいけない。
        ///   </para>
        /// </summary>
        ///
        /// <param name="i_count">
        ///   i_selections 内のテキスト選択の数を指定する。
        /// </param>
        ///
        /// <param name="i_selections">
        ///   TS_SELECTION_ACP 構造体を通して選択されたテキストのスタイル、開始位置、
        ///   終了位置を指定する。
        /// 
        ///   <para>
        ///     開始位置と終了位置が同じ場合は、指定した位置にキャレットを配置する。
        ///     ドキュメント内に一度に一つのみキャレットを配置できる。
        ///   </para>
        /// </param>
        public void SetSelection(int i_count, TS_SELECTION_ACP[] i_selections)
        {
            #if TSF_DEBUG_OUTPUT
            using(var dbgout = new DebugOut("{0}({1}, {2})",
                                            MethodBase.GetCurrentMethod().Name,
                                            i_selections[0].start,
                                            i_selections[0].end))
            #endif
            {
                if( SetSelectionIndex == null )
                    throw new NotImplementedException();

                if( i_count != 1 )
                {
                    throw new COMException(
                        "選択は1つだけしかサポートしていません。",
                        UnmanagedAPI.WinError.HRESULT.E_INVALIDARG
                    );
                }

                if( IsLocked(LockFlags.TS_LF_READWRITE) == false )
                {
                    throw new COMException(
                        "ロックされていません。",
                        TsResult.TS_E_NOLOCK
                    );
                }
                
                SetSelectionIndex(i_selections[0].start, i_selections[0].end);
            }
        }


        //========================================================================================


        /// <summary>
        /// ITextStoreACP::GetText() の実装。
        ///   <para>
        ///   指定された位置のテキストに関する情報を返す。このメソッドは可視状態およ
        ///   び不可視状態のテキストや埋め込まれたデータがテキストに関連付けられてい
        ///   るかどうかを返す。
        ///   </para>
        /// </summary>
        ///
        /// <param name="i_startIndex">
        ///   開始位置を指定する。
        /// </param>
        ///
        /// <param name="i_endIndex">
        ///   終了位置を指定する。このパラメータが -1 の場合はテキストストア内の全て
        ///   のテキストを返す。
        /// </param>
        ///
        /// <param name="o_plainText">
        ///   プレインテキストデータを受け取るバッファを指定する。このパラメータが
        ///   NULL の場合は、 cchPlainReq は 0 でなければいけない。
        /// </param>
        ///
        /// <param name="i_plainTextLength">
        ///   プレインテキストの文字数を指定する。
        /// </param>
        ///
        /// <param name="o_plainTextLength">
        ///   プレインテキストバッファへコピーされた文字数を受け取る。このパラメータ
        ///   は NULL を指定できない。値が必要でないときに使用する。
        /// </param>
        ///
        /// <param name="o_runInfos">
        ///   TS_RUNINFO　構造体の配列を受け取る。i_runInfoLength が 0 の場合は NULL。
        /// </param>
        ///
        /// <param name="i_runInfoLength">
        ///   o_runInfos の許容数を指定する。
        /// </param>
        ///
        /// <param name="o_runInfoLength">
        ///   o_runInfosに書き込まれた数を受け取る。このパラメータは NULL を指定で
        ///   きない。
        /// </param>
        ///
        /// <param name="o_nextUnreadCharPos">
        ///   次の読み込んでいない文字の位置を受け取る。このパラメータは NULL を指定
        ///   できない。
        /// </param>
        ///
        /// <remark>
        ///   このメソッドを使う呼び出し元は ITextStoreACP::RequestLock() メソッドを
        ///   呼ぶことでドキュメントに読取専用ロックをかけなければいけない。ロックし
        ///   ていない場合はメソッドは失敗し、TF_E_NOLOCK を返す。
        ///
        ///   <para>
        ///     アプリケーションはまた、内部の理由によって戻り値を切り取ることができる。
        ///     呼び出し元は必須の戻り値を取得するために戻された文字やテキストのラン
        ///     数を注意深く調査しなければいけない。戻り値が不完全ならば、戻り値が完
        ///     全なものとなるまでメソッドを繰り返し呼び出さなければいけない。
        ///   </para>
        ///
        ///   <para>
        ///     呼び出し元は i_runInfoLength パラメータを0にセットし、o_runInfos 
        ///     パラメータを NULL にすることで、プレインテキストを要求できる。しかし
        ///     ながら、呼び出し元は o_plainTextLength に非 NULL の有効な値を提
        ///     供しなければならない、パラメータを使用しないとしても。
        ///   </para>
        ///
        ///   <para>
        ///     i_endIndex が -1 の場合、ストリームの最後がセットされたものとして処
        ///     理しなければいけない。その他の場合、0 以上でなければいけない。
        ///   </para>
        ///
        ///   <para>
        ///     メソッドを抜ける際に、o_nextUnreadCharPos は戻り値によって参照
        ///     されていないストリーム内の次の文字の位置をセットされていなければなら
        ///     ない。呼び出し元はこれを使用して複数の GetText() 呼び出しで素早くテ
        ///     キストをスキャンする。
        ///   </para>
        ///
        /// </remark>
        public void GetText(
            int             i_startIndex,
            int             i_endIndex,
            char[]          o_plainText,
            int             i_plainTextLength,
            out int         o_plainTextLength,
            TS_RUNINFO[]    o_runInfos,
            int             i_runInfoLength,
            out int         o_runInfoLength,
            out int         o_nextUnreadCharPos
        )
        {
            #if TSF_DEBUG_OUTPUT
            using(var dbgout = new DebugOut("{0}()", MethodBase.GetCurrentMethod().Name) )
            #endif
            {
                if( GetStringLength == null || GetString == null )
                    throw new NotImplementedException();

                if( IsLocked(LockFlags.TS_LF_READ) == false )
                {
                    throw new COMException(
                        "読取用ロックがされていません。",
                        TsResult.TS_E_NOLOCK
                    );
                }

                if( (i_endIndex != -1 && i_startIndex > i_endIndex)
                ||  i_startIndex < 0 || i_startIndex > GetStringLength()
                ||  i_endIndex > GetStringLength() )
                {
                    throw new COMException(
                        "インデックスが無効です。",
                        UnmanagedAPI.WinError.HRESULT.E_INVALIDARG
                    );
                }

                var textLength = 0;
                var copyLength = 0;

                if( i_endIndex == -1 )
                    textLength = GetStringLength() - i_startIndex;
                else
                    textLength = i_endIndex - i_startIndex;
                copyLength = Math.Min(i_plainTextLength, textLength);

                // 文字列を格納。
                var text = GetString(i_startIndex,copyLength);
                for(int i = 0; i < copyLength; i++)
                {
                    o_plainText[i] = text[i];
                }

                // 文字数を格納。
                o_plainTextLength = copyLength;
                // RUNINFO を格納。
                if( i_runInfoLength > 0 )
                {
                    o_runInfos[0].type   = TsRunType.TS_RT_PLAIN;
                    o_runInfos[0].length = copyLength;
                }
                o_runInfoLength = 1;

                // 次の文字の位置を格納。
                o_nextUnreadCharPos = i_startIndex + copyLength;
            }
        }


        //========================================================================================


        /// <summary>
        /// ITextStoreACP::SetText() の実装。
        ///   <para>
        ///   SetText() メソッドは与えられた文字位置のテキスト選択をセットする。
        ///   </para>
        /// </summary>
        ///
        /// <param name="i_flags">
        ///   TS_ST_CORRECTION がセットされている場合、テキストは既存のコンテントの
        ///   移動(訂正)であり、特別なテキストマークアップ情報(メタデータ)が保持され
        ///   る - .wav ファイルデータや言語IDなど。クライアントは保持されるマークアッ
        ///   プ情報のタイプを定義する。
        /// </param>
        ///
        /// <param name="i_startIndex">
        ///   置換するテキストの開始位置を指定する。
        /// </param>
        ///
        /// <param name="i_endIndex">
        ///   置換するテキストの終了位置を指定する。-1 の場合は無視される。
        /// </param>
        ///
        /// <param name="i_text">
        ///   置き換えるテキストへのポインタを指定する。テキストの文字数は i_length
        ///   パラメータによって指定されるため、テキスト文字列は NULL 終端文字を持っ
        ///   ていない。
        /// </param>
        ///
        /// <param name="i_length">
        ///   i_text の文字数。
        /// </param>
        ///
        /// <param name="o_textChange">
        ///   次のデータをもつ TS_TEXTCHANGE 構造体へのポインタ。
        ///
        ///   <list type="table">
        ///     <listheader>
        ///       <term>メンバー名</term>
        ///       <description>意味</description>
        ///     </listheader>
        ///     <item>
        ///       <term>acpStart</term>
        ///       <description>
        ///         テキストがドキュメントに挿入される前の開始位置。
        ///       </description>
        ///     </item>
        ///     <item>
        ///       <term>acpOldEnd</term>
        ///       <description>
        ///         テキストがドキュメントに挿入される前の終了位置。値は挿入位置であ
        ///         る acpStart と同じ値となる。もし、acpStart と異なっている場合は、
        ///         テキストはテキストの挿入の前に選択されていた。
        ///       </description>
        ///     </item>
        ///     <item>
        ///       <term>acpNewEnd</term>
        ///       <description>
        ///         テキストを挿入した後の終了位置。
        ///       </description>
        ///     </item>
        ///   </list>
        /// </param>
        public void SetText(
            SetTextFlags        i_flags,
            int                 i_startIndex,
            int                 i_endIndex,
            char[]              i_text,
            int                 i_length,
            out TS_TEXTCHANGE   o_textChange
        )
        {
            #if TSF_DEBUG_OUTPUT
            using(var dbgout = new DebugOut("{0}({1}, {2})",
                                            MethodBase.GetCurrentMethod().Name,
                                            i_startIndex, i_endIndex) )
            #endif
            {
                var selections = new TS_SELECTION_ACP[]
                {
                    new TS_SELECTION_ACP
                    {
                        start = i_startIndex,
                        end   = i_endIndex,
                        style = new TS_SELECTIONSTYLE
                        {
                            ase = TsActiveSelEnd.TS_AE_END,
                            interimChar = false
                        }
                    }
                };

                int startIndex = 0, endIndex = 0;
                SetSelection(1, selections);
                InsertTextAtSelection(
                    InsertAtSelectionFlags.TF_IAS_NOQUERY,
                    i_text,
                    i_length,
                    out startIndex,
                    out endIndex,
                    out o_textChange
                );
            }
        }


        //========================================================================================

        
        /// <summary>
        /// ITextStoreACP::GetFormattedText() の実装。NotImplementedException() 例外を投げる。
        /// <para>
        /// 指定されたテキスト文字列について Formatted Text データを返す。呼び
        /// 出し元はこのメソッドを呼ぶ前に読み書き両用ロックをかけなければいけない。
        /// </para>
        /// </summary>
        public void GetFormattedText(int i_start, int i_end, out object o_obj)
        {
            #if TSF_DEBUG_OUTPUT
            using(var dbgout = new DebugOut("{0}()", MethodBase.GetCurrentMethod().Name) )
            #endif
            {
                throw new NotImplementedException();
            }
        }


        //========================================================================================

        
        /// <summary>
        /// ITextStoreACP::GetEmbedded() の実装。NotImplementedException() 例外を投げる。
        /// <para>
        /// 組み込みオブジェクトを取得する。
        /// </para>
        /// </summary>
        public void GetEmbedded(
            int         i_position,
            ref Guid    i_guidService,
            ref Guid    i_riid,
            out object  o_obj
        )
        {
            #if TSF_DEBUG_OUTPUT
            using(var dbgout = new DebugOut("{0}()", MethodBase.GetCurrentMethod().Name) )
            #endif
            {
                throw new NotImplementedException();
            }
        }


        //========================================================================================

        
        /// <summary>
        /// ITextStoreACP::QueryInsertEmbedded() の実装。NotImplementedException() 例外を投げる。
        /// <para>
        /// 組み込みオブジェクトを挿入できるか問い合わせる。
        /// </para>
        /// </summary>
        public void QueryInsertEmbedded(
            ref Guid    i_guidService,
            int         i_formatEtc,
            out bool    o_insertable
        )
        {
            #if TSF_DEBUG_OUTPUT
            using(var dbgout = new DebugOut("{0}()", MethodBase.GetCurrentMethod().Name) )
            #endif
            {
                throw new NotImplementedException();
            }
        }


        //========================================================================================

        
        /// <summary>
        /// ITextStoreACP::QueryInsertEmbedded() の実装。NotImplementedException() 例外を投げる。
        /// <para>
        /// 組み込みオブジェクトを挿入する。
        /// </para>
        /// </summary>
        public void InsertEmbedded(
            InsertEmbeddedFlags i_flags,
            int                 i_start,
            int                 i_end,
            object              i_obj,
            out TS_TEXTCHANGE   o_textChange
        )
        {
            #if TSF_DEBUG_OUTPUT
            using(var dbgout = new DebugOut("{0}()", MethodBase.GetCurrentMethod().Name) )
            #endif
            {
                throw new NotImplementedException();
            }
        }


        //========================================================================================


        /// <summary>
        /// ITextStoreACP::InsertTextAtSelection() の実装。
        /// <para>
        ///   ITextStoreACP::InsertTextAtSelection() メソッドは挿入位置や選択位置に
        ///   テキストを挿入する。呼び出し元はテキストを挿入する前に読み書き両用ロッ
        ///   クをかけていなければならない。
        /// </para>
        /// </summary>
        ///
        /// <param name="i_flags">
        ///   o_startIndex, o_endIndex パラメータと TS_TEXTCHANGE 構造体のどち
        ///   らがテキストの挿入の結果を含んでいるかを示す。
        ///   TF_IAS_NOQUERY と TF_IAS_QUERYONLY フラグは同時に指定できない。
        ///
        ///   <list type="table">
        ///     <listheader>
        ///       <term>値</term>
        ///       <description>意味</description>
        ///     </listheader>
        ///     <item>
        ///       <term>0</term>
        ///       <description>
        ///         テキスト挿入が発生し、o_startIndex と o_endIndex パラメータ
        ///         はテキスト挿入の結果を含んでいる。TS_TEXTCHANGE 構造体は 0 で埋め
        ///         られていなければならない。
        ///       </description>
        ///     </item>
        ///     <item>
        ///       <term>TF_IAS_NOQUERY</term>
        ///       <description>
        ///         テキストは挿入され、o_startIndex と o_endIndex パラメータ
        ///         の値は NULL にでき、TS_TEXTCHANGE 構造体は埋められなければいけない。
        ///         このフラグはテキスト挿入の結果を見るために使用する。
        ///       </description>
        ///     </item>
        ///     <item>
        ///       <term>TF_IAS_QUERYONLY</term>
        ///       <description>
        ///         テキストは挿入されず、o_startIndex と o_endIndex パラメータ
        ///         はテキスト挿入の結果を含む。これらのパラメータの値は、アプリケー
        ///         ションがどのようにドキュメントにテキストを挿入するかに依存している。
        ///         詳しくは注意を見ること。このフラグは実際にはテキストを挿入しない
        ///         でテキスト挿入の結果を見るために使用する。このフラグを使う場合は
        ///         TS_TEXTCHANGE 構造体を埋める必要はない。
        ///       </description>
        ///     </item>
        ///   </list>
        /// </param>
        ///
        /// <param name="i_text">
        ///   挿入する文字列へのポインタ。NULL 終端にすることができる。
        /// </param>
        ///
        /// <param name="i_length">
        ///   テキスト長を指定する。
        /// </param>
        ///
        /// <param name="o_startIndex">
        ///   テキスト挿入が発生した場所の開始位置へのポインタ。
        /// </param>
        ///
        /// <param name="o_endIndex">
        ///   テキスト挿入が発生した場所の終了位置へのポインタ。このパラメータは挿入
        ///   の場合、o_startIndex パラメータと同じ値になる。
        /// </param>
        ///
        /// <param name="o_textChange">
        ///   次のメンバーを持つ TS_TEXTCHANGE 構造体へのポインタ。
        ///
        ///   <list type="table">
        ///     <listheader>
        ///       <term>メンバー名</term>
        ///       <description>意味</description>
        ///     </listheader>
        ///     <item>
        ///       <term>acpStart</term>
        ///       <description>
        ///         テキストがドキュメントに挿入される前の開始位置。
        ///       </description>
        ///     </item>
        ///     <item>
        ///       <term>acpOldEnd</term>
        ///       <description>
        ///         テキストがドキュメントに挿入される前の終了位置。値は挿入位置であ
        ///         る acpStart と同じ値となる。もし、acpStart と異なっている場合は、
        ///         テキストはテキストの挿入の前に選択されていた。
        ///       </description>
        ///     </item>
        ///     <item>
        ///       <term>acpNewEnd</term>
        ///       <description>
        ///         テキストを挿入した後の終了位置。
        ///       </description>
        ///     </item>
        ///   </list>
        /// </param>
        ///
        /// <remark>
        ///   o_startIndex と o_endIndex パラメータの値はアプリケーションがド
        ///   キュメントにどのようにテキストを挿入したかによる。たとえば、アプリケー
        ///   ションがテキストを挿入後、挿入したテキストの開始位置にカーソルをセット
        ///   したなら、o_startIndex と o_endIndex パラメータは TS_TEXTCHANGE 
        ///   構造体の acpStart メンバと同じ値になる。
        ///   <para>
        ///     アプリケーションは ITextStoreACPSink::OnTextChange() メソッドをこの
        ///     メソッド内で呼ぶべきではない。
        ///   </para>
        /// </remark>
        public void InsertTextAtSelection(
            InsertAtSelectionFlags  i_flags,
            char[]                  i_text,
            int                     i_length,
            out int                 o_startIndex, 
            out int                 o_endIndex, 
            out TS_TEXTCHANGE       o_textChange
        )
        {
            #if TSF_DEBUG_OUTPUT
            using(var dbgout = new DebugOut("{0}()", MethodBase.GetCurrentMethod().Name) )
            #endif
            {
                if( GetSelectionIndex == null || InsertAtSelection == null )
                    throw new NotImplementedException();

                int selectionStart = 0, selectionEnd = 0;
                GetSelectionIndex(out selectionStart, out selectionEnd);

                // 問い合わせのみで実際には操作を行わない
                if( (i_flags & InsertAtSelectionFlags.TF_IAS_QUERYONLY) != 0 )
                {
                    o_startIndex = Math.Min(selectionStart, selectionEnd);
                    o_endIndex   = Math.Max(selectionStart, selectionEnd);//o_startIndex + i_length;
                    
                    o_textChange.start  = o_startIndex;
                    o_textChange.oldEnd = o_endIndex;
                    o_textChange.newEnd = o_startIndex + i_length;
                }
                else
                {
                    var start = Math.Min(selectionStart, selectionEnd);
                    var end   = Math.Max(selectionStart, selectionEnd);

                    #if TSF_DEBUG_OUTPUT
                        Debug.WriteLine(string.Format(
                            "start: {0}, end: {1}, text: {2}",
                            start, end, new string(i_text)
                        ));
                    #endif

                    InsertAtSelection(new string(i_text));

                    o_startIndex = start;
                    o_endIndex   = start + i_length;

                    o_textChange.start  = start;
                    o_textChange.oldEnd = end;
                    o_textChange.newEnd = start + i_length;

                    // InsertAtSelection() 内でカーソル位置を更新しているため、ここでは不要。
                    // 改行した時に位置が狂う。
                    // SetSelectionIndex(start, start + i_length);
                }
            }
        }


        //========================================================================================

        
        /// <summary>
        /// ITextStoreACP::QueryInsertEmbedded() の実装。NotImplementedException() 例外を投げる。
        /// <para>
        ///   選択位置もしくは挿入位置に組み込みオブジェクトを挿入する。
        /// </para>
        /// </summary>
        public void InsertEmbeddedAtSelection(
            InsertAtSelectionFlags  flags,
            object                  obj,
            out int                 start,
            out int                 end,
            out TS_TEXTCHANGE       change
        )
        {
            #if TSF_DEBUG_OUTPUT
            using(var dbgout = new DebugOut("{0}()", MethodBase.GetCurrentMethod().Name) )
            #endif
            {
                throw new NotImplementedException();
            }
        }


        //========================================================================================

        
        /// <summary>
        /// ITextStoreACP::RequestSupportedAttrs() の実装。MODE BIAS のみサポート。
        /// <para>
        ///   ドキュメントのサポートされている属性を取得する。
        /// </para>
        /// </summary>
        /// 
        /// <param name="i_flags">
        ///   後続の ITextStoreACP::RetrieveRequestedAttrs() メソッド呼び出し
        ///   がサポートされる属性を含むかどうかを指定する。もし、
        ///   TS_ATTR_FIND_WANT_VALUE フラグが指定されたなら、後続の
        ///   ITextStoreAcp::RetrieveRequestedAttrs() 呼び出しの後に
        ///   TS_ATTRVAL 構造体のそれらはデフォルト属性の値になる。もし、他の
        ///   フラグが指定されたなら属性がサポートされているかどうか確認する
        ///   だけで、 TS_ATTRVAL 構造体の varValue メンバには VT_EMPTY がセッ
        ///   トされる。
        /// </param>
        /// 
        /// <param name="i_length">
        ///   取得するサポートされる属性の数を指定する。
        /// </param>
        /// 
        /// <param name="i_filterAttributes">
        ///   確認するための属性が指定された TS_ATTRID データタイプへのポイン
        ///   タ。メソッドは TS_ATTRID によって指定された属性のみを返す(他の
        ///   属性をサポートしていたとしても)。
        /// </param>
        public void RequestSupportedAttrs(
            AttributeFlags  i_flags,
            int             i_length,
            Guid[]          i_filterAttributes
        )
        {
            #if TSF_DEBUG_OUTPUT
            using(var dbgout = new DebugOut("{0}()", MethodBase.GetCurrentMethod().Name) )
            #endif
            {
                for(int i = 0; i < i_length; i++)
                {
                    if( i_filterAttributes[i].Equals(_attributeInfo[0].attrID) )
                    {
                        _attributeInfo[0].flags = AttributeInfoFlags.Requested;
                        if( (i_flags & AttributeFlags.TS_ATTR_FIND_WANT_VALUE) != 0 )
                        {
                            _attributeInfo[0].flags |= AttributeInfoFlags.Default;
                        }
                        else
                        {
                            _attributeInfo[0].currentValue = _attributeInfo[0].defaultValue;
                        }
                    }
                }
            }
        }


        //========================================================================================


        /// <summary>
        /// ITextStoreACP::RequestAttrsAtPosition() の実装。MODE BIAS のみサポート。
        /// <para>
        ///   指定した文字位置の属性を取得する。
        /// </para>
        /// </summary>
        /// 
        /// <param name="i_position">
        ///   ドキュメント内の開始位置。
        /// </param>
        /// 
        /// <param name="i_length">
        ///   取得する属性の数。
        /// </param>
        /// 
        /// <param name="i_filterAttributes">
        ///   確認するための属性が指定された TS_ATTRID データタイプへのポインタ。
        /// </param>
        /// 
        /// <param name="i_flags">
        ///   0 でなければいけない。
        /// </param>
        public void RequestAttrsAtPosition(
            int             i_position,
            int             i_length,
            Guid[]          i_filterAttributes,
            AttributeFlags  i_flags
        )
        {
            #if TSF_DEBUG_OUTPUT
            using(var dbgout = new DebugOut("{0}()", MethodBase.GetCurrentMethod().Name) )
            #endif
            {
                for(int i = 0; i < i_length; i++)
                {
                    if( i_filterAttributes[i].Equals(_attributeInfo[0].attrID) )
                    {
                        _attributeInfo[0].flags = AttributeInfoFlags.Requested;
                        if( (i_flags & AttributeFlags.TS_ATTR_FIND_WANT_VALUE) != 0 )
                        {
                            _attributeInfo[0].flags |= AttributeInfoFlags.Default;
                        }
                        else
                        {
                            _attributeInfo[0].currentValue = _attributeInfo[0].defaultValue;
                        }
                    }
                }
            }
        }


        //========================================================================================

        
        /// <summary>
        ///   ITextStoreACP::RequestAttrsTransitioningAtPosition() の実装。現在、サポートする属性は
        ///   無い。
        ///   <para>
        ///   指定した文字位置の属性のリストを取得する。
        ///   </para>
        /// </summary>
        /// <param name="i_position">ドキュメント内の開始位置。</param>
        /// 
        /// <param name="i_length">取得する属性の数。</param>
        /// 
        /// <param name="i_filterAttributes">
        ///   RetrieveRequestedAttr() メソッドを呼ぶための属性を指定する。
        ///   このパラメータがセットされていなければメソッドは指定された位置で
        ///   始まる属性を返す。他の指定可能な値は以下のとおり。
        ///   <list type="table">
        ///     <listheader>
        ///       <term>値</term>
        ///       <description>意味</description>
        ///     </listheader>
        ///     <item>
        ///       <term>TS_ATTR_FIND_WANT_END</term>
        ///       <description>
        ///         指定した文字位置で終了する属性を取得する。
        ///       </description>
        ///     </item>
        ///     <item>
        ///       <term>TS_ATTR_FIND_WANT_VALUD</term>
        ///       <description>
        ///         属性の取得に加えて属性の値を取得する。属性値は
        ///         ITextStoreACP::RetrieveRequestedAttrs() メソッド呼び出しの
        ///         TS_ATTRVAL 構造体の varValue メンバにセットされる。
        ///       </description>
        ///     </item>
        ///   </list>
        /// </param>
        /// <param name="i_flags"></param>
        public void RequestAttrsTransitioningAtPosition(
            int             i_position,
            int             i_length,
            Guid[]          i_filterAttributes,
            AttributeFlags  i_flags
        )
        {
            #if TSF_DEBUG_OUTPUT
            using(var dbgout = new DebugOut("{0}()", MethodBase.GetCurrentMethod().Name))
            #endif
            {
                // 何もしない。
            }
        }


        //========================================================================================


        /// <summary>
        /// ITextStoreACP::QueryInsertEmbedded() の実装。NotImplementedException() 例外を投げる。
        /// <para>
        ///   属性変更が発生している場所の文字位置を決定する。
        /// </para>
        /// </summary>
        /// <param name="i_start">属性変更を検索する開始位置を指定する。</param>
        /// <param name="i_halt">属性変更を検索する終了位置を指定する。</param>
        /// <param name="i_length">チェックするための属性の数を指定する。</param>
        /// <param name="i_filterAttributes">
        ///   チェックするための属性を指定する TS_ATTRID データタイプへのポインタ。
        /// </param>
        /// <param name="i_flags">
        ///   検索方向を指定する。デフォルトではフォワード検索。
        ///   <list type="table">
        ///     <listheader>
        ///       <term>フラグ</term>
        ///       <description>意味</description>
        ///     </listheader>
        ///     
        ///     <item>
        ///       <term>TS_ATTR_FIND_BACKWARDS</term>
        ///       <description>バックワード検索。</description>
        ///     </item>
        ///     <item>
        ///       <term>tS_ATTR_FIND_WANT_OFFSET</term>
        ///       <description>
        ///         o_foundOffset パラメータは i_start からの属性変更のオフセットを受け取る。
        ///       </description>
        ///     </item>
        ///   </list>
        /// </param>
        /// <param name="o_nextIndex">
        ///   属性変更をチェックする次の文字位置を受け取る。
        /// </param>
        /// <param name="o_found">
        ///   属性変更が発見された場合に TRUE を受け取る。そのほかは FALSE を受け取る。
        /// </param>
        /// <param name="o_foundOffset">
        ///   属性変更の開始位置(文字位置ではない)を受け取る。TS_ATTR_FIND_WANT_OFFSET
        ///   フラグが dwFlags にセットされていれば、 acpStart からのオフセットを受け取る。
        /// </param>
        public void FindNextAttrTransition(
            int             i_start,
            int             i_halt,
            int             i_length,
            Guid[]          i_filterAttributes,
            AttributeFlags  i_flags,
            out int         o_nextIndex,
            out bool        o_found,
            out int         o_foundOffset
        )
        {
            #if TSF_DEBUG_OUTPUT
            using(var dbgout = new DebugOut("{0}()", MethodBase.GetCurrentMethod().Name) )
            #endif
            {
                throw new NotImplementedException();
            }
        }


        //========================================================================================

        
        /// <summary>
        /// ITextStoreACP::RetrieveRequestedAttrs() の実装。
        /// <para>
        ///   ITextStoreACP::RequestAttrsAtPosition(),
        ///   TextStoreACP::RequestAttrsTransitioningAtPosition(),
        ///   ITextStoreACP::RequestSupportedAttrs() によって取得された属性を返す。
        /// </para>
        /// </summary>
        /// <param name="i_length">取得するサポートされる属性の数を指定する。</param>
        /// <param name="o_attributeVals">
        ///   サポートされる属性を受け取る TS_ATTRVAL 構造体へのポインタ。この
        ///   構造体のメンバはメソッド呼び出しの i_flags パラメータによる。
        /// </param>
        /// <param name="o_fetchedLength">サポートされる属性の数を受け取る。</param>
        public void RetrieveRequestedAttrs(
            int             i_length,
            TS_ATTRVAL[]    o_attributeVals,
            out int         o_fetchedLength
        )
        {
            #if TSF_DEBUG_OUTPUT
            using(var dbgout = new DebugOut("{0}()", MethodBase.GetCurrentMethod().Name) )
            #endif
            {
                o_fetchedLength = 0;
                for(int i = 0; i < _attributeInfo.Length && o_fetchedLength < i_length; i++)
                {
                    if( (_attributeInfo[i].flags & AttributeInfoFlags.Requested) != 0 )
                    {
                        o_attributeVals[o_fetchedLength].overlappedId = 0;
                        o_attributeVals[o_fetchedLength].attributeId = _attributeInfo[i].attrID;
                        o_attributeVals[o_fetchedLength].val = _attributeInfo[i].currentValue;

                        if( (_attributeInfo[i].flags & AttributeInfoFlags.Default) != 0 )
                            _attributeInfo[i].currentValue = _attributeInfo[i].defaultValue;

                        o_fetchedLength++;
                        _attributeInfo[i].flags = AttributeInfoFlags.None;
                    }
                }
            }
        }


        //========================================================================================

        
        /// <summary>
        /// ITextStoreACP::GetEndACP() の実装。全文字数ではなく1行の文字数を返す。
        /// <para>
        /// ドキュメント内の文字数を取得する。
        /// </para>
        /// </summary>
        ///
        /// <param name="o_length">最後の文字位置 + 1 を受け取る。</param>
        public void GetEndACP(out int o_length)
        {
            #if TSF_DEBUG_OUTPUT
            using(var dbgout = new DebugOut("{0}()", MethodBase.GetCurrentMethod().Name) )
            #endif
            {
                if( GetStringLength == null )
                    throw new NotImplementedException();

                if( IsLocked(LockFlags.TS_LF_READ) == false )
                {
                    throw new COMException(
                        "読取用ロックがされていません。",
                        TsResult.TS_E_NOLOCK
                    );
                }

                o_length = GetStringLength();
            }
        }


        //========================================================================================

        
        /// <summary>
        /// ITextStoreACP::GetActiveView() の実装。返すビューの値は 1 固定。
        /// <para>
        /// 現在のアクティブビューの TsViewCookie データタイプを返す。
        /// </para>
        /// </summary>
        public void GetActiveView(out int o_viewCookie)
        {
            #if TSF_DEBUG_OUTPUT
            using(var dbgout = new DebugOut("{0}()", MethodBase.GetCurrentMethod().Name) )
            #endif
            {
                o_viewCookie = 1;
            }
        }


        //========================================================================================

        
        /// <summary>
        /// ITextStoreACP::GetACPFromPoint() の実装。NotImplementedException() 例外を投げる。
        /// <para>
        ///   スクリーン座標をアプリケーション文字位置に変換する。
        /// </para>
        /// </summary>
        /// <param name="i_viewCookie">コンテキストビューを指定する。</param>
        /// <param name="i_point">スクリーン座標の位置を示す POINT 構造体へのポインタ。</param>
        /// <param name="i_flags">
        ///   文字バウンディングボックスからの相対位置に基づくスクリーン座標を
        ///   戻すための文字位置を指定する。デフォルトでは、返される文字位置は
        ///   スクリーン座標を含む文字バウンディングボックスをもつ文字の位置で
        ///   ある。もし、ポイントが文字を囲むボックス外ならメソッドは NULL ま
        ///   たは TF_E_INVALIDPOINT を返す。その他のフラグは以下のとおり。
        ///   <list>
        ///     <listheader>
        ///       <term>フラグ</term>
        ///       <description>意味</description>
        ///     </listheader>
        ///     
        ///     <item>
        ///       <term>GXFPF_ROUND_NEAREST</term>
        ///       <description>
        ///         もし、スクリーン座標での点が文字バウンディングボックスの中
        ///         に含まれている場合、返される文字位置は ptScreen にもっとも
        ///         近いバウンディングの端を持つ文字の位置である。
        ///       </description>
        ///     </item>
        ///     <item>
        ///       <term>GXFPF_NEAREST</term>
        ///       <description>
        ///         もし、スクリーン座標での点が文字バウンディングボックスに含
        ///         まれていないなら、もっとも近い文字位置が返される。
        ///         <para>
        ///         http://msdn.microsoft.com/en-us/library/ms538418(v=VS.85).aspx の Remarks 参照
        ///         </para>
        ///       </description>
        ///     </item>
        ///   </list>
        /// </param>
        /// <param name="o_index"></param>
        public void GetACPFromPoint(
            int                         i_viewCookie,
            ref POINT                   i_point,
            GetPositionFromPointFlags   i_flags,
            out int                     o_index
        )
        {
            #if TSF_DEBUG_OUTPUT
            using(var dbgout = new DebugOut("{0}()", MethodBase.GetCurrentMethod().Name) )
            #endif
            {
                throw new NotImplementedException();
            }
        }


        //========================================================================================

        
        /// <summary>
        /// ITextStoreACP::GetTextExt() の実装。
        /// <para>
        ///   指定した文字位置のスクリーン座標を返す。読み取り専用ロックをかけて呼ば
        ///   なければいけない。
        /// </para>
        /// </summary>
        ///
        /// <param name="i_viewCookie">
        ///   コンテキストビューを指定する。
        /// </param>
        ///
        /// <param name="i_startIndex">
        ///   ドキュメント内の取得するテキストの開始位置を指定する。
        /// </param>
        ///
        /// <param name="i_endIndex">
        ///   ドキュメント内の取得するテキストの終了位置を指定する。
        /// </param>
        ///
        /// <param name="o_rect">
        ///   指定した文字位置のテキストのスクリーン座標でのバウンディングボックスを
        ///   受け取る。
        /// </param>
        ///
        /// <param name="o_isClipped">
        ///   バウンディングボックスがクリッピングされたかどうかを受け取る。TRUE の
        ///   場合、バウンディングボックスはクリップされたテキストを含み、完全な要求
        ///   されたテキストの範囲を含んでいない。バウンディングボックスは要求された
        ///   範囲が可視状態でないため、クリップされた。
        /// </param>
        ///
        /// <remark>
        ///   ドキュメントウィンドウが最小化されていたり、指定されたテキストが現在表
        ///   示されていないならば、メソッドは S_OK を返して o_rect パラメータに
        ///   { 0, 0, 0, 0 }をセットしなければいけない。
        ///
        ///   <para>
        ///     TSF マネージャー側から変換候補ウィンドウの表示位置を割り出すために使用
        ///     される。
        ///   </para>
        /// </remark>
        public void GetTextExt(
            int         i_viewCookie,
            int         i_startIndex,
            int         i_endIndex,
            out RECT    o_rect,
            out bool    o_isClipped
        )
        {
            #if TSF_DEBUG_OUTPUT
            using(var dbgout = new DebugOut("{0}()", MethodBase.GetCurrentMethod().Name) )
            #endif
            {
                if( GetStringLength == null )
                    throw new NotImplementedException();

                // 読取用ロックの確認。
                if( IsLocked(LockFlags.TS_LF_READ) == false )
                {
                    throw new COMException(
                        "読取用ロックがされていません。",
                        TsResult.TS_E_NOLOCK
                    );
                }

                if( (i_endIndex != -1 && i_startIndex > i_endIndex)
                ||  i_startIndex < 0 || i_startIndex > GetStringLength()
                ||  i_endIndex > GetStringLength() )
                {
                    throw new COMException(
                        "インデックスが無効です。",
                        UnmanagedAPI.WinError.HRESULT.E_INVALIDARG
                    );
                }

                if( i_endIndex == -1 )
                    i_endIndex = GetStringLength();
                
                var pointTopLeft = new POINT();
                var pointBotttomRight = new POINT();
                GetStringExtent(i_startIndex, i_endIndex, out pointTopLeft, out pointBotttomRight);

                o_rect.left   = (int)pointTopLeft.x;
                o_rect.top    = (int)pointTopLeft.y;
                o_rect.bottom = (int)pointBotttomRight.y;//startFormattedText.Height);
                o_rect.right  = (int)pointBotttomRight.x;
                o_isClipped   = false;
            }
        }


        //========================================================================================

        
        /// <summary>
        /// ITextStoreACP::GetScreenExt() の実装。
        /// <para>
        ///   テキストストリームが描画されるディスプレイサーフェイスのスクリーン座標
        ///   でのバウンディングボックスを取得する。
        /// </para>
        /// </summary>
        ///
        /// <remark>
        ///   ドキュメントウィンドウが最小化されていたり、指定されたテキストが現在表
        ///   示されていないならば、メソッドは S_OK を返して o_rect パラメータに
        ///   { 0, 0, 0, 0 }をセットしなければいけない。
        /// </remark>
        public void GetScreenExt(int i_viewCookie, out RECT o_rect)
        {
            #if TSF_DEBUG_OUTPUT
            using(var dbgout = new DebugOut("{0}()", MethodBase.GetCurrentMethod().Name) )
            #endif
            {
                if( GetScreenExtent == null )
                    throw new NotImplementedException();

                POINT pointTopLeft, pointBottomRight;

                GetScreenExtent(out pointTopLeft, out pointBottomRight);

                o_rect.left     = (int)pointTopLeft.x;
                o_rect.top      = (int)pointTopLeft.y;
                o_rect.right    = (int)pointBottomRight.x;
                o_rect.bottom   = (int)pointBottomRight.y;
            }
        }


        //========================================================================================

        
        /// <summary>
        ///   現在のドキュメントに一致するウィンドウのハンドルを取得する。
        /// </summary>
        ///
        /// <param name="i_viewCookie">
        ///   現在のドキュメントに一致する TsViewCookie データタイプを指定する。
        /// </param>
        ///
        /// <param name="o_hwnd">
        ///   現在のドキュメントに一致するウィンドウのハンドルへのポインタを受け取る。
        ///   一致するウィンドウがなければ NULL にできる。
        /// </param>
        ///
        /// <remark>
        ///   ドキュメントはメモリにあるが、スクリーンに表示されていない場合や、ウィ
        ///   ンドウがないコントロールや、ウィンドウがないコントロールのオーナーのウィ
        ///   ンドウハンドルを認識しない場合、ドキュメントは一致するウィンドウハンド
        ///   ルをもてない。呼び出し元は メソッドが成功したとしても o_hwnd パラメー
        ///   タに非 NULL 値を受け取ると想定してはいけない。
        /// </remark>
        public void GetWnd(int i_viewCookie, out IntPtr o_hwnd)
        {
            #if TSF_DEBUG_OUTPUT
            using(var dbgout = new DebugOut("{0}()", MethodBase.GetCurrentMethod().Name) )
            #endif
            {
                if( GetHWnd != null )
                    o_hwnd = GetHWnd();
                else
                    o_hwnd = IntPtr.Zero;
            }
        }

        #endregion


        
        //========================================================================================
        //  TSF 関連のラッパメソッド
        //========================================================================================
        #region TSF 関連のラッパメソッド
        /// <summary>
        /// スレッドマネージャの生成
        /// </summary>
        /// 
        /// <exception cref="COMException">
        /// スレッドマネージャーの生成に失敗した場合。
        /// </exception>
        protected void CreateThreadMgr()
        {
            if( _threadMgr == null )
            {
                TextFrameworkFunctions.TF_GetThreadMgr(out _threadMgr);
                if( _threadMgr == null )
                {
                    Type clsid = Type.GetTypeFromCLSID(TfDeclarations.CLSID_TF_ThreadMgr);
                    _threadMgr = Activator.CreateInstance(clsid) as ITfThreadMgr;

                    if( _threadMgr == null )
                    {
                        const string message = "スレッドマネージャーの生成に失敗しました。";
                        Debug.WriteLine(message);
                        throw new COMException(message, HRESULT.E_NOTIMPL);
                    }
                }
            }
        }


        //====================================================================


        /// <summary>
        /// スレッドマネージャーの解放。
        /// </summary>
        protected void DestroyThreadMgr()
        {
            if( _threadMgr != null )
            {
                try { _threadMgr.Deactivate(); } catch(Exception) {}

                ReleaseComObject("_threadMgr", ref _threadMgr);
            }
        }


        //====================================================================


        /// <summary>
        /// カテゴリマネージャーの生成。
        /// </summary>
        /// 
        /// <exception cref="COMException">
        /// カテゴリマネージャーの生成に失敗した場合。
        /// </exception>
        protected void CreateCategoryMgr()
        {
            if( _categoryMgr == null )
            {
                var clsid = Type.GetTypeFromCLSID(TfDeclarations.CLSID_TF_CategoryMgr);
                _categoryMgr = Activator.CreateInstance(clsid) as ITfCategoryMgr;

                if( _categoryMgr == null )
                {
                    const string message = "カテゴリマネージャーの生成に失敗しました。";
                    Debug.WriteLine(message);
                    throw new COMException(message, HRESULT.E_NOTIMPL);
                }
            }
        }


        //====================================================================


        /// <summary>
        /// カテゴリマネージャーの解放。
        /// </summary>
        protected void DestroyCategoryMgr()
        {
            ReleaseComObject("_categoryMgr", ref _categoryMgr);
        }

        
        //====================================================================


        /// <summary>
        /// 表示属性マネージャーの生成。
        /// </summary>
        /// 
        /// <exception cref="COMException">
        /// 表示属性マネージャーの生成に失敗した場合。
        /// </exception>
        protected void CreateDisplayAttributeMgr()
        {
            if( _displayAttributeMgr == null )
            {
                var clsid = Type.GetTypeFromCLSID(TfDeclarations.CLSID_TF_DisplayAttributeMgr);
                _displayAttributeMgr = Activator.CreateInstance(clsid) as ITfDisplayAttributeMgr;

                if( _displayAttributeMgr == null )
                {
                    const string message = "表示属性マネージャーの生成に失敗しました。";
                    Debug.WriteLine(message);
                    throw new COMException(message, HRESULT.E_NOTIMPL);
                }
            }
        }


        //====================================================================


        /// <summary>
        /// 表示属性マネージャーの解放。
        /// </summary>
        protected void DestroyDisplayAttributeMgr()
        {
            ReleaseComObject("_displayAttributeMgr", ref _displayAttributeMgr);
        }

        
        //====================================================================


        /// <summary>
        /// ドキュメントマネージャーの解放
        /// </summary>
        protected void DestroyDocumentMgr()
        {
            if( _documentMgr != null )
            {
                try { _documentMgr.Pop(PopFlags.TF_POPF_ALL); } catch(Exception) {}

                ReleaseComObject("_documentMgr", ref _documentMgr);
            }
        }

        
        //====================================================================


        /// <summary>
        /// COM オブジェクトのリリースとデバッグメッセージ出力。
        /// </summary>
        protected static void ReleaseComObject<ComObject>(
            string          i_objectName,
            ref ComObject   io_comObject
        )
        {
            if( io_comObject != null )
            {
                var refCount = Marshal.ReleaseComObject(io_comObject);
                #if TSF_DEBUG_OUTPUT
                    Debug.Print(
                        "Marshal.ReleaseComObject({0}) returns {1}.",
                        i_objectName,
                        refCount
                    );
                #endif

                io_comObject = default(ComObject);
            }
        }
        #endregion

        
        //========================================================================================
        //  ITfContextOwnerCompositionSink インターフェイスの実装
        //========================================================================================
        #region "ITfContextOwnerCompositionSink インターフェイスの実装"
        /// <summary>コンポジション入力が開始された時の処理。</summary>
        public void OnStartComposition(ITfCompositionView view, out bool ok)
        {
            #if TSF_DEBUG_OUTPUT
            using(var dbgout = new DebugOut("{0}()", MethodBase.GetCurrentMethod().Name) )
            #endif
            {
                if( CompositionStarted != null )
                    ok = CompositionStarted();
                else
                    ok = true;
            }
        }


        //========================================================================================


        /// <summary>コンポジションが変更された時の処理。</summary>
        public void OnUpdateComposition(ITfCompositionView view, ITfRange rangeNew)
        {
            #if TSF_DEBUG_OUTPUT
            using(var dbgout = new DebugOut("{0}()", MethodBase.GetCurrentMethod().Name) )
            #endif
            {
                if (rangeNew == null)
                    return;

                int start, count;
                
                ITfRangeACP rangeAcp = (ITfRangeACP)rangeNew;
                
                rangeAcp.GetExtent(out start, out count);
                
                if (this.CompositionUpdated != null)
                    this.CompositionUpdated(start, start + count);
            }
        }

        
        //========================================================================================


        /// <summary>コンポジション入力が終了した時の処理。</summary>
        public void OnEndComposition(ITfCompositionView view)
        {
            #if TSF_DEBUG_OUTPUT
            using(var dbgout = new DebugOut("{0}()", MethodBase.GetCurrentMethod().Name) )
            #endif
            {
                if( CompositionEnded != null )
                    CompositionEnded();
            }
        }
        #endregion "ITfContextOwnerCompositionSink インターフェイスの実装"


        //========================================================================================
        //  フィールド
        //========================================================================================
        protected ITfCategoryMgr            _categoryMgr;
        protected ITfDisplayAttributeMgr    _displayAttributeMgr;
        protected ITfThreadMgr              _threadMgr;
        protected ITfDocumentMgr            _documentMgr;
        protected ITfContext                _context;
        protected ITfFunctionProvider       _functionProvider;
        protected uint                      _editCookie = 0;
        protected ITextStoreACPSink         _sink;
        protected ITextStoreACPServices     _services;
        protected ITfFnReconversion         _reconversion;
        protected AdviseFlags               _adviseFlags = 0;
        protected LockFlags                 _lockFlags = 0;
        protected bool                      _pendingLockUpgrade = false;

        /// <summary>
        /// AttributeInfo　で使用されるフラグ。各属性の状態を示す。
        /// </summary>
        protected enum AttributeInfoFlags
        {
            /// <summary>何もない。</summary>
            None        = 0,
            /// <summary>デフォルト値の要求。</summary>
            Default     = 1 << 0,
            /// <summary>要求された。</summary>
            Requested   = 1 << 1
        }

        protected struct AttributeInfo
        {
            public Guid                 attrID;
            public AttributeInfoFlags   flags;
            public VARIANT              currentValue;
            public VARIANT              defaultValue;
        }

        protected AttributeInfo[]       _attributeInfo = new AttributeInfo[1];
    }
}
