require 'temppath/version'
require 'uuidtools'
require 'pathname'
require 'tmpdir'
require 'fileutils'

# Temppath creates temporary path. The differences from standard tempfile.rb are
# that this library generates Pathname objects with no files and filenames are
# based on UUID. Files in paths generated by this are deleted when Ruby exits.
#
# @example Create a temporary path
#   path = Temppath.create
#   #=> #<Pathname:/tmp/ruby-temppath-20130407-5775-w5k77l/f41bd6c5-fc99-4b7a-8f68-95b7ae4a6b22>
#   path.exist? #=> false
#   path.open("w")
#   "%o" % path.stat.mode #=> "100600" (default permission 0600)
# @example Touch a temporary file
#   path = Temppath.touch
#   path.exist? #=> true
#   path.file?  #=> true
#   "%o" % path.stat.mode #=> "100600"
# @example Create a temporary directory
#   path = Temppath.mkdir
#   path.exist?     #=> true
#   path.directory? #=> true
#   "%o" % path.stat.mode #=> "40700"
# @example Use temporary path generator
#   temppath = Temppath::Generator.new("/tmp/other-dir")
#   temppath.create
#   temppath.mkdir
#   temppath.touch
module Temppath
  # OriginalMethodHolder keeps some original methods of Pathname.
  module OriginalMethodHolder
    # @api private
    def self.extended(obj)
      # keep original methods
      obj.instance_exec do
        class << self
          alias :orig_open :open
          alias :orig_mkdir :mkdir
          alias :orig_mkpath :mkpath
          alias :orig_sysopen :sysopen
        end
      end
    end
  end

  # SecurePermissionMethods provides some Pathname methods with secure default
  # permission.
  module SecurePermissionMethods
    # Open file with permission 0600 by default. Otherwise, this is same as
    # Pathname#open.
    def open(*args, &b)
      args[1] = 0600 unless args[1]
      orig_open(*args, &b)
    end

    # Make a directory with permission 0700 by defautl. Otherwise, this is same
    # as Pathname#mkdir.
    def mkdir(*args)
      args[0] = 0700 unless args[0]
      orig_mkdir(*args)
    end

    # Make paths with permission 0700 by defautl. Otherwise, this is same as
    # Pathname#mkpath.
    def mkpath(*args)
      args[0] = {} unless args[0]
      args[0] = args[0].merge(mode: 0700) unless args[0][:mode]
      FileUtils.mkpath(self.to_s, *args)
    end

    # Open file with permission 0600 by default. Otherwise, this is same as
    # Pathname#sysopen.
    def sysopen(*args)
      args[1] = 0600 unless args[1]
      orig_sysopen(*args)
    end
  end

  # Generator generates temporary path in the base directory.
  class Generator
    # @return [Pathname]
    #   defalut base directory for paths created by Temppath
    attr_reader :basedir

    # @return [String]
    #   defalut base name
    attr_accessor :basename

    # @return [Boolean]
    #   true if unlink mode is enabled
    attr_accessor :unlink

    # @param basedir [Pathname]
    #   generator's base directory
    # @param option [Hash]
    # @option option [String] :basename
    #   prefix of filename
    def initialize(basedir, option={})
      @basedir = Pathname.new(basedir)
      @basename = (option[:basename] || "").to_s
      @unlink = true

      # extend basedir with secure methods
      @basedir.extend OriginalMethodHolder
      @basedir.extend SecurePermissionMethods

      # register a cleaner for temporary directory
      Kernel.at_exit do
        if @unlink
          remove_basedir rescue Errno::ENOENT
        end
      end
    end

    # Create a temporary path. This method creates no files.
    #
    # @param option [Hash]
    # @option option [String] :basename
    #   prefix of filename
    # @option option [Pathname] :basedir
    #   pathname of base directory
    def create(option={})
      _basename = option[:basename] || @basename
      _basedir = @basedir
      if option[:basedir]
        _basedir = Pathname.new(option[:basedir])

        # extend basedir with secure methods
        _basedir.extend OriginalMethodHolder
        _basedir.extend SecurePermissionMethods
      end

      # init basedir
      unless _basedir.exist?
        _basedir.mkpath
      end

      # make a path
      path = Pathname.new(_basedir) + (_basename.to_s + generate_uuid)

      # extend path object with secure methods
      path.extend OriginalMethodHolder
      path.extend SecurePermissionMethods

      # register a file cleaner if the path is not in basedir
      if _basedir != @basedir
        Kernel.at_exit do
          if @unlink
            FileUtils.remove_entry_secure(path) rescue Errno::ENOENT
          end
        end
      end

      return path
    end

    # Create a temporary directory.
    #
    # @param option [Hash]
    # @option option [Integer] :mode
    #   mode for the directory permission
    # @option option [String] :basename
    #   prefix of directory name
    # @option option [Pathname] :basedir
    #   pathname of base directory
    def mkdir(option={})
      mode = option[:mode] || 0700
      path = create(option)
      path.mkdir(mode)
      return path
    end

    # Create a empty file.
    #
    # @param option [Hash]
    # @option option [Integer] :mode
    #   mode for the file permission
    # @option option [String] :basename
    #   prefix of filename
    # @option option [Pathname] :basedir
    #   pathname of base directory
    def touch(option={})
      mode = option[:mode] || 0600
      path = create(option)
      path.open("w", mode)
      return path
    end

    # Remove current temporary directory.
    #
    # @return [void]
    def remove_basedir
      FileUtils.remove_entry_secure(@basedir) if @basedir.exist?
    end

    private

    # Generate random UUID for filename of temporary path.
    #
    # @return [String]
    #   UUID string
    def generate_uuid
      UUIDTools::UUID.random_create.to_s
    end
  end

  class << self
    # Create a new base directory.
    #
    # @return [Pathname]
    #   base directory
    def create_basedir
      Pathname.new(Dir.mktmpdir("ruby-temppath-"))
    end
    private :create_basedir
  end

  # default generator
  @generator = Generator.new(create_basedir)

  class << self
    # Return base directory of paths created by Temppath.
    #
    # @return [Pathname]
    #   base directory
    def basedir
      @generator.basedir
    end

    # Return base name of paths created by Temppath.
    def basename
      @generator.basename
    end

    # Set the base name.
    def basename=(name)
      @generator.basename = name
    end

    # Return true if unlink mode is enabled.
    #
    # @return [Boolean]
    #   true if unlink mode is enabled
    def unlink
      @generator.unlink
    end

    # Set true or false for unlink mode.
    #
    # @param b [Boolean]
    #   unlink mode
    def unlink=(b)
      @generator.unlink = b
    end

    # Create a temporary path. This method creates no files.
    #
    # @param option [Hash]
    # @option option [String] :basename
    #   prefix of filename
    # @option option [Pathname] :basedir
    #   pathname of base directory
    def create(option={})
      @generator.create(option)
    end

    # Create a temporary directory.
    #
    # @param option [Hash]
    # @option option [Integer] :mode
    #   mode for the directory permission
    # @option option [String] :basename
    #   prefix of directory name
    # @option option [Pathname] :basedir
    #   pathname of base directory
    def mkdir(option={})
      @generator.mkdir(option)
    end

    # Create a empty file.
    def touch(option={})
      @generator.touch(option)
    end

    # Remove current base directory and change to use a new base directory.
    #
    # @param basedir [Pathname]
    #   new base directory, or nil
    # @return [Pathname]
    #   new base directory
    def update_basedir(basedir=nil)
      @generator.remove_basedir
      _basedir = basedir || create_basedir
      @generator = Generator.new(_basedir)
      return _basedir
    end

    # Remove current temporary directory.
    #
    # @return [void]
    def remove_basedir
      @generator.remove_basedir
    end
  end
end
