/**
*** Copyright 2005-2007 Intel Corporation.  All Rights Reserved.
***
*** The source code contained or described herein and all documents related
*** to the source code ("Material") are owned by Intel Corporation or its
*** suppliers or licensors.  Title to the Material remains with Intel
*** Corporation or its suppliers and licensors.  The Material is protected
*** by worldwide copyright laws and treaty provisions.  No part of the
*** Material may be used, copied, reproduced, modified, published, uploaded,
*** posted, transmitted, distributed, or disclosed in any way without
*** Intel's prior express written permission.
***
*** No license under any patent, copyright, trade secret or other
*** intellectual property right is granted to or conferred upon you by
*** disclosure or delivery of the Materials, either expressly, by
*** implication, inducement, estoppel or otherwise.  Any license under such
*** intellectual property rights must be express and approved by Intel in
*** writing.
***
**/

/*
 * Provide the infrastructure needed to make the use of Cluster OpenMP in C++ easier.
 *
 * Specifically we provide 
 *
 * 1) an overloaded version of operator new so that one can write
 *     Foo * p = new kmp_sharable Foo;
 *
 * 2) an allocator class (kmp_sharable_allocator) which can be used with STL containers to 
 *    ensure that their data is allocated in shared space.
 *
 * 3) a class (kmp_sharable_base) which can be used as a base class to ensure that allocation 
 *    of all class objects derived from it occurs in sharable space.
 */

#ifndef __KMP_SHARABLE_H
#define __KMP_SHARABLE_H

#if !defined(__cplusplus)
# error "kmp_sharable.h is a C++ header. It cannot be used in C code."
#endif

#include <new>
#include <memory>

#if (_CLUSTER_OPENMP)
#include <omp.h>

struct __kmp_sharable_t
{
};

/* This object is never actually allocated, all we're interested in is
 * ensuring that the correct overloaded function is called, and this
 * provides a convenient argument to force that.
 */
extern const __kmp_sharable_t __attribute__ ((weak)) __kmp_sharable;

/* This is a rather unpleasant hack. Instead of having the user write
 *    int * p = new (kmp_sharable) int;
 * we have them write
 *    int * p = new kmp_sharable int;
 * that way in the non-CLOMP version we can macro kmp_sharable to nothing
 * and get 
 *    int * p = new int;
 * as against 
 *    int * p = new() int;
 * which is a syntax error.
 */
#define kmp_sharable (__kmp_sharable)

/* Overloaded operators to allocate sharable space. (cf the nothrow versions). */
extern void * operator new   (std::size_t, const __kmp_sharable_t &) throw (std::bad_alloc);         
extern void * operator new[] (std::size_t, const __kmp_sharable_t &) throw (std::bad_alloc);         
extern void   operator delete  (void *, const __kmp_sharable_t &) throw ();
extern void   operator delete[](void *, const __kmp_sharable_t &) throw ();

/* A template class which provides the interfaces of an allocator, and so can be used with STL
 * containers to force their contents into sharable store. Note, though, that STL containers
 * do not use the allocator to allocate the container class itself, so you must do something like
 *
 * std::vector <int, kmp_sharable_allocator<int>> * siv = new kmp_sharable std::vector<int, kmp_sharable_allocator<int>>;
 */

template <class _T> 
class kmp_sharable_allocator 
{
 public:
    typedef std::size_t     size_type;
    typedef std::ptrdiff_t  difference_type;

    typedef _T         value_type;
    typedef _T *       pointer;
    typedef _T const * const_pointer;
    typedef _T &       reference;
    typedef _T const & const_reference;

    template <class _T1> struct rebind
    { 
	typedef kmp_sharable_allocator<_T1> other; 
    };
    
    kmp_sharable_allocator() throw() {}
    kmp_sharable_allocator( const kmp_sharable_allocator&) throw() {}
    template <class _T1>
	kmp_sharable_allocator(const kmp_sharable_allocator<_T1>&) throw() {}
    ~kmp_sharable_allocator() throw() {}

    pointer       address(reference _r)       const { return &_r; }
    const_pointer address(const_reference _r) const { return &_r; }

    pointer allocate (size_type _count, const_pointer = 0)
    {
	void * space = kmp_sharable_malloc(_count * sizeof (value_type));

	if (space == 0)
	    throw std::bad_alloc();

	return static_cast<pointer> (space);
    }

    void deallocate (pointer _p, size_type)
    {
	kmp_sharable_free (static_cast<void *>(_p));
    }

    void construct (pointer _p, const_reference _v) { new(_p) value_type (_v); }
    void destroy   (pointer _p) { _p->~value_type(); }
    
    /* It's not easy to find this limit, so we just say something suitably large */
    size_type max_size() const throw() { return static_cast<size_type>(-1)/sizeof (value_type); }
};

/* Specialisation for kmp_sharable_allocator<void> */
template<> 
class kmp_sharable_allocator<void>
{
    typedef void        value_type;
    typedef void*       pointer;
    typedef const void* const_pointer;

    template <class _T1> struct rebind 
    { 
	typedef kmp_sharable_allocator<_T1> other; 
    };
};

template <class _T> 
bool operator== (const kmp_sharable_allocator<_T>&, const kmp_sharable_allocator<_T>&) throw()
{
    return true;
}

template <class _T> 
bool operator!= (const kmp_sharable_allocator<_T>&, const kmp_sharable_allocator<_T>&) throw()
{
    return false;
}

/* The base class for mixing in to force sharable allocation of class objects. */
class kmp_sharable_base
{
 public:
    static void * operator new  (std::size_t size) throw (std::bad_alloc)         
    { 
	void * result = kmp_sharable_malloc (size); 

	if (!result)
	    throw std::bad_alloc();

	return result;
    }
    static void operator delete (void * block, std::size_t) throw () 
    { 
	kmp_sharable_free (block); 
    }

    static void * operator new  (std::size_t size, const std::nothrow_t &) throw () 
    { 
	return kmp_sharable_malloc (size); 
    }        

    static void operator delete (void * block, const std::nothrow_t &) throw () 
    { 
	kmp_sharable_free (block); 
    }        

    static void * operator new[]  (std::size_t size) throw (std::bad_alloc)         
    { 
	void * result = kmp_sharable_malloc (size); 

	if (!result)
	    throw std::bad_alloc();

	return result;
    }         
    static void operator delete[] (void * block, std::size_t) throw() 
    { 
	kmp_sharable_free (block); 
    }

    static void * operator new[]  (std::size_t size, const std::nothrow_t &) throw () 
    { 
	return kmp_sharable_malloc (size); 
    }        

    static void operator delete[] (void * block, const std::nothrow_t &) throw () 
    { 
	kmp_sharable_free (block); 
    }        

    virtual ~kmp_sharable_base() {}
};

#else

/* Not being compiled with Cluster OpenMP, so just make these disappear. */
#define kmp_sharable
#define kmp_sharable_allocator std::allocator

/* Empty base class so that it has no effect when not compiling for Cluster OpenMP */
class kmp_sharable_base
{
};

#endif /* _CLUSTER_OPENMP */

#endif /* __KMP_SHARABLE_H */
