/*=========================================================================

  Program:   Visualization Toolkit
  Module:    $RCSfile: vtkHexahedron.h,v $
  Language:  C++
  Date:      $Date: 2002/12/26 18:24:21 $
  Version:   $Revision: 1.65 $

  Copyright (c) 1993-2002 Ken Martin, Will Schroeder, Bill Lorensen 
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even 
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
// .NAME vtkHexahedron - a cell that represents a linear 3D hexahedron
// .SECTION Description
// vtkHexahedron is a concrete implementation of vtkCell to represent a
// linear, 3D rectangular hexahedron (e.g., "brick" topology). vtkHexahedron
// uses the standard isoparametric shape functions for a linear
// hexahedron. The hexahedron is defined by the eight points (0-7) where
// (0,1,2,3) is the base of the hexahedron which, using the right hand rule,
// forms a quadrilaterial whose normal points in the direction of the 
// opposite face (4,5,6,7).

#ifndef __vtkHexahedron_h
#define __vtkHexahedron_h

#include "vtkCell3D.h"

class vtkLine;
class vtkQuad;

class VTK_COMMON_EXPORT vtkHexahedron : public vtkCell3D
{
public:
  static vtkHexahedron *New();
  vtkTypeRevisionMacro(vtkHexahedron,vtkCell3D);

  // Description:
  // See vtkCell3D API for description of these methods.
  virtual void GetEdgePoints(int edgeId, int* &pts);
  virtual void GetFacePoints(int faceId, int* &pts);
  virtual float *GetParametricCoords();

  // Description:
  // See the vtkCell API for descriptions of these methods.
  int GetCellType() {return VTK_HEXAHEDRON;}
  int GetNumberOfEdges() {return 12;}
  int GetNumberOfFaces() {return 6;}
  vtkCell *GetEdge(int edgeId);
  vtkCell *GetFace(int faceId);
  int CellBoundary(int subId, float pcoords[3], vtkIdList *pts);
  void Contour(float value, vtkDataArray *cellScalars, 
               vtkPointLocator *locator, vtkCellArray *verts, 
               vtkCellArray *lines, vtkCellArray *polys,
               vtkPointData *inPd, vtkPointData *outPd,
               vtkCellData *inCd, vtkIdType cellId, vtkCellData *outCd);

  int EvaluatePosition(float x[3], float* closestPoint,
                       int& subId, float pcoords[3],
                       float& dist2, float *weights);
  void EvaluateLocation(int& subId, float pcoords[3], float x[3],
                        float *weights);
  int IntersectWithLine(float p1[3], float p2[3], float tol, float& t,
                        float x[3], float pcoords[3], int& subId);
  int Triangulate(int index, vtkIdList *ptIds, vtkPoints *pts);
  void Derivatives(int subId, float pcoords[3], float *values, 
                   int dim, float *derivs);

  // Description:
  // Hexahedron specific
  static void InterpolationFunctions(float pcoords[3], float weights[8]);
  static void InterpolationDerivs(float pcoords[3], float derivs[24]);
  static int *GetEdgeArray(int edgeId);
  static int *GetFaceArray(int faceId);

  // Description:
  // Given parametric coordinates compute inverse Jacobian transformation
  // matrix. Returns 9 elements of 3x3 inverse Jacobian plus interpolation
  // function derivatives.
  void JacobianInverse(float pcoords[3], double **inverse, float derivs[24]);

protected:
  vtkHexahedron();
  ~vtkHexahedron();

  vtkLine *Line;
  vtkQuad *Quad;

private:
  vtkHexahedron(const vtkHexahedron&);  // Not implemented.
  void operator=(const vtkHexahedron&);  // Not implemented.
};

#endif


