﻿using System;
using System.IO;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Windows;
using System.Windows.Shapes;
using System.Windows.Controls;
using System.Windows.Data;
using System.Windows.Documents;
using System.Windows.Input;
using System.Windows.Media;
using System.Windows.Media.Imaging;
using System.Windows.Navigation;
using utility;

namespace dxf2phun
{
    /// <summary>
    /// MainWindow.xaml の相互作用ロジック
    /// </summary>
    public partial class MainWindow : Window
    {
        public MainWindow()
        {
            InitializeComponent();
        }

        private void TextBox_Drop(object sender, DragEventArgs e)
        {
            try
            {
                string[] aryStr = (string[])e.Data.GetData(DataFormats.FileDrop);
                
                //ガード節
                if (aryStr.Count() != 1)
                {
                    MessageBox.Show("ファイルは一つのみドロップできます。");
                    return;
                }
                if (System.IO.Path.GetExtension(aryStr[0])!=".dxf")
                {
                    MessageBox.Show("ファイル形式はdxfファイルのみ対応しています。");
                    return;
                }

                //処理
                Label label = (Label)sender;
                label.Content = aryStr[0];
            }
            catch (Exception ex)
            {
                MessageBox.Show(ErrorHandler.getErrStr(ex, System.Reflection.MethodBase.GetCurrentMethod().Name));
            }
        }

        private void TextBox_DragEnter(object sender, DragEventArgs e)
        {
            try
            {
                if (e.Data.GetDataPresent(DataFormats.FileDrop) == true)
                {
                e.Effects = DragDropEffects.All;
                }
                else
                {
                    e.Effects = DragDropEffects.None;
                }
            }
            catch (Exception ex)
            {
                MessageBox.Show(ErrorHandler.getErrStr(ex, System.Reflection.MethodBase.GetCurrentMethod().Name));
            }
        }

        private void Button_Click(object sender, RoutedEventArgs e)
        {
            try
            {
                List<object> objList = ParseFile2Objects((string)labelInputFile.Content);
                objList = Arc2Lines(objList, Convert.ToDouble(textBoxArcIntervalDeg.Text));
                /*
                string str = "";
                foreach (Line line in listline)
                {
                    str += String.Format("{0},{1}\r\n", line.X1, line.Y1);
                   str += String.Format("{0},{1}\r\n", line.X2, line.Y2);
                }
                textBoxOut.Text = str;
                return;
                */
                objList = SortLines(objList, Convert.ToDouble(textBoxEps.Text));
                objList = AxisShift(objList);

                Button btn = (Button)sender;
                if (btn.Name == "btnConvertMM")
                {
                    objList = CalcMulti(objList , 0.001);
                }

                //オブジェクトからLine要素のみを取り出す
                List<Line> lineList = new List<Line>();
                foreach (object obj in objList)
                {
                    if (obj is Line)
                    {
                        lineList.Add((Line)obj);
                    }
                }

                //ポリゴンを作成
                textBoxOut.Clear();
                textBoxOut.Text += "Scene.addPolygon {\r\n";
                textBoxOut.Text += "    surfaces = [[";
                string[] strAry = new string[lineList.Count()];
                for (int i = 0; i < lineList.Count(); ++i)
                {
                    strAry[i] = String.Format("[{0:F7},{1:F7}]", lineList[i].X1, lineList[i].Y1);
                }
                textBoxOut.Text += String.Join(",", strAry); 
                textBoxOut.Text += "]];\r\n";
                textBoxOut.Text += "    geomID = 100;\r\n";
                textBoxOut.Text += "};\r\n";

                //ポイント要素はヒンジを生成
                foreach (object obj in objList)
                {
                    if (obj is Point)
                    {
                        Point point = (Point)obj;
                        textBoxOut.Text += "Scene.addHinge {\r\n";
                        textBoxOut.Text += "    geom1 = 0;\r\n";
                        textBoxOut.Text += "    geom0 = 100;\r\n";
                        textBoxOut.Text += String.Format("    geom0pos = [{0:F7}, {1:F7}];\r\n",point.X,point.Y);
                        textBoxOut.Text += "};\r\n"; 
                    }
                }

                Clipboard.SetDataObject(textBoxOut.Text);
            }
            catch (Exception ex)
            {
                MessageBox.Show(ErrorHandler.getErrStr(ex, System.Reflection.MethodBase.GetCurrentMethod().Name));
            }
        }

        //-----------------------------------------------------------------------------------------
        //内部ロジック
        //-----------------------------------------------------------------------------------------


        //-----------------------------------------------------------------------------------------
        private static List<object> ParseFile2Objects(string filename)
        {
            string str_source;
            using (StreamReader sr = new StreamReader(filename))
            {
                str_source = sr.ReadToEnd();
            }

            StringReader strReader = new StringReader(str_source);
            int lineCount = 0;
            List<object> objectList = new List<object>();
            //try
            //{
                //ENTITIESのみ処理する
                while (true)
                {
                    //ガード節
                    if (strReader.Peek() == -1)
                    {
                        throw new ApplicationException("dxfファイル内に「ENTITIES」が見つかりません。");
                    }
                    //処理
                    string line = strReader.ReadLine();
                    ++lineCount;
                    if (line.ToUpper() == "ENTITIES")
                    {
                        break;
                    }
                }


                object p_obj = null; //現在処理中のエンティティを指すオブジェクト
                string currentEntity = "";//現在処理中のエンティティを指す文字列
                while (true)
                {
                    //ガード節
                    if (strReader.Peek() == -1)
                    {
                        throw new ApplicationException("「ENTITIES」と対をなす「ENDSEC」が見つかりません。");
                    }
                    //処理
                    int groupCode = Convert.ToInt32(strReader.ReadLine().Trim());
                    string groupData = strReader.ReadLine().Trim();
                    ++lineCount;
                    ++lineCount;

                    //エンティティの処理
                    if (currentEntity == "LINE")
                    {
                        Line line = (Line)p_obj;
                        if (groupCode == 10)
                        {
                            line.X1 = Convert.ToDouble(groupData);
                        }
                        if (groupCode == 11)
                        {
                            line.X2 = Convert.ToDouble(groupData);
                        }
                        if (groupCode == 20)
                        {
                            line.Y1 = Convert.ToDouble(groupData);
                        }
                        if (groupCode == 21)
                        {
                            line.Y2 = Convert.ToDouble(groupData);
                        }
                    }
                    if (currentEntity == "ARC")
                    {
                        Arc arc = (Arc)p_obj;
                        if (groupCode == 10)
                        {
                            arc.CenterPointX = Convert.ToDouble(groupData);
                        }
                        if (groupCode == 20)
                        {
                            arc.CenterPointY = Convert.ToDouble(groupData);
                        }
                        if (groupCode == 40)
                        {
                            arc.Radius = Convert.ToDouble(groupData);
                        }
                        if (groupCode == 50)
                        {
                            arc.StartAngleDeg = Convert.ToDouble(groupData);
                        }
                        if (groupCode == 51)
                        {
                            arc.EndAngleDeg = Convert.ToDouble(groupData);
                        }
                    }
                    if (currentEntity == "POINT")
                    {
                        Point point = (Point)p_obj;
                        if (groupCode == 10)
                        {
                            point.X = Convert.ToDouble(groupData);
                        }
                        if (groupCode == 20)
                        {
                            point.Y = Convert.ToDouble(groupData);
                        }
                    }


                    //エンティティのタイプを表すグループデータ
                    if (groupCode == 0)
                    {
                        //続行中の処理の終了処理
                        if (currentEntity == "LINE")
                        {
                            objectList.Add(p_obj);
                        }
                        if (currentEntity == "ARC")
                        {
                            objectList.Add(p_obj);
                        }
                        if (currentEntity == "POINT")
                        {
                            objectList.Add(p_obj);
                        }

                        //新規処理の準備
                        bool handled = false;
                        if (groupData.ToUpper() == "LINE")
                        {
                            p_obj = new Line();
                            currentEntity = "LINE";
                            handled = true;
                        }
                        if (groupData.ToUpper() == "ARC")
                        {
                            p_obj = new Arc();
                            currentEntity = "ARC";
                            handled = true;
                        }
                        if (groupData.ToUpper() == "POINT")
                        {
                            p_obj = new Point();
                            currentEntity = "POINT";
                            handled = true;
                        }

                        //セクション終了
                        if (groupData.ToUpper() == "ENDSEC")
                        {
                            currentEntity = "ENDSEC";
                            handled = true;
                            break;
                        }
                        //解釈できないエンティティのタイプを表すグループデータ
                        if (!handled)
                        {
                            currentEntity = "";
                        }
                    }
                }//end while
            //}//end try
            //catch (Exception ex)
            //{
            //    string msg = String.Format("{0:d}行目でエラーが発生しました。\r\n{1:s}", lineCount, ex.Message);
            //    throw new ApplicationException(msg, ex);
            //}

            return objectList;
        }

        private static List<object> Arc2Lines(List<object> objList, double arcIntervalDeg)
        {
            List<object> retObjList = new List<object>();
            foreach (object obj in objList)
            {
                if (obj is Arc)
                {
                    Arc arc = (Arc)obj;

                    //円弧の終了角度が必ず開始角度より大きくなるようにする。
                    if (arc.StartAngleDeg > arc.EndAngleDeg)
                    {
                        arc.EndAngleDeg += 360;
                    }
                    double currentAngleDeg = arc.StartAngleDeg;
                    bool isEnd=false;
                    while (!isEnd)
                    {
                        Line line = new Line();
                        line.X1 = arc.Radius * Math.Cos(currentAngleDeg / 180 * Math.PI)+arc.CenterPointX;
                        line.Y1 = arc.Radius * Math.Sin(currentAngleDeg / 180 * Math.PI)+arc.CenterPointY;
                        currentAngleDeg += arcIntervalDeg;
                        if (arc.EndAngleDeg < currentAngleDeg)
                        {
                            currentAngleDeg = arc.EndAngleDeg;
                            isEnd = true;
                        }
                        line.X2 = arc.Radius * Math.Cos(currentAngleDeg / 180 * Math.PI) + arc.CenterPointX;
                        line.Y2 = arc.Radius * Math.Sin(currentAngleDeg / 180 * Math.PI) + arc.CenterPointY;
                        retObjList.Add(line);
                    }
                }else{
                    retObjList.Add(obj);
                }
            }
            return retObjList;
        }

        //-----------------------------------------------------------------------------------------
        private static List<object> SortLines(List<object> objList, double eps)
        {
            //線とその他のオブジェクトに分離する。
            List<object> retList = new List<object>();
            List<Line> lineList = new List<Line>();
            foreach (object obj in objList)
            {
                if (obj is Line)
                {
                    lineList.Add((Line)obj);
                }
                else
                {
                    retList.Add(obj);
                }
            }

            //線のオブジェクトをソートする
            List<Line> retLineList = new List<Line>();
            retLineList.Add(lineList[0]);
            lineList.RemoveAt(0);

            int count = lineList.Count();
            for (int i = 0; i < count; ++i)
            {
                bool isFound=false;
                foreach (Line line in lineList)
                {
                    double diffX = Math.Abs(line.X1 - retLineList[i].X2);
                    double diffY = Math.Abs(line.Y1 - retLineList[i].Y2);
                    if ((diffX < eps) && (diffY < eps))
                    {
                        retLineList.Add(line);
                        lineList.Remove(line);
                        isFound = true;
                        break;
                    }
                }
                if (!isFound)
                {
                    foreach (Line line in lineList)
                    {
                        double diffX = Math.Abs(line.X2 - retLineList[i].X2);
                        double diffY = Math.Abs(line.Y2 - retLineList[i].Y2);
                        if ((diffX < eps) && (diffY < eps))
                        {
                            double tempX = line.X1;
                            double tempY = line.Y1;
                            line.X1 = line.X2;
                            line.Y1 = line.Y2;
                            line.X2 = tempX;
                            line.Y2 = tempY;
                            retLineList.Add(line);
                            lineList.Remove(line);
                            isFound = true;
                            break;
                        }
                    }
                }
                if (!isFound)
                {
                    throw new ApplicationException("線が閉じていません。");
                }
            }

            //ソートされたかチェックする。
            double diffXc = Math.Abs(retLineList[0].X1 - retLineList[retLineList.Count() - 1].X2);
            double diffYc = Math.Abs(retLineList[0].Y1 - retLineList[retLineList.Count() - 1].Y2);
            bool isCollect = (diffXc < eps) && (diffYc<eps);
            if(!isCollect)
            {
                throw new ApplicationException("線が閉じていません。");
            }

            //線のオブジェクトをその他のオブジェクトと一緒にする。
            foreach (Line line in retLineList)
            {
                retList.Add(line);
            }

            return retList;
        }
        //-----------------------------------------------------------------------------------------
        //※ソートされたリストに対し有効
        private static List<object> AxisShift(List<object> objList)
        {
            double minX = double.MaxValue;
            double minY = double.MaxValue;

            foreach (object obj in objList)
            {
                if (obj is Line)
                {
                    Line line = (Line)obj;
                    minX = (line.X1 < minX) ? line.X1 : minX;
                    minY = (line.Y1 < minY) ? line.Y1 : minY;
                }
                else if(obj is Arc)
                {
                    Arc arc = (Arc)obj;
                    minX = (arc.CenterPointX < minX) ? arc.CenterPointX : minX;
                    minY = (arc.CenterPointY < minY) ? arc.CenterPointY : minY;
                }
                else if (obj is Point)
                {
                    Point point = (Point)obj;
                    minX = (point.X < minX) ? point.X :minX;
                    minY = (point.Y < minY) ? point.Y :minY;
                }
            }

            foreach (object obj in objList)
            {
                if (obj is Line)
                {
                    Line line = (Line)obj;
                    line.X1 -= minX;
                    line.X2 -= minX;
                    line.Y1 -= minY;
                    line.Y2 -= minY;
                }
                else if (obj is Arc)
                {
                    Arc arc = (Arc)obj;
                    arc.CenterPointX -= minX;
                    arc.CenterPointY -= minY;
                }
                else if (obj is Point)
                {
                    Point point = (Point)obj;
                    point.X -= minX;
                    point.Y -= minY;
                }
            }
            return objList;
        }
 
        //-----------------------------------------------------------------------------------------
        //※ソートされたリストに対し有効
        private static List<object> CalcMulti(List<object> objList, double multi)
        {
            foreach (object obj in objList)
            {
                if (obj is Line)
                {
                    Line line = (Line)obj;
                    line.X1 *= multi;
                    line.X2 *= multi;
                    line.Y1 *= multi;
                    line.Y2 *= multi;
                }
                else if (obj is Arc)
                {
                    Arc arc = (Arc)obj;
                    arc.CenterPointX *= multi;
                    arc.CenterPointY *= multi;
                    arc.Radius *= multi;
                }
                else if (obj is Point)
                {
                    Point point = (Point)obj;
                    point.X *= multi;
                    point.Y *= multi;
                }
            }
            return objList;
        }
    }
}
