

#include "Collation.h"

#include "frameworks/base/cmds/statsd/src/atoms.pb.h"

#include <set>
#include <vector>

#include <getopt.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

using namespace google::protobuf;
using namespace std;

namespace android {
namespace stats_log_api_gen {

const int PULL_ATOM_START_ID = 1000;

int maxPushedAtomId = 2;

using android::os::statsd::Atom;

/**
 * Turn lower and camel case into upper case with underscores.
 */
static string
make_constant_name(const string& str)
{
    string result;
    const int N = str.size();
    bool underscore_next = false;
    for (int i=0; i<N; i++) {
        char c = str[i];
        if (c >= 'A' && c <= 'Z') {
            if (underscore_next) {
                result += '_';
                underscore_next = false;
            }
        } else if (c >= 'a' && c <= 'z') {
            c = 'A' + c - 'a';
            underscore_next = true;
        } else if (c == '_') {
            underscore_next = false;
        }
        result += c;
    }
    return result;
}

static const char*
cpp_type_name(java_type_t type)
{
    switch (type) {
        case JAVA_TYPE_BOOLEAN:
            return "bool";
        case JAVA_TYPE_INT:
        case JAVA_TYPE_ENUM:
            return "int32_t";
        case JAVA_TYPE_LONG:
            return "int64_t";
        case JAVA_TYPE_FLOAT:
            return "float";
        case JAVA_TYPE_DOUBLE:
            return "double";
        case JAVA_TYPE_STRING:
            return "char const*";
        default:
            return "UNKNOWN";
    }
}

static const char*
java_type_name(java_type_t type)
{
    switch (type) {
        case JAVA_TYPE_BOOLEAN:
            return "boolean";
        case JAVA_TYPE_INT:
        case JAVA_TYPE_ENUM:
            return "int";
        case JAVA_TYPE_LONG:
            return "long";
        case JAVA_TYPE_FLOAT:
            return "float";
        case JAVA_TYPE_DOUBLE:
            return "double";
        case JAVA_TYPE_STRING:
            return "java.lang.String";
        default:
            return "UNKNOWN";
    }
}

static int write_stats_log_cpp(FILE *out, const Atoms &atoms,
                               const AtomDecl &attributionDecl) {
    // Print prelude
    fprintf(out, "// This file is autogenerated\n");
    fprintf(out, "\n");

    fprintf(out, "#include <mutex>\n");
    fprintf(out, "#include <chrono>\n");
    fprintf(out, "#include <thread>\n");
    fprintf(out, "#include <cutils/properties.h>\n");
    fprintf(out, "#include <stats_event_list.h>\n");
    fprintf(out, "#include <log/log.h>\n");
    fprintf(out, "#include <statslog.h>\n");
    fprintf(out, "#include <utils/SystemClock.h>\n");
    fprintf(out, "\n");

    fprintf(out, "namespace android {\n");
    fprintf(out, "namespace util {\n");
    fprintf(out, "// the single event tag id for all stats logs\n");
    fprintf(out, "const static int kStatsEventTag = 1937006964;\n");
    fprintf(out, "const static bool kStatsdEnabled = property_get_bool(\"ro.statsd.enable\", true);\n");

    std::set<string> kTruncatingAtomNames = {"mobile_radio_power_state_changed",
                                             "audio_state_changed",
                                             "call_state_changed",
                                             "phone_signal_strength_changed",
                                             "mobile_bytes_transfer_by_fg_bg",
                                             "mobile_bytes_transfer"};
    fprintf(out,
            "const std::set<int> "
            "AtomsInfo::kNotTruncatingTimestampAtomWhiteList = {\n");
    for (set<AtomDecl>::const_iterator atom = atoms.decls.begin();
         atom != atoms.decls.end(); atom++) {
        if (kTruncatingAtomNames.find(atom->name) ==
            kTruncatingAtomNames.end()) {
            string constant = make_constant_name(atom->name);
            fprintf(out, " %s,\n", constant.c_str());
        }
    }
    fprintf(out, "};\n");
    fprintf(out, "\n");

    fprintf(out,
            "const std::set<int> AtomsInfo::kAtomsWithAttributionChain = {\n");
    for (set<AtomDecl>::const_iterator atom = atoms.decls.begin();
         atom != atoms.decls.end(); atom++) {
        for (vector<AtomField>::const_iterator field = atom->fields.begin();
             field != atom->fields.end(); field++) {
            if (field->javaType == JAVA_TYPE_ATTRIBUTION_CHAIN) {
                string constant = make_constant_name(atom->name);
                fprintf(out, " %s,\n", constant.c_str());
                break;
            }
        }
    }
    fprintf(out, "};\n");
    fprintf(out, "\n");

    fprintf(out, "static std::map<int, int> getAtomUidField() {\n");
    fprintf(out, "  std::map<int, int> uidField;\n");
    for (set<AtomDecl>::const_iterator atom = atoms.decls.begin();
         atom != atoms.decls.end(); atom++) {
        if (atom->uidField == 0) {
            continue;
        }
        fprintf(out,
                "\n    // Adding uid field for atom "
                "(%d)%s\n",
                atom->code, atom->name.c_str());
        fprintf(out, "    uidField[static_cast<int>(%s)] = %d;\n",
                make_constant_name(atom->name).c_str(), atom->uidField);
    }

    fprintf(out, "    return uidField;\n");
    fprintf(out, "};\n");

    fprintf(out,
            "const std::map<int, int> AtomsInfo::kAtomsWithUidField = "
            "getAtomUidField();\n");

    fprintf(out,
            "static std::map<int, StateAtomFieldOptions> "
            "getStateAtomFieldOptions() {\n");
    fprintf(out, "    std::map<int, StateAtomFieldOptions> options;\n");
    fprintf(out, "    StateAtomFieldOptions opt;\n");
    for (set<AtomDecl>::const_iterator atom = atoms.decls.begin();
         atom != atoms.decls.end(); atom++) {
        if (atom->primaryFields.size() == 0 && atom->exclusiveField == 0) {
            continue;
        }
        fprintf(out,
                "\n    // Adding primary and exclusive fields for atom "
                "(%d)%s\n",
                atom->code, atom->name.c_str());
        fprintf(out, "    opt.primaryFields.clear();\n");
        for (const auto& field : atom->primaryFields) {
            fprintf(out, "    opt.primaryFields.push_back(%d);\n", field);
        }

        fprintf(out, "    opt.exclusiveField = %d;\n", atom->exclusiveField);
        fprintf(out, "    options[static_cast<int>(%s)] = opt;\n",
                make_constant_name(atom->name).c_str());
    }

    fprintf(out, "    return options;\n");
    fprintf(out, "  }\n");

    fprintf(out,
            "const std::map<int, StateAtomFieldOptions> "
            "AtomsInfo::kStateAtomsFieldOptions = "
            "getStateAtomFieldOptions();\n");


    fprintf(out, "int64_t lastRetryTimestampNs = -1;\n");
    fprintf(out, "const int64_t kMinRetryIntervalNs = NS_PER_SEC * 60 * 20; // 20 minutes\n");
    fprintf(out, "static std::mutex mLogdRetryMutex;\n");

    // Print write methods
    fprintf(out, "\n");
    for (set<vector<java_type_t>>::const_iterator signature = atoms.signatures.begin();
        signature != atoms.signatures.end(); signature++) {
        int argIndex;

        fprintf(out, "int\n");
        fprintf(out, "try_stats_write(int32_t code");
        argIndex = 1;
        for (vector<java_type_t>::const_iterator arg = signature->begin();
            arg != signature->end(); arg++) {
            if (*arg == JAVA_TYPE_ATTRIBUTION_CHAIN) {
                for (auto chainField : attributionDecl.fields) {
                    if (chainField.javaType == JAVA_TYPE_STRING) {
                            fprintf(out, ", const std::vector<%s>& %s",
                                 cpp_type_name(chainField.javaType),
                                 chainField.name.c_str());
                    } else {
                            fprintf(out, ", const %s* %s, size_t %s_length",
                                 cpp_type_name(chainField.javaType),
                                 chainField.name.c_str(), chainField.name.c_str());
                    }
                }
            } else {
                fprintf(out, ", %s arg%d", cpp_type_name(*arg), argIndex);
            }
            argIndex++;
        }
        fprintf(out, ")\n");

        fprintf(out, "{\n");
        argIndex = 1;
        fprintf(out, "  if (kStatsdEnabled) {\n");
        fprintf(out, "    stats_event_list event(kStatsEventTag);\n");
        fprintf(out, "    event << android::elapsedRealtimeNano();\n\n");
        fprintf(out, "    event << code;\n\n");
        for (vector<java_type_t>::const_iterator arg = signature->begin();
            arg != signature->end(); arg++) {
            if (*arg == JAVA_TYPE_ATTRIBUTION_CHAIN) {
                for (const auto &chainField : attributionDecl.fields) {
                    if (chainField.javaType == JAVA_TYPE_STRING) {
                        fprintf(out, "    if (%s_length != %s.size()) {\n",
                            attributionDecl.fields.front().name.c_str(), chainField.name.c_str());
                        fprintf(out, "        return -EINVAL;\n");
                        fprintf(out, "    }\n");
                    }
                }
                fprintf(out, "\n    event.begin();\n");
                fprintf(out, "    for (size_t i = 0; i < %s_length; ++i) {\n",
                    attributionDecl.fields.front().name.c_str());
                fprintf(out, "        event.begin();\n");
                for (const auto &chainField : attributionDecl.fields) {
                    if (chainField.javaType == JAVA_TYPE_STRING) {
                        fprintf(out, "        if (%s[i] != NULL) {\n", chainField.name.c_str());
                        fprintf(out, "           event << %s[i];\n", chainField.name.c_str());
                        fprintf(out, "        } else {\n");
                        fprintf(out, "           event << \"\";\n");
                        fprintf(out, "        }\n");
                    } else {
                        fprintf(out, "        event << %s[i];\n", chainField.name.c_str());
                    }
                }
                fprintf(out, "        event.end();\n");
                fprintf(out, "    }\n");
                fprintf(out, "    event.end();\n\n");
            } else {
                if (*arg == JAVA_TYPE_STRING) {
                    fprintf(out, "    if (arg%d == NULL) {\n", argIndex);
                    fprintf(out, "        arg%d = \"\";\n", argIndex);
                    fprintf(out, "    }\n");
                }
                fprintf(out, "    event << arg%d;\n", argIndex);
            }
            argIndex++;
        }

        fprintf(out, "    return event.write(LOG_ID_STATS);\n");
        fprintf(out, "  } else {\n");
        fprintf(out, "    return 1;\n");
        fprintf(out, "  }\n");
        fprintf(out, "}\n");
        fprintf(out, "\n");
    }

   for (set<vector<java_type_t>>::const_iterator signature = atoms.signatures.begin();
       signature != atoms.signatures.end(); signature++) {
       int argIndex;

       fprintf(out, "int \n");
       fprintf(out, "stats_write(int32_t code");
       argIndex = 1;
       for (vector<java_type_t>::const_iterator arg = signature->begin();
           arg != signature->end(); arg++) {
           if (*arg == JAVA_TYPE_ATTRIBUTION_CHAIN) {
               for (auto chainField : attributionDecl.fields) {
                   if (chainField.javaType == JAVA_TYPE_STRING) {
                           fprintf(out, ", const std::vector<%s>& %s",
                                cpp_type_name(chainField.javaType),
                                chainField.name.c_str());
                   } else {
                           fprintf(out, ", const %s* %s, size_t %s_length",
                                cpp_type_name(chainField.javaType),
                                chainField.name.c_str(), chainField.name.c_str());
                   }
               }
           } else {
               fprintf(out, ", %s arg%d", cpp_type_name(*arg), argIndex);
           }
           argIndex++;
       }
       fprintf(out, ")\n");

       fprintf(out, "{\n");
       fprintf(out, "  int ret = 0;\n");

       fprintf(out, "  for(int retry = 0; retry < 2; ++retry) {\n");
       fprintf(out, "      ret =  try_stats_write(code");

       argIndex = 1;
       for (vector<java_type_t>::const_iterator arg = signature->begin();
           arg != signature->end(); arg++) {
           if (*arg == JAVA_TYPE_ATTRIBUTION_CHAIN) {
               for (auto chainField : attributionDecl.fields) {
                   if (chainField.javaType == JAVA_TYPE_STRING) {
                           fprintf(out, ", %s",
                                chainField.name.c_str());
                   } else {
                           fprintf(out, ",  %s,  %s_length",
                                chainField.name.c_str(), chainField.name.c_str());
                   }
               }
           } else {
               fprintf(out, ", arg%d", argIndex);
           }
           argIndex++;
       }
       fprintf(out, ");\n");
       fprintf(out, "      if (ret >= 0) { return retry; }\n");


       fprintf(out, "      {\n");
       fprintf(out, "          std::lock_guard<std::mutex> lock(mLogdRetryMutex);\n");
       fprintf(out, "          if ((android::elapsedRealtimeNano() - lastRetryTimestampNs) <= "
                                "kMinRetryIntervalNs) break;\n");
       fprintf(out, "          lastRetryTimestampNs = android::elapsedRealtimeNano();\n");
       fprintf(out, "      }\n");
       fprintf(out, "      std::this_thread::sleep_for(std::chrono::milliseconds(10));\n");
       fprintf(out, "  }\n");
       fprintf(out, "  return ret;\n");
       fprintf(out, "}\n");
       fprintf(out, "\n");
   }

    for (set<vector<java_type_t>>::const_iterator signature = atoms.non_chained_signatures.begin();
        signature != atoms.non_chained_signatures.end(); signature++) {
        int argIndex;

        fprintf(out, "int\n");
        fprintf(out, "try_stats_write_non_chained(int32_t code");
        argIndex = 1;
        for (vector<java_type_t>::const_iterator arg = signature->begin();
            arg != signature->end(); arg++) {
            fprintf(out, ", %s arg%d", cpp_type_name(*arg), argIndex);
            argIndex++;
        }
        fprintf(out, ")\n");

        fprintf(out, "{\n");
        argIndex = 1;
        fprintf(out, "  if (kStatsdEnabled) {\n");
        fprintf(out, "    stats_event_list event(kStatsEventTag);\n");
        fprintf(out, "    event << android::elapsedRealtimeNano();\n\n");
        fprintf(out, "    event << code;\n\n");
        for (vector<java_type_t>::const_iterator arg = signature->begin();
            arg != signature->end(); arg++) {
            if (argIndex == 1) {
                fprintf(out, "    event.begin();\n\n");
                fprintf(out, "    event.begin();\n");
            }
            if (*arg == JAVA_TYPE_STRING) {
                fprintf(out, "    if (arg%d == NULL) {\n", argIndex);
                fprintf(out, "        arg%d = \"\";\n", argIndex);
                fprintf(out, "    }\n");
            }
            fprintf(out, "    event << arg%d;\n", argIndex);
            if (argIndex == 2) {
                fprintf(out, "    event.end();\n\n");
                fprintf(out, "    event.end();\n\n");
            }
            argIndex++;
        }

        fprintf(out, "    return event.write(LOG_ID_STATS);\n");
        fprintf(out, "  } else {\n");
        fprintf(out, "    return 1;\n");
        fprintf(out, "  }\n");
        fprintf(out, "}\n");
        fprintf(out, "\n");
    }

   for (set<vector<java_type_t>>::const_iterator signature = atoms.non_chained_signatures.begin();
       signature != atoms.non_chained_signatures.end(); signature++) {
       int argIndex;

       fprintf(out, "int\n");
       fprintf(out, "stats_write_non_chained(int32_t code");
       argIndex = 1;
       for (vector<java_type_t>::const_iterator arg = signature->begin();
           arg != signature->end(); arg++) {
           fprintf(out, ", %s arg%d", cpp_type_name(*arg), argIndex);
           argIndex++;
       }
       fprintf(out, ")\n");

       fprintf(out, "{\n");

       fprintf(out, "  int ret = 0;\n");
       fprintf(out, "  for(int retry = 0; retry < 2; ++retry) {\n");
       fprintf(out, "      ret =  try_stats_write_non_chained(code");

       argIndex = 1;
       for (vector<java_type_t>::const_iterator arg = signature->begin();
           arg != signature->end(); arg++) {
           fprintf(out, ", arg%d",   argIndex);
           argIndex++;
       }
       fprintf(out, ");\n");
       fprintf(out, "      if (ret >= 0) { return retry; }\n");

       fprintf(out, "      {\n");
       fprintf(out, "          std::lock_guard<std::mutex> lock(mLogdRetryMutex);\n");
       fprintf(out, "          if ((android::elapsedRealtimeNano() - lastRetryTimestampNs) <= "
                                "kMinRetryIntervalNs) break;\n");
       fprintf(out, "          lastRetryTimestampNs = android::elapsedRealtimeNano();\n");
       fprintf(out, "      }\n");

       fprintf(out, "      std::this_thread::sleep_for(std::chrono::milliseconds(10));\n");
       fprintf(out, "  }\n");
       fprintf(out, "  return ret;\n");
       fprintf(out, "}\n");

       fprintf(out, "\n");
   }


    // Print footer
    fprintf(out, "\n");
    fprintf(out, "} // namespace util\n");
    fprintf(out, "} // namespace android\n");

    return 0;
}

void build_non_chained_decl_map(const Atoms& atoms,
                                std::map<int, set<AtomDecl>::const_iterator>* decl_map){
    for (set<AtomDecl>::const_iterator atom = atoms.non_chained_decls.begin();
        atom != atoms.non_chained_decls.end(); atom++) {
        decl_map->insert(std::make_pair(atom->code, atom));
    }
}

static void write_cpp_usage(
    FILE* out, const string& method_name, const string& atom_code_name,
    const AtomDecl& atom, const AtomDecl &attributionDecl) {
    fprintf(out, "     * Usage: %s(StatsLog.%s", method_name.c_str(), atom_code_name.c_str());
    for (vector<AtomField>::const_iterator field = atom.fields.begin();
            field != atom.fields.end(); field++) {
        if (field->javaType == JAVA_TYPE_ATTRIBUTION_CHAIN) {
            for (auto chainField : attributionDecl.fields) {
                if (chainField.javaType == JAVA_TYPE_STRING) {
                    fprintf(out, ", const std::vector<%s>& %s",
                         cpp_type_name(chainField.javaType),
                         chainField.name.c_str());
                } else {
                    fprintf(out, ", const %s* %s, size_t %s_length",
                         cpp_type_name(chainField.javaType),
                         chainField.name.c_str(), chainField.name.c_str());
                }
            }
        } else {
            fprintf(out, ", %s %s", cpp_type_name(field->javaType), field->name.c_str());
        }
    }
    fprintf(out, ");\n");
}

static void write_cpp_method_header(
    FILE* out, const string& method_name, const set<vector<java_type_t>>& signatures,
    const AtomDecl &attributionDecl) {
    for (set<vector<java_type_t>>::const_iterator signature = signatures.begin();
            signature != signatures.end(); signature++) {
        fprintf(out, "int %s(int32_t code ", method_name.c_str());
        int argIndex = 1;
        for (vector<java_type_t>::const_iterator arg = signature->begin();
            arg != signature->end(); arg++) {
            if (*arg == JAVA_TYPE_ATTRIBUTION_CHAIN) {
                for (auto chainField : attributionDecl.fields) {
                    if (chainField.javaType == JAVA_TYPE_STRING) {
                        fprintf(out, ", const std::vector<%s>& %s",
                            cpp_type_name(chainField.javaType), chainField.name.c_str());
                    } else {
                        fprintf(out, ", const %s* %s, size_t %s_length",
                            cpp_type_name(chainField.javaType),
                            chainField.name.c_str(), chainField.name.c_str());
                    }
                }
            } else {
                fprintf(out, ", %s arg%d", cpp_type_name(*arg), argIndex);
            }
            argIndex++;
        }
        fprintf(out, ");\n");

    }
}

static int
write_stats_log_header(FILE* out, const Atoms& atoms, const AtomDecl &attributionDecl)
{
    // Print prelude
    fprintf(out, "// This file is autogenerated\n");
    fprintf(out, "\n");
    fprintf(out, "#pragma once\n");
    fprintf(out, "\n");
    fprintf(out, "#include <stdint.h>\n");
    fprintf(out, "#include <vector>\n");
    fprintf(out, "#include <map>\n");
    fprintf(out, "#include <set>\n");
    fprintf(out, "\n");

    fprintf(out, "namespace android {\n");
    fprintf(out, "namespace util {\n");
    fprintf(out, "\n");
    fprintf(out, "/*\n");
    fprintf(out, " * API For logging statistics events.\n");
    fprintf(out, " */\n");
    fprintf(out, "\n");
    fprintf(out, "/**\n");
    fprintf(out, " * Constants for atom codes.\n");
    fprintf(out, " */\n");
    fprintf(out, "enum {\n");

    std::map<int, set<AtomDecl>::const_iterator> atom_code_to_non_chained_decl_map;
    build_non_chained_decl_map(atoms, &atom_code_to_non_chained_decl_map);

    size_t i = 0;
    // Print constants
    for (set<AtomDecl>::const_iterator atom = atoms.decls.begin();
        atom != atoms.decls.end(); atom++) {
        string constant = make_constant_name(atom->name);
        fprintf(out, "\n");
        fprintf(out, "    /**\n");
        fprintf(out, "     * %s %s\n", atom->message.c_str(), atom->name.c_str());
        write_cpp_usage(out, "stats_write", constant, *atom, attributionDecl);

        auto non_chained_decl = atom_code_to_non_chained_decl_map.find(atom->code);
        if (non_chained_decl != atom_code_to_non_chained_decl_map.end()) {
            write_cpp_usage(out, "stats_write_non_chained", constant, *non_chained_decl->second,
                attributionDecl);
        }
        fprintf(out, "     */\n");
        char const* const comma = (i == atoms.decls.size() - 1) ? "" : ",";
        fprintf(out, "    %s = %d%s\n", constant.c_str(), atom->code, comma);
        if (atom->code < PULL_ATOM_START_ID && atom->code > maxPushedAtomId) {
            maxPushedAtomId = atom->code;
        }
        i++;
    }
    fprintf(out, "\n");
    fprintf(out, "};\n");
    fprintf(out, "\n");

    fprintf(out, "struct StateAtomFieldOptions {\n");
    fprintf(out, "  std::vector<int> primaryFields;\n");
    fprintf(out, "  int exclusiveField;\n");
    fprintf(out, "};\n");
    fprintf(out, "\n");

    fprintf(out, "struct AtomsInfo {\n");
    fprintf(out,
            "  const static std::set<int> "
            "kNotTruncatingTimestampAtomWhiteList;\n");
    fprintf(out, "  const static std::map<int, int> kAtomsWithUidField;\n");
    fprintf(out,
            "  const static std::set<int> kAtomsWithAttributionChain;\n");
    fprintf(out,
            "  const static std::map<int, StateAtomFieldOptions> "
            "kStateAtomsFieldOptions;\n");
    fprintf(out, "};\n");

    fprintf(out, "const static int kMaxPushedAtomId = %d;\n\n",
            maxPushedAtomId);

    // Print write methods
    fprintf(out, "//\n");
    fprintf(out, "// Write methods\n");
    fprintf(out, "//\n");
    write_cpp_method_header(out, "stats_write", atoms.signatures, attributionDecl);

    fprintf(out, "//\n");
    fprintf(out, "// Write flattened methods\n");
    fprintf(out, "//\n");
    write_cpp_method_header(out, "stats_write_non_chained", atoms.non_chained_signatures,
        attributionDecl);

    fprintf(out, "\n");
    fprintf(out, "} // namespace util\n");
    fprintf(out, "} // namespace android\n");

    return 0;
}

static void write_java_usage(
    FILE* out, const string& method_name, const string& atom_code_name,
    const AtomDecl& atom, const AtomDecl &attributionDecl) {
    fprintf(out, "     * Usage: StatsLog.%s(StatsLog.%s",
        method_name.c_str(), atom_code_name.c_str());
    for (vector<AtomField>::const_iterator field = atom.fields.begin();
        field != atom.fields.end(); field++) {
        if (field->javaType == JAVA_TYPE_ATTRIBUTION_CHAIN) {
            for (auto chainField : attributionDecl.fields) {
                fprintf(out, ", %s[] %s",
                    java_type_name(chainField.javaType), chainField.name.c_str());
            }
        } else {
            fprintf(out, ", %s %s", java_type_name(field->javaType), field->name.c_str());
        }
    }
    fprintf(out, ");\n");
}

static void write_java_method(
    FILE* out, const string& method_name, const set<vector<java_type_t>>& signatures,
    const AtomDecl &attributionDecl) {
    for (set<vector<java_type_t>>::const_iterator signature = signatures.begin();
        signature != signatures.end(); signature++) {
        fprintf(out, "    public static native int %s(int code", method_name.c_str());
        int argIndex = 1;
        for (vector<java_type_t>::const_iterator arg = signature->begin();
            arg != signature->end(); arg++) {
            if (*arg == JAVA_TYPE_ATTRIBUTION_CHAIN) {
                for (auto chainField : attributionDecl.fields) {
                    fprintf(out, ", %s[] %s",
                        java_type_name(chainField.javaType), chainField.name.c_str());
                }
            } else {
                fprintf(out, ", %s arg%d", java_type_name(*arg), argIndex);
            }
            argIndex++;
        }
        fprintf(out, ");\n");
    }
}


static int
write_stats_log_java(FILE* out, const Atoms& atoms, const AtomDecl &attributionDecl)
{
    // Print prelude
    fprintf(out, "// This file is autogenerated\n");
    fprintf(out, "\n");
    fprintf(out, "package android.util;\n");
    fprintf(out, "\n");
    fprintf(out, "\n");
    fprintf(out, "/**\n");
    fprintf(out, " * API For logging statistics events.\n");
    fprintf(out, " * @hide\n");
    fprintf(out, " */\n");
    fprintf(out, "public class StatsLogInternal {\n");
    fprintf(out, "    // Constants for atom codes.\n");

    std::map<int, set<AtomDecl>::const_iterator> atom_code_to_non_chained_decl_map;
    build_non_chained_decl_map(atoms, &atom_code_to_non_chained_decl_map);

    // Print constants for the atom codes.
    for (set<AtomDecl>::const_iterator atom = atoms.decls.begin();
            atom != atoms.decls.end(); atom++) {
        string constant = make_constant_name(atom->name);
        fprintf(out, "\n");
        fprintf(out, "    /**\n");
        fprintf(out, "     * %s %s\n", atom->message.c_str(), atom->name.c_str());
        write_java_usage(out, "write", constant, *atom, attributionDecl);
        auto non_chained_decl = atom_code_to_non_chained_decl_map.find(atom->code);
        if (non_chained_decl != atom_code_to_non_chained_decl_map.end()) {
            write_java_usage(out, "write_non_chained", constant, *non_chained_decl->second,
             attributionDecl);
        }
        fprintf(out, "     */\n");
        fprintf(out, "    public static final int %s = %d;\n", constant.c_str(), atom->code);
    }
    fprintf(out, "\n");

    // Print constants for the enum values.
    fprintf(out, "    // Constants for enum values.\n\n");
    for (set<AtomDecl>::const_iterator atom = atoms.decls.begin();
        atom != atoms.decls.end(); atom++) {
        for (vector<AtomField>::const_iterator field = atom->fields.begin();
            field != atom->fields.end(); field++) {
            if (field->javaType == JAVA_TYPE_ENUM) {
                fprintf(out, "    // Values for %s.%s\n", atom->message.c_str(),
                    field->name.c_str());
                for (map<int, string>::const_iterator value = field->enumValues.begin();
                    value != field->enumValues.end(); value++) {
                    fprintf(out, "    public static final int %s__%s__%s = %d;\n",
                        make_constant_name(atom->message).c_str(),
                        make_constant_name(field->name).c_str(),
                        make_constant_name(value->second).c_str(),
                        value->first);
                }
                fprintf(out, "\n");
            }
        }
    }

    // Print write methods
    fprintf(out, "    // Write methods\n");
    write_java_method(out, "write", atoms.signatures, attributionDecl);
    write_java_method(out, "write_non_chained", atoms.non_chained_signatures, attributionDecl);

    fprintf(out, "}\n");

    return 0;
}

static const char*
jni_type_name(java_type_t type)
{
    switch (type) {
        case JAVA_TYPE_BOOLEAN:
            return "jboolean";
        case JAVA_TYPE_INT:
        case JAVA_TYPE_ENUM:
            return "jint";
        case JAVA_TYPE_LONG:
            return "jlong";
        case JAVA_TYPE_FLOAT:
            return "jfloat";
        case JAVA_TYPE_DOUBLE:
            return "jdouble";
        case JAVA_TYPE_STRING:
            return "jstring";
        default:
            return "UNKNOWN";
    }
}

static const char*
jni_array_type_name(java_type_t type)
{
    switch (type) {
        case JAVA_TYPE_INT:
            return "jintArray";
        case JAVA_TYPE_STRING:
            return "jobjectArray";
        default:
            return "UNKNOWN";
    }
}

static string
jni_function_name(const string& method_name, const vector<java_type_t>& signature)
{
    string result("StatsLog_" + method_name);
    for (vector<java_type_t>::const_iterator arg = signature.begin();
        arg != signature.end(); arg++) {
        switch (*arg) {
            case JAVA_TYPE_BOOLEAN:
                result += "_boolean";
                break;
            case JAVA_TYPE_INT:
            case JAVA_TYPE_ENUM:
                result += "_int";
                break;
            case JAVA_TYPE_LONG:
                result += "_long";
                break;
            case JAVA_TYPE_FLOAT:
                result += "_float";
                break;
            case JAVA_TYPE_DOUBLE:
                result += "_double";
                break;
            case JAVA_TYPE_STRING:
                result += "_String";
                break;
            case JAVA_TYPE_ATTRIBUTION_CHAIN:
              result += "_AttributionChain";
              break;
            default:
                result += "_UNKNOWN";
                break;
        }
    }
    return result;
}

static const char*
java_type_signature(java_type_t type)
{
    switch (type) {
        case JAVA_TYPE_BOOLEAN:
            return "Z";
        case JAVA_TYPE_INT:
        case JAVA_TYPE_ENUM:
            return "I";
        case JAVA_TYPE_LONG:
            return "J";
        case JAVA_TYPE_FLOAT:
            return "F";
        case JAVA_TYPE_DOUBLE:
            return "D";
        case JAVA_TYPE_STRING:
            return "Ljava/lang/String;";
        default:
            return "UNKNOWN";
    }
}

static string
jni_function_signature(const vector<java_type_t>& signature, const AtomDecl &attributionDecl)
{
    string result("(I");
    for (vector<java_type_t>::const_iterator arg = signature.begin();
        arg != signature.end(); arg++) {
        if (*arg == JAVA_TYPE_ATTRIBUTION_CHAIN) {
            for (auto chainField : attributionDecl.fields) {
                result += "[";
                result += java_type_signature(chainField.javaType);
            }
        } else {
            result += java_type_signature(*arg);
        }
    }
    result += ")I";
    return result;
}

static int
write_stats_log_jni(FILE* out, const string& java_method_name, const string& cpp_method_name,
    const set<vector<java_type_t>>& signatures, const AtomDecl &attributionDecl)
{
    // Print write methods
    for (set<vector<java_type_t>>::const_iterator signature = signatures.begin();
        signature != signatures.end(); signature++) {
        int argIndex;

        fprintf(out, "static int\n");
        fprintf(out, "%s(JNIEnv* env, jobject clazz UNUSED, jint code",
                jni_function_name(java_method_name, *signature).c_str());
        argIndex = 1;
        for (vector<java_type_t>::const_iterator arg = signature->begin();
                arg != signature->end(); arg++) {
            if (*arg == JAVA_TYPE_ATTRIBUTION_CHAIN) {
                for (auto chainField : attributionDecl.fields) {
                    fprintf(out, ", %s %s", jni_array_type_name(chainField.javaType),
                        chainField.name.c_str());
                }
            } else {
                fprintf(out, ", %s arg%d", jni_type_name(*arg), argIndex);
            }
            argIndex++;
        }
        fprintf(out, ")\n");

        fprintf(out, "{\n");

        // Prepare strings
        argIndex = 1;
        bool hadStringOrChain = false;
        for (vector<java_type_t>::const_iterator arg = signature->begin();
                arg != signature->end(); arg++) {
            if (*arg == JAVA_TYPE_STRING) {
                hadStringOrChain = true;
                fprintf(out, "    const char* str%d;\n", argIndex);
                fprintf(out, "    if (arg%d != NULL) {\n", argIndex);
                fprintf(out, "        str%d = env->GetStringUTFChars(arg%d, NULL);\n",
                        argIndex, argIndex);
                fprintf(out, "    } else {\n");
                fprintf(out, "        str%d = NULL;\n", argIndex);
                fprintf(out, "    }\n");
            } else if (*arg == JAVA_TYPE_ATTRIBUTION_CHAIN) {
                hadStringOrChain = true;
                for (auto chainField : attributionDecl.fields) {
                    fprintf(out, "    size_t %s_length = env->GetArrayLength(%s);\n",
                        chainField.name.c_str(), chainField.name.c_str());
                    if (chainField.name != attributionDecl.fields.front().name) {
                        fprintf(out, "    if (%s_length != %s_length) {\n",
                            chainField.name.c_str(),
                            attributionDecl.fields.front().name.c_str());
                        fprintf(out, "        return -EINVAL;\n");
                        fprintf(out, "    }\n");
                    }
                    if (chainField.javaType == JAVA_TYPE_INT) {
                        fprintf(out, "    jint* %s_array = env->GetIntArrayElements(%s, NULL);\n",
                            chainField.name.c_str(), chainField.name.c_str());
                    } else if (chainField.javaType == JAVA_TYPE_STRING) {
                        fprintf(out, "    std::vector<%s> %s_vec;\n",
                            cpp_type_name(chainField.javaType), chainField.name.c_str());
                        fprintf(out, "    std::vector<ScopedUtfChars*> scoped_%s_vec;\n",
                            chainField.name.c_str());
                        fprintf(out, "    for (size_t i = 0; i < %s_length; ++i) {\n",
                            chainField.name.c_str());
                        fprintf(out, "        jstring jstr = "
                            "(jstring)env->GetObjectArrayElement(%s, i);\n",
                             chainField.name.c_str());
                        fprintf(out, "        if (jstr == NULL) {\n");
                        fprintf(out, "            %s_vec.push_back(NULL);\n",
                            chainField.name.c_str());
                        fprintf(out, "        } else {\n");
                        fprintf(out, "            ScopedUtfChars* scoped_%s = "
                            "new ScopedUtfChars(env, jstr);\n",
                             chainField.name.c_str());
                        fprintf(out, "            %s_vec.push_back(scoped_%s->c_str());\n",
                                chainField.name.c_str(), chainField.name.c_str());
                        fprintf(out, "            scoped_%s_vec.push_back(scoped_%s);\n",
                                chainField.name.c_str(), chainField.name.c_str());
                        fprintf(out, "        }\n");
                        fprintf(out, "    }\n");
                    }
                    fprintf(out, "\n");
                }
            }
            argIndex++;
        }
        // Emit this to quiet the unused parameter warning if there were no strings or attribution
        // chains.
        if (!hadStringOrChain) {
            fprintf(out, "    (void)env;\n");
        }

        // stats_write call
        argIndex = 1;
        fprintf(out, "   int ret =  android::util::%s(code", cpp_method_name.c_str());
        for (vector<java_type_t>::const_iterator arg = signature->begin();
                arg != signature->end(); arg++) {
            if (*arg == JAVA_TYPE_ATTRIBUTION_CHAIN) {
                for (auto chainField : attributionDecl.fields) {
                    if (chainField.javaType == JAVA_TYPE_INT) {
                        fprintf(out, ", (const %s*)%s_array, %s_length",
                            cpp_type_name(chainField.javaType),
                            chainField.name.c_str(), chainField.name.c_str());
                    } else if (chainField.javaType == JAVA_TYPE_STRING) {
                        fprintf(out, ", %s_vec", chainField.name.c_str());
                    }
                }
            } else {
                const char *argName = (*arg == JAVA_TYPE_STRING) ? "str" : "arg";
                fprintf(out, ", (%s)%s%d", cpp_type_name(*arg), argName, argIndex);
            }
            argIndex++;
        }
        fprintf(out, ");\n");
        fprintf(out, "\n");

        // Clean up strings
        argIndex = 1;
        for (vector<java_type_t>::const_iterator arg = signature->begin();
                arg != signature->end(); arg++) {
            if (*arg == JAVA_TYPE_STRING) {
                fprintf(out, "    if (str%d != NULL) {\n", argIndex);
                fprintf(out, "        env->ReleaseStringUTFChars(arg%d, str%d);\n",
                        argIndex, argIndex);
                fprintf(out, "    }\n");
            } else if (*arg == JAVA_TYPE_ATTRIBUTION_CHAIN) {
                for (auto chainField : attributionDecl.fields) {
                    if (chainField.javaType == JAVA_TYPE_INT) {
                        fprintf(out, "    env->ReleaseIntArrayElements(%s, %s_array, 0);\n",
                            chainField.name.c_str(), chainField.name.c_str());
                    } else if (chainField.javaType == JAVA_TYPE_STRING) {
                        fprintf(out, "    for (size_t i = 0; i < scoped_%s_vec.size(); ++i) {\n",
                            chainField.name.c_str());
                        fprintf(out, "        delete scoped_%s_vec[i];\n", chainField.name.c_str());
                        fprintf(out, "    }\n");
                    }
                }
            }
            argIndex++;
        }
        fprintf(out, "    return ret;\n");

        fprintf(out, "}\n");
        fprintf(out, "\n");
    }


    return 0;
}

void write_jni_registration(FILE* out, const string& java_method_name,
    const set<vector<java_type_t>>& signatures, const AtomDecl &attributionDecl) {
    for (set<vector<java_type_t>>::const_iterator signature = signatures.begin();
            signature != signatures.end(); signature++) {
        fprintf(out, "    { \"%s\", \"%s\", (void*)%s },\n",
            java_method_name.c_str(),
            jni_function_signature(*signature, attributionDecl).c_str(),
            jni_function_name(java_method_name, *signature).c_str());
    }
}

static int
write_stats_log_jni(FILE* out, const Atoms& atoms, const AtomDecl &attributionDecl)
{
    // Print prelude
    fprintf(out, "// This file is autogenerated\n");
    fprintf(out, "\n");

    fprintf(out, "#include <statslog.h>\n");
    fprintf(out, "\n");
    fprintf(out, "#include <nativehelper/JNIHelp.h>\n");
    fprintf(out, "#include <nativehelper/ScopedUtfChars.h>\n");
    fprintf(out, "#include <utils/Vector.h>\n");
    fprintf(out, "#include \"core_jni_helpers.h\"\n");
    fprintf(out, "#include \"jni.h\"\n");
    fprintf(out, "\n");
    fprintf(out, "#define UNUSED  __attribute__((__unused__))\n");
    fprintf(out, "\n");

    fprintf(out, "namespace android {\n");
    fprintf(out, "\n");

    write_stats_log_jni(out, "write", "stats_write", atoms.signatures, attributionDecl);
    write_stats_log_jni(out, "write_non_chained", "stats_write_non_chained",
        atoms.non_chained_signatures, attributionDecl);

    // Print registration function table
    fprintf(out, "/*\n");
    fprintf(out, " * JNI registration.\n");
    fprintf(out, " */\n");
    fprintf(out, "static const JNINativeMethod gRegisterMethods[] = {\n");
    write_jni_registration(out, "write", atoms.signatures, attributionDecl);
    write_jni_registration(out, "write_non_chained", atoms.non_chained_signatures, attributionDecl);
    fprintf(out, "};\n");
    fprintf(out, "\n");

    // Print registration function
    fprintf(out, "int register_android_util_StatsLog(JNIEnv* env) {\n");
    fprintf(out, "    return RegisterMethodsOrDie(\n");
    fprintf(out, "            env,\n");
    fprintf(out, "            \"android/util/StatsLog\",\n");
    fprintf(out, "            gRegisterMethods, NELEM(gRegisterMethods));\n");
    fprintf(out, "}\n");

    fprintf(out, "\n");
    fprintf(out, "} // namespace android\n");
    return 0;
}

static void
print_usage()
{
    fprintf(stderr, "usage: stats-log-api-gen OPTIONS\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "OPTIONS\n");
    fprintf(stderr, "  --cpp FILENAME       the header file to output\n");
    fprintf(stderr, "  --header FILENAME    the cpp file to output\n");
    fprintf(stderr, "  --help               this message\n");
    fprintf(stderr, "  --java FILENAME      the java file to output\n");
    fprintf(stderr, "  --jni FILENAME       the jni file to output\n");
}

/**
 * Do the argument parsing and execute the tasks.
 */
static int
run(int argc, char const*const* argv)
{
    string cppFilename;
    string headerFilename;
    string javaFilename;
    string jniFilename;

    int index = 1;
    while (index < argc) {
        if (0 == strcmp("--help", argv[index])) {
            print_usage();
            return 0;
        } else if (0 == strcmp("--cpp", argv[index])) {
            index++;
            if (index >= argc) {
                print_usage();
                return 1;
            }
            cppFilename = argv[index];
        } else if (0 == strcmp("--header", argv[index])) {
            index++;
            if (index >= argc) {
                print_usage();
                return 1;
            }
            headerFilename = argv[index];
        } else if (0 == strcmp("--java", argv[index])) {
            index++;
            if (index >= argc) {
                print_usage();
                return 1;
            }
            javaFilename = argv[index];
        } else if (0 == strcmp("--jni", argv[index])) {
            index++;
            if (index >= argc) {
                print_usage();
                return 1;
            }
            jniFilename = argv[index];
        }
        index++;
    }

    if (cppFilename.size() == 0
            && headerFilename.size() == 0
            && javaFilename.size() == 0
            && jniFilename.size() == 0) {
        print_usage();
        return 1;
    }

    // Collate the parameters
    Atoms atoms;
    int errorCount = collate_atoms(Atom::descriptor(), &atoms);
    if (errorCount != 0) {
        return 1;
    }

    AtomDecl attributionDecl;
    vector<java_type_t> attributionSignature;
    collate_atom(android::os::statsd::AttributionNode::descriptor(),
                 &attributionDecl, &attributionSignature);

    // Write the .cpp file
    if (cppFilename.size() != 0) {
        FILE* out = fopen(cppFilename.c_str(), "w");
        if (out == NULL) {
            fprintf(stderr, "Unable to open file for write: %s\n", cppFilename.c_str());
            return 1;
        }
        errorCount = android::stats_log_api_gen::write_stats_log_cpp(
            out, atoms, attributionDecl);
        fclose(out);
    }

    // Write the .h file
    if (headerFilename.size() != 0) {
        FILE* out = fopen(headerFilename.c_str(), "w");
        if (out == NULL) {
            fprintf(stderr, "Unable to open file for write: %s\n", headerFilename.c_str());
            return 1;
        }
        errorCount = android::stats_log_api_gen::write_stats_log_header(
            out, atoms, attributionDecl);
        fclose(out);
    }

    // Write the .java file
    if (javaFilename.size() != 0) {
        FILE* out = fopen(javaFilename.c_str(), "w");
        if (out == NULL) {
            fprintf(stderr, "Unable to open file for write: %s\n", javaFilename.c_str());
            return 1;
        }
        errorCount = android::stats_log_api_gen::write_stats_log_java(
            out, atoms, attributionDecl);
        fclose(out);
    }

    // Write the jni file
    if (jniFilename.size() != 0) {
        FILE* out = fopen(jniFilename.c_str(), "w");
        if (out == NULL) {
            fprintf(stderr, "Unable to open file for write: %s\n", jniFilename.c_str());
            return 1;
        }
        errorCount = android::stats_log_api_gen::write_stats_log_jni(
            out, atoms, attributionDecl);
        fclose(out);
    }

    return 0;
}

}
}

/**
 * Main.
 */
int
main(int argc, char const*const* argv)
{
    GOOGLE_PROTOBUF_VERIFY_VERSION;

    return android::stats_log_api_gen::run(argc, argv);
}
