/*
 * Copyright (C) 2017 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.android.settingslib.accessibility;

import android.accessibilityservice.AccessibilityServiceInfo;
import android.content.Context;
import android.provider.Settings;
import android.view.accessibility.AccessibilityManager;

import java.util.List;

/**
 * A helper class to assist determining the state of the accessibility button that can appear
 * within the software-rendered navigation area.
 */
public class AccessibilityButtonHelper {
    public static boolean isRequestedByMagnification(Context ctx) {
        return Settings.Secure.getInt(ctx.getContentResolver(),
                Settings.Secure.ACCESSIBILITY_DISPLAY_MAGNIFICATION_NAVBAR_ENABLED, 0) == 1;
    }

    public static boolean isRequestedByAccessibilityService(Context ctx) {
        final AccessibilityManager accessibilityManager = ctx.getSystemService(
                AccessibilityManager.class);
        List<AccessibilityServiceInfo> services =
                accessibilityManager.getEnabledAccessibilityServiceList(
                        AccessibilityServiceInfo.FEEDBACK_ALL_MASK);
        if (services != null) {
            for (int i = 0, size = services.size(); i < size; i++) {
                if ((services.get(i).flags
                        & AccessibilityServiceInfo.FLAG_REQUEST_ACCESSIBILITY_BUTTON)
                        != 0) {
                    return true;
                }
            }
        }
        return false;
    }

    public static boolean isRequested(Context ctx) {
        return isRequestedByMagnification(ctx) || isRequestedByAccessibilityService(ctx);
    }
}
