/*
 * Copyright (C) 2017 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package android.net.util;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

import android.support.test.filters.SmallTest;
import android.support.test.runner.AndroidJUnit4;

import org.junit.Test;
import org.junit.runner.RunWith;

import java.io.ByteArrayOutputStream;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.Vector;

@RunWith(AndroidJUnit4.class)
@SmallTest
public class SharedLogTest {
    private static final String TIMESTAMP_PATTERN =
            "^[0-9][0-9]-[0-9][0-9] [0-9][0-9]:[0-9][0-9]:[0-9][0-9].[0-9][0-9][0-9]";
    private static final String TIMESTAMP = "mm-dd HH:MM:SS.xxx";

    @Test
    public void testBasicOperation() {
        final SharedLog logTop = new SharedLog("top");
        logTop.mark("first post!");

        final SharedLog logLevel2a = logTop.forSubComponent("twoA");
        final SharedLog logLevel2b = logTop.forSubComponent("twoB");
        logLevel2b.e("2b or not 2b");
        logLevel2a.w("second post?");

        final SharedLog logLevel3 = logLevel2a.forSubComponent("three");
        logTop.log("still logging");
        logLevel3.log("3 >> 2");
        logLevel2a.mark("ok: last post");

        final String[] expected = {
            TIMESTAMP + " - MARK first post!",
            TIMESTAMP + " - [twoB] ERROR 2b or not 2b",
            TIMESTAMP + " - [twoA] WARN second post?",
            TIMESTAMP + " - still logging",
            TIMESTAMP + " - [twoA.three] 3 >> 2",
            TIMESTAMP + " - [twoA] MARK ok: last post",
        };
        // Verify the logs are all there and in the correct order.
        verifyLogLines(expected, logTop);

        // In fact, because they all share the same underlying LocalLog,
        // every subcomponent SharedLog's dump() is identical.
        verifyLogLines(expected, logLevel2a);
        verifyLogLines(expected, logLevel2b);
        verifyLogLines(expected, logLevel3);
    }

    private static void verifyLogLines(String[] expected, SharedLog log) {
        final ByteArrayOutputStream ostream = new ByteArrayOutputStream();
        final PrintWriter pw = new PrintWriter(ostream, true);
        log.dump(null, pw, null);

        final String dumpOutput = ostream.toString();
        assertTrue(dumpOutput != null);
        assertTrue(!"".equals(dumpOutput));

        final String[] lines = dumpOutput.split("\n");
        assertEquals(expected.length, lines.length);

        for (int i = 0; i < lines.length; i++) {
            // Fix up the timestamps.
            lines[i] = lines[i].replaceAll(TIMESTAMP_PATTERN, TIMESTAMP);
        }

        for (int i = 0; i < expected.length; i++) {
            assertEquals(expected[i], lines[i]);
        }
    }
}
