/*
 * Mesa 3-D graphics library
 *
 * Copyright (C) 1999-2007  Brian Paul   All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 */

#include <stdbool.h>
#include "glheader.h"
#include "api_validate.h"
#include "arrayobj.h"
#include "bufferobj.h"
#include "context.h"
#include "imports.h"
#include "mtypes.h"
#include "pipelineobj.h"
#include "enums.h"
#include "state.h"
#include "transformfeedback.h"
#include "uniforms.h"
#include "vbo/vbo.h"
#include "program/prog_print.h"


static bool
check_blend_func_error(struct gl_context *ctx)
{
   /* The ARB_blend_func_extended spec's ERRORS section says:
    *
    *    "The error INVALID_OPERATION is generated by Begin or any procedure
    *     that implicitly calls Begin if any draw buffer has a blend function
    *     requiring the second color input (SRC1_COLOR, ONE_MINUS_SRC1_COLOR,
    *     SRC1_ALPHA or ONE_MINUS_SRC1_ALPHA), and a framebuffer is bound that
    *     has more than the value of MAX_DUAL_SOURCE_DRAW_BUFFERS-1 active
    *     color attachements."
    */
   for (unsigned i = ctx->Const.MaxDualSourceDrawBuffers;
	i < ctx->DrawBuffer->_NumColorDrawBuffers;
	i++) {
      if (ctx->Color.Blend[i]._UsesDualSrc) {
	 _mesa_error(ctx, GL_INVALID_OPERATION,
		     "dual source blend on illegal attachment");
	 return false;
      }
   }

   if (ctx->Color.BlendEnabled && ctx->Color._AdvancedBlendMode) {
      /* The KHR_blend_equation_advanced spec says:
       *
       *    "If any non-NONE draw buffer uses a blend equation found in table
       *     X.1 or X.2, the error INVALID_OPERATION is generated by Begin or
       *     any operation that implicitly calls Begin (such as DrawElements)
       *     if:
       *
       *       * the draw buffer for color output zero selects multiple color
       *         buffers (e.g., FRONT_AND_BACK in the default framebuffer); or
       *
       *       * the draw buffer for any other color output is not NONE."
       */
      if (ctx->DrawBuffer->ColorDrawBuffer[0] == GL_FRONT_AND_BACK) {
         _mesa_error(ctx, GL_INVALID_OPERATION,
                     "advanced blending is active and draw buffer for color "
                     "output zero selects multiple color buffers");
         return false;
      }

      for (unsigned i = 1; i < ctx->DrawBuffer->_NumColorDrawBuffers; i++) {
         if (ctx->DrawBuffer->ColorDrawBuffer[i] != GL_NONE) {
            _mesa_error(ctx, GL_INVALID_OPERATION,
                        "advanced blending is active with multiple color "
                        "draw buffers");
            return false;
         }
      }

      /* The KHR_blend_equation_advanced spec says:
       *
       *    "Advanced blending equations require the use of a fragment shader
       *     with a matching "blend_support" layout qualifier.  If the current
       *     blend equation is found in table X.1 or X.2, and the active
       *     fragment shader does not include the layout qualifier matching
       *     the blend equation or "blend_support_all_equations", the error
       *     INVALID_OPERATION is generated [...]"
       */
      const struct gl_program *prog = ctx->FragmentProgram._Current;
      const GLbitfield blend_support = !prog ? 0 : prog->sh.fs.BlendSupport;

      if ((blend_support & ctx->Color._AdvancedBlendMode) == 0) {
         _mesa_error(ctx, GL_INVALID_OPERATION,
                     "fragment shader does not allow advanced blending mode "
                     "(%s)",
                      _mesa_enum_to_string(ctx->Color.Blend[0].EquationRGB));
      }
   }

   return true;
}


/**
 * Prior to drawing anything with glBegin, glDrawArrays, etc. this function
 * is called to see if it's valid to render.  This involves checking that
 * the current shader is valid and the framebuffer is complete.
 * It also check the current pipeline object is valid if any.
 * If an error is detected it'll be recorded here.
 * \return GL_TRUE if OK to render, GL_FALSE if not
 */
GLboolean
_mesa_valid_to_render(struct gl_context *ctx, const char *where)
{
   /* This depends on having up to date derived state (shaders) */
   if (ctx->NewState)
      _mesa_update_state(ctx);

   if (ctx->API == API_OPENGL_COMPAT) {
      /* Any shader stages that are not supplied by the GLSL shader and have
       * assembly shaders enabled must now be validated.
       */
      if (!ctx->_Shader->CurrentProgram[MESA_SHADER_VERTEX] &&
          ctx->VertexProgram.Enabled &&
          !_mesa_arb_vertex_program_enabled(ctx)) {
         _mesa_error(ctx, GL_INVALID_OPERATION,
                     "%s(vertex program not valid)", where);
         return GL_FALSE;
      }

      if (!ctx->_Shader->CurrentProgram[MESA_SHADER_FRAGMENT]) {
         if (ctx->FragmentProgram.Enabled &&
             !_mesa_arb_fragment_program_enabled(ctx)) {
            _mesa_error(ctx, GL_INVALID_OPERATION,
                        "%s(fragment program not valid)", where);
            return GL_FALSE;
         }

         /* If drawing to integer-valued color buffers, there must be an
          * active fragment shader (GL_EXT_texture_integer).
          */
         if (ctx->DrawBuffer && ctx->DrawBuffer->_IntegerBuffers) {
            _mesa_error(ctx, GL_INVALID_OPERATION,
                        "%s(integer format but no fragment shader)", where);
            return GL_FALSE;
         }
      }
   }

   /* A pipeline object is bound */
   if (ctx->_Shader->Name && !ctx->_Shader->Validated) {
      if (!_mesa_validate_program_pipeline(ctx, ctx->_Shader)) {
         _mesa_error(ctx, GL_INVALID_OPERATION,
                     "glValidateProgramPipeline failed to validate the "
                     "pipeline");
         return GL_FALSE;
      }
   }

   /* If a program is active and SSO not in use, check if validation of
    * samplers succeeded for the active program. */
   if (ctx->_Shader->ActiveProgram && ctx->_Shader != ctx->Pipeline.Current) {
      char errMsg[100];
      if (!_mesa_sampler_uniforms_are_valid(ctx->_Shader->ActiveProgram,
                                            errMsg, 100)) {
         _mesa_error(ctx, GL_INVALID_OPERATION, "%s", errMsg);
         return GL_FALSE;
      }
   }

   if (ctx->DrawBuffer->_Status != GL_FRAMEBUFFER_COMPLETE_EXT) {
      _mesa_error(ctx, GL_INVALID_FRAMEBUFFER_OPERATION_EXT,
                  "%s(incomplete framebuffer)", where);
      return GL_FALSE;
   }

   if (!check_blend_func_error(ctx)) {
      return GL_FALSE;
   }

   /* From the GL_NV_fill_rectangle spec:
    *
    * "An INVALID_OPERATION error is generated by Begin or any Draw command if
    *  only one of the front and back polygon mode is FILL_RECTANGLE_NV."
    */
   if ((ctx->Polygon.FrontMode == GL_FILL_RECTANGLE_NV) !=
       (ctx->Polygon.BackMode == GL_FILL_RECTANGLE_NV)) {
      _mesa_error(ctx, GL_INVALID_OPERATION,
                  "GL_FILL_RECTANGLE_NV must be used as both front/back "
                  "polygon mode or neither");
      return GL_FALSE;
   }

#ifdef DEBUG
   if (ctx->_Shader->Flags & GLSL_LOG) {
      struct gl_program **prog = ctx->_Shader->CurrentProgram;

      for (unsigned i = 0; i < MESA_SHADER_STAGES; i++) {
	 if (prog[i] == NULL || prog[i]->_Used)
	    continue;

	 /* This is the first time this shader is being used.
	  * Append shader's constants/uniforms to log file.
	  *
	  * Only log data for the program target that matches the shader
	  * target.  It's possible to have a program bound to the vertex
	  * shader target that also supplied a fragment shader.  If that
	  * program isn't also bound to the fragment shader target we don't
	  * want to log its fragment data.
	  */
	 _mesa_append_uniforms_to_file(prog[i]);
      }

      for (unsigned i = 0; i < MESA_SHADER_STAGES; i++) {
	 if (prog[i] != NULL)
	    prog[i]->_Used = GL_TRUE;
      }
   }
#endif

   return GL_TRUE;
}


/**
 * Check if OK to draw arrays/elements.
 */
static bool
check_valid_to_render(struct gl_context *ctx, const char *function)
{
   if (!_mesa_valid_to_render(ctx, function)) {
      return false;
   }

   /* Section 6.3.2 from the GL 4.5:
    * "Any GL command which attempts to read from, write to, or change
    *  the state of a buffer object may generate an INVALID_OPERATION error if
    *  all or part of the buffer object is mapped ... However, only commands
    *  which explicitly describe this error are required to do so. If an error
    *  is not generated, such commands will have undefined results and may
    *  result in GL interruption or termination."
    *
    * Only some buffer API functions require INVALID_OPERATION with mapped
    * buffers. No other functions list such an error, thus it's not required
    * to report INVALID_OPERATION for draw calls with mapped buffers.
    */
   if (!ctx->Const.AllowMappedBuffersDuringExecution &&
       !_mesa_all_buffers_are_unmapped(ctx->Array.VAO)) {
      _mesa_error(ctx, GL_INVALID_OPERATION,
                  "%s(vertex buffers are mapped)", function);
      return false;
   }

   /* Section 11.2 (Tessellation) of the ES 3.2 spec says:
    *
    * "An INVALID_OPERATION error is generated by any command that
    *  transfers vertices to the GL if the current program state has
    *  one but not both of a tessellation control shader and tessellation
    *  evaluation shader."
    *
    * The OpenGL spec argues that this is allowed because a tess ctrl shader
    * without a tess eval shader can be used with transform feedback.
    * However, glBeginTransformFeedback doesn't allow GL_PATCHES and
    * therefore doesn't allow tessellation.
    *
    * Further investigation showed that this is indeed a spec bug and
    * a tess ctrl shader without a tess eval shader shouldn't have been
    * allowed, because there is no API in GL 4.0 that can make use this
    * to produce something useful.
    *
    * Also, all vendors except one don't support a tess ctrl shader without
    * a tess eval shader anyway.
    */
   if (ctx->TessCtrlProgram._Current && !ctx->TessEvalProgram._Current) {
      _mesa_error(ctx, GL_INVALID_OPERATION,
                  "%s(tess eval shader is missing)", function);
      return false;
   }

   switch (ctx->API) {
   case API_OPENGLES2:
      /* Section 11.2 (Tessellation) of the ES 3.2 spec says:
       *
       * "An INVALID_OPERATION error is generated by any command that
       *  transfers vertices to the GL if the current program state has
       *  one but not both of a tessellation control shader and tessellation
       *  evaluation shader."
       */
      if (_mesa_is_gles3(ctx) &&
          ctx->TessEvalProgram._Current && !ctx->TessCtrlProgram._Current) {
         _mesa_error(ctx, GL_INVALID_OPERATION,
                     "%s(tess ctrl shader is missing)", function);
         return false;
      }
      break;

   case API_OPENGL_CORE:
      /* Section 10.4 (Drawing Commands Using Vertex Arrays) of the OpenGL 4.5
       * Core Profile spec says:
       *
       *     "An INVALID_OPERATION error is generated if no vertex array
       *     object is bound (see section 10.3.1)."
       */
      if (ctx->Array.VAO == ctx->Array.DefaultVAO) {
         _mesa_error(ctx, GL_INVALID_OPERATION, "%s(no VAO bound)", function);
         return false;
      }
      break;

   case API_OPENGLES:
   case API_OPENGL_COMPAT:
      break;

   default:
      unreachable("Invalid API value in check_valid_to_render()");
   }

   return true;
}


/**
 * Is 'mode' a valid value for glBegin(), glDrawArrays(), glDrawElements(),
 * etc?  The set of legal values depends on whether geometry shaders/programs
 * are supported.
 * Note: This may be called during display list compilation.
 */
bool
_mesa_is_valid_prim_mode(const struct gl_context *ctx, GLenum mode)
{
   /* The overwhelmingly common case is (mode <= GL_TRIANGLE_FAN).  Test that
    * first and exit.  You would think that a switch-statement would be the
    * right approach, but at least GCC 4.7.2 generates some pretty dire code
    * for the common case.
    */
   if (likely(mode <= GL_TRIANGLE_FAN))
      return true;

   if (mode <= GL_POLYGON)
      return (ctx->API == API_OPENGL_COMPAT);

   if (mode <= GL_TRIANGLE_STRIP_ADJACENCY)
      return _mesa_has_geometry_shaders(ctx);

   if (mode == GL_PATCHES)
      return _mesa_has_tessellation(ctx);

   return false;
}


/**
 * Is 'mode' a valid value for glBegin(), glDrawArrays(), glDrawElements(),
 * etc?  Also, do additional checking related to transformation feedback.
 * Note: this function cannot be called during glNewList(GL_COMPILE) because
 * this code depends on current transform feedback state.
 * Also, do additional checking related to tessellation shaders.
 */
GLboolean
_mesa_valid_prim_mode(struct gl_context *ctx, GLenum mode, const char *name)
{
   bool valid_enum = _mesa_is_valid_prim_mode(ctx, mode);

   if (!valid_enum) {
      _mesa_error(ctx, GL_INVALID_ENUM, "%s(mode=%x)", name, mode);
      return GL_FALSE;
   }

   /* From the OpenGL 4.5 specification, section 11.3.1:
    *
    * The error INVALID_OPERATION is generated if Begin, or any command that
    * implicitly calls Begin, is called when a geometry shader is active and:
    *
    * * the input primitive type of the current geometry shader is
    *   POINTS and <mode> is not POINTS,
    *
    * * the input primitive type of the current geometry shader is
    *   LINES and <mode> is not LINES, LINE_STRIP, or LINE_LOOP,
    *
    * * the input primitive type of the current geometry shader is
    *   TRIANGLES and <mode> is not TRIANGLES, TRIANGLE_STRIP or
    *   TRIANGLE_FAN,
    *
    * * the input primitive type of the current geometry shader is
    *   LINES_ADJACENCY_ARB and <mode> is not LINES_ADJACENCY_ARB or
    *   LINE_STRIP_ADJACENCY_ARB, or
    *
    * * the input primitive type of the current geometry shader is
    *   TRIANGLES_ADJACENCY_ARB and <mode> is not
    *   TRIANGLES_ADJACENCY_ARB or TRIANGLE_STRIP_ADJACENCY_ARB.
    *
    * The GL spec doesn't mention any interaction with tessellation, which
    * is clearly a spec bug. The same rule should apply, but instead of
    * the draw primitive mode, the tessellation evaluation shader primitive
    * mode should be used for the checking.
   */
   if (ctx->_Shader->CurrentProgram[MESA_SHADER_GEOMETRY]) {
      const GLenum geom_mode =
         ctx->_Shader->CurrentProgram[MESA_SHADER_GEOMETRY]->
            info.gs.input_primitive;
      struct gl_program *tes =
         ctx->_Shader->CurrentProgram[MESA_SHADER_TESS_EVAL];
      GLenum mode_before_gs = mode;

      if (tes) {
         if (tes->info.tess.point_mode)
            mode_before_gs = GL_POINTS;
         else if (tes->info.tess.primitive_mode == GL_ISOLINES)
            mode_before_gs = GL_LINES;
         else
            /* the GL_QUADS mode generates triangles too */
            mode_before_gs = GL_TRIANGLES;
      }

      switch (mode_before_gs) {
      case GL_POINTS:
         valid_enum = (geom_mode == GL_POINTS);
         break;
      case GL_LINES:
      case GL_LINE_LOOP:
      case GL_LINE_STRIP:
         valid_enum = (geom_mode == GL_LINES);
         break;
      case GL_TRIANGLES:
      case GL_TRIANGLE_STRIP:
      case GL_TRIANGLE_FAN:
         valid_enum = (geom_mode == GL_TRIANGLES);
         break;
      case GL_QUADS:
      case GL_QUAD_STRIP:
      case GL_POLYGON:
         valid_enum = false;
         break;
      case GL_LINES_ADJACENCY:
      case GL_LINE_STRIP_ADJACENCY:
         valid_enum = (geom_mode == GL_LINES_ADJACENCY);
         break;
      case GL_TRIANGLES_ADJACENCY:
      case GL_TRIANGLE_STRIP_ADJACENCY:
         valid_enum = (geom_mode == GL_TRIANGLES_ADJACENCY);
         break;
      default:
         valid_enum = false;
         break;
      }
      if (!valid_enum) {
         _mesa_error(ctx, GL_INVALID_OPERATION,
                     "%s(mode=%s vs geometry shader input %s)",
                     name,
                     _mesa_lookup_prim_by_nr(mode_before_gs),
                     _mesa_lookup_prim_by_nr(geom_mode));
         return GL_FALSE;
      }
   }

   /* From the OpenGL 4.0 (Core Profile) spec (section 2.12):
    *
    *     "Tessellation operates only on patch primitives. If tessellation is
    *      active, any command that transfers vertices to the GL will
    *      generate an INVALID_OPERATION error if the primitive mode is not
    *      PATCHES.
    *      Patch primitives are not supported by pipeline stages below the
    *      tessellation evaluation shader. If there is no active program
    *      object or the active program object does not contain a tessellation
    *      evaluation shader, the error INVALID_OPERATION is generated by any
    *      command that transfers vertices to the GL if the primitive mode is
    *      PATCHES."
    *
    */
   if (ctx->_Shader->CurrentProgram[MESA_SHADER_TESS_EVAL] ||
       ctx->_Shader->CurrentProgram[MESA_SHADER_TESS_CTRL]) {
      if (mode != GL_PATCHES) {
         _mesa_error(ctx, GL_INVALID_OPERATION,
                     "only GL_PATCHES valid with tessellation");
         return GL_FALSE;
      }
   }
   else {
      if (mode == GL_PATCHES) {
         _mesa_error(ctx, GL_INVALID_OPERATION,
                     "GL_PATCHES only valid with tessellation");
         return GL_FALSE;
      }
   }

   /* From the GL_EXT_transform_feedback spec:
    *
    *     "The error INVALID_OPERATION is generated if Begin, or any command
    *      that performs an explicit Begin, is called when:
    *
    *      * a geometry shader is not active and <mode> does not match the
    *        allowed begin modes for the current transform feedback state as
    *        given by table X.1.
    *
    *      * a geometry shader is active and the output primitive type of the
    *        geometry shader does not match the allowed begin modes for the
    *        current transform feedback state as given by table X.1.
    *
    */
   if (_mesa_is_xfb_active_and_unpaused(ctx)) {
      GLboolean pass = GL_TRUE;

      if(ctx->_Shader->CurrentProgram[MESA_SHADER_GEOMETRY]) {
         switch (ctx->_Shader->CurrentProgram[MESA_SHADER_GEOMETRY]->
                    info.gs.output_primitive) {
         case GL_POINTS:
            pass = ctx->TransformFeedback.Mode == GL_POINTS;
            break;
         case GL_LINE_STRIP:
            pass = ctx->TransformFeedback.Mode == GL_LINES;
            break;
         case GL_TRIANGLE_STRIP:
            pass = ctx->TransformFeedback.Mode == GL_TRIANGLES;
            break;
         default:
            pass = GL_FALSE;
         }
      }
      else if (ctx->_Shader->CurrentProgram[MESA_SHADER_TESS_EVAL]) {
         struct gl_program *tes =
            ctx->_Shader->CurrentProgram[MESA_SHADER_TESS_EVAL];
         if (tes->info.tess.point_mode)
            pass = ctx->TransformFeedback.Mode == GL_POINTS;
         else if (tes->info.tess.primitive_mode == GL_ISOLINES)
            pass = ctx->TransformFeedback.Mode == GL_LINES;
         else
            pass = ctx->TransformFeedback.Mode == GL_TRIANGLES;
      }
      else {
         switch (mode) {
         case GL_POINTS:
            pass = ctx->TransformFeedback.Mode == GL_POINTS;
            break;
         case GL_LINES:
         case GL_LINE_STRIP:
         case GL_LINE_LOOP:
            pass = ctx->TransformFeedback.Mode == GL_LINES;
            break;
         default:
            pass = ctx->TransformFeedback.Mode == GL_TRIANGLES;
            break;
         }
      }
      if (!pass) {
         _mesa_error(ctx, GL_INVALID_OPERATION,
                         "%s(mode=%s vs transform feedback %s)",
                         name,
                         _mesa_lookup_prim_by_nr(mode),
                         _mesa_lookup_prim_by_nr(ctx->TransformFeedback.Mode));
         return GL_FALSE;
      }
   }

   /* From GL_INTEL_conservative_rasterization spec:
    *
    * The conservative rasterization option applies only to polygons with
    * PolygonMode state set to FILL. Draw requests for polygons with different
    * PolygonMode setting or for other primitive types (points/lines) generate
    * INVALID_OPERATION error.
    */
   if (ctx->IntelConservativeRasterization) {
      GLboolean pass = GL_TRUE;

      switch (mode) {
      case GL_POINTS:
      case GL_LINES:
      case GL_LINE_LOOP:
      case GL_LINE_STRIP:
      case GL_LINES_ADJACENCY:
      case GL_LINE_STRIP_ADJACENCY:
         pass = GL_FALSE;
         break;
      case GL_TRIANGLES:
      case GL_TRIANGLE_STRIP:
      case GL_TRIANGLE_FAN:
      case GL_QUADS:
      case GL_QUAD_STRIP:
      case GL_POLYGON:
      case GL_TRIANGLES_ADJACENCY:
      case GL_TRIANGLE_STRIP_ADJACENCY:
         if (ctx->Polygon.FrontMode != GL_FILL ||
             ctx->Polygon.BackMode != GL_FILL)
            pass = GL_FALSE;
         break;
      default:
         pass = GL_FALSE;
      }
      if (!pass) {
         _mesa_error(ctx, GL_INVALID_OPERATION,
                     "mode=%s invalid with GL_INTEL_conservative_rasterization",
                     _mesa_lookup_prim_by_nr(mode));
         return GL_FALSE;
      }
   }

   return GL_TRUE;
}

/**
 * Verify that the element type is valid.
 *
 * Generates \c GL_INVALID_ENUM and returns \c false if it is not.
 */
static bool
valid_elements_type(struct gl_context *ctx, GLenum type, const char *name)
{
   switch (type) {
   case GL_UNSIGNED_BYTE:
   case GL_UNSIGNED_SHORT:
   case GL_UNSIGNED_INT:
      return true;

   default:
      _mesa_error(ctx, GL_INVALID_ENUM, "%s(type = %s)", name,
                  _mesa_enum_to_string(type));
      return false;
   }
}

static bool
validate_DrawElements_common(struct gl_context *ctx,
                             GLenum mode, GLsizei count, GLenum type,
                             const GLvoid *indices,
                             const char *caller)
{
   /* Section 2.14.2 (Transform Feedback Primitive Capture) of the OpenGL ES
    * 3.1 spec says:
    *
    *   The error INVALID_OPERATION is also generated by DrawElements,
    *   DrawElementsInstanced, and DrawRangeElements while transform feedback
    *   is active and not paused, regardless of mode.
    *
    * The OES_geometry_shader_spec says:
    *
    *    Issues:
    *
    *    ...
    *
    *    (13) Does this extension change how transform feedback operates
    *    compared to unextended OpenGL ES 3.0 or 3.1?
    *
    *    RESOLVED: Yes... Since we no longer require being able to predict how
    *    much geometry will be generated, we also lift the restriction that
    *    only DrawArray* commands are supported and also support the
    *    DrawElements* commands for transform feedback.
    *
    * This should also be reflected in the body of the spec, but that appears
    * to have been overlooked.  The body of the spec only explicitly allows
    * the indirect versions.
    */
   if (_mesa_is_gles3(ctx) &&
       !_mesa_has_OES_geometry_shader(ctx) &&
       _mesa_is_xfb_active_and_unpaused(ctx)) {
      _mesa_error(ctx, GL_INVALID_OPERATION,
                  "%s(transform feedback active)", caller);
      return false;
   }

   if (count < 0) {
      _mesa_error(ctx, GL_INVALID_VALUE, "%s(count)", caller);
      return false;
   }

   if (!_mesa_valid_prim_mode(ctx, mode, caller)) {
      return false;
   }

   if (!valid_elements_type(ctx, type, caller))
      return false;

   if (!check_valid_to_render(ctx, caller))
      return false;

   return true;
}

/**
 * Error checking for glDrawElements().  Includes parameter checking
 * and VBO bounds checking.
 * \return GL_TRUE if OK to render, GL_FALSE if error found
 */
GLboolean
_mesa_validate_DrawElements(struct gl_context *ctx,
                            GLenum mode, GLsizei count, GLenum type,
                            const GLvoid *indices)
{
   FLUSH_CURRENT(ctx, 0);

   return validate_DrawElements_common(ctx, mode, count, type, indices,
                                       "glDrawElements");
}


/**
 * Error checking for glMultiDrawElements().  Includes parameter checking
 * and VBO bounds checking.
 * \return GL_TRUE if OK to render, GL_FALSE if error found
 */
GLboolean
_mesa_validate_MultiDrawElements(struct gl_context *ctx,
                                 GLenum mode, const GLsizei *count,
                                 GLenum type, const GLvoid * const *indices,
                                 GLsizei primcount)
{
   GLsizei i;

   FLUSH_CURRENT(ctx, 0);

   /*
    * Section 2.3.1 (Errors) of the OpenGL 4.5 (Core Profile) spec says:
    *
    *    "If a negative number is provided where an argument of type sizei or
    *     sizeiptr is specified, an INVALID_VALUE error is generated."
    *
    * and in the same section:
    *
    *    "In other cases, there are no side effects unless otherwise noted;
    *     the command which generates the error is ignored so that it has no
    *     effect on GL state or framebuffer contents."
    *
    * Hence, check both primcount and all the count[i].
    */
   if (primcount < 0) {
      _mesa_error(ctx, GL_INVALID_VALUE,
                  "glMultiDrawElements(primcount=%d)", primcount);
      return GL_FALSE;
   }

   for (i = 0; i < primcount; i++) {
      if (count[i] < 0) {
         _mesa_error(ctx, GL_INVALID_VALUE,
                     "glMultiDrawElements(count)" );
         return GL_FALSE;
      }
   }

   if (!_mesa_valid_prim_mode(ctx, mode, "glMultiDrawElements")) {
      return GL_FALSE;
   }

   if (!valid_elements_type(ctx, type, "glMultiDrawElements"))
      return GL_FALSE;

   if (!check_valid_to_render(ctx, "glMultiDrawElements"))
      return GL_FALSE;

   /* Not using a VBO for indices, so avoid NULL pointer derefs later.
    */
   if (!_mesa_is_bufferobj(ctx->Array.VAO->IndexBufferObj)) {
      for (i = 0; i < primcount; i++) {
         if (!indices[i])
            return GL_FALSE;
      }
   }

   return GL_TRUE;
}


/**
 * Error checking for glDrawRangeElements().  Includes parameter checking
 * and VBO bounds checking.
 * \return GL_TRUE if OK to render, GL_FALSE if error found
 */
GLboolean
_mesa_validate_DrawRangeElements(struct gl_context *ctx, GLenum mode,
                                 GLuint start, GLuint end,
                                 GLsizei count, GLenum type,
                                 const GLvoid *indices)
{
   FLUSH_CURRENT(ctx, 0);

   if (end < start) {
      _mesa_error(ctx, GL_INVALID_VALUE, "glDrawRangeElements(end<start)");
      return GL_FALSE;
   }

   return validate_DrawElements_common(ctx, mode, count, type, indices,
                                       "glDrawRangeElements");
}


static bool
need_xfb_remaining_prims_check(const struct gl_context *ctx)
{
   /* From the GLES3 specification, section 2.14.2 (Transform Feedback
    * Primitive Capture):
    *
    *   The error INVALID_OPERATION is generated by DrawArrays and
    *   DrawArraysInstanced if recording the vertices of a primitive to the
    *   buffer objects being used for transform feedback purposes would result
    *   in either exceeding the limits of any buffer object’s size, or in
    *   exceeding the end position offset + size − 1, as set by
    *   BindBufferRange.
    *
    * This is in contrast to the behaviour of desktop GL, where the extra
    * primitives are silently dropped from the transform feedback buffer.
    *
    * This text is removed in ES 3.2, presumably because it's not really
    * implementable with geometry and tessellation shaders.  In fact,
    * the OES_geometry_shader spec says:
    *
    *    "(13) Does this extension change how transform feedback operates
    *     compared to unextended OpenGL ES 3.0 or 3.1?
    *
    *     RESOLVED: Yes. Because dynamic geometry amplification in a geometry
    *     shader can make it difficult if not impossible to predict the amount
    *     of geometry that may be generated in advance of executing the shader,
    *     the draw-time error for transform feedback buffer overflow conditions
    *     is removed and replaced with the GL behavior (primitives are not
    *     written and the corresponding counter is not updated)..."
    */
   return _mesa_is_gles3(ctx) && _mesa_is_xfb_active_and_unpaused(ctx) &&
          !_mesa_has_OES_geometry_shader(ctx) &&
          !_mesa_has_OES_tessellation_shader(ctx);
}


static bool
validate_draw_arrays(struct gl_context *ctx, const char *func,
                     GLenum mode, GLsizei count, GLsizei numInstances)
{
   FLUSH_CURRENT(ctx, 0);

   if (count < 0) {
      _mesa_error(ctx, GL_INVALID_VALUE, "%s(count)", func);
      return false;
   }

   if (!_mesa_valid_prim_mode(ctx, mode, func))
      return false;

   if (!check_valid_to_render(ctx, func))
      return false;

   if (need_xfb_remaining_prims_check(ctx)) {
      struct gl_transform_feedback_object *xfb_obj
         = ctx->TransformFeedback.CurrentObject;
      size_t prim_count = vbo_count_tessellated_primitives(mode, count, numInstances);
      if (xfb_obj->GlesRemainingPrims < prim_count) {
         _mesa_error(ctx, GL_INVALID_OPERATION,
                     "%s(exceeds transform feedback size)", func);
         return false;
      }
      xfb_obj->GlesRemainingPrims -= prim_count;
   }

   if (count == 0)
      return false;

   return true;
}

/**
 * Called from the tnl module to error check the function parameters and
 * verify that we really can draw something.
 * \return GL_TRUE if OK to render, GL_FALSE if error found
 */
GLboolean
_mesa_validate_DrawArrays(struct gl_context *ctx, GLenum mode, GLsizei count)
{
   return validate_draw_arrays(ctx, "glDrawArrays", mode, count, 1);
}


GLboolean
_mesa_validate_DrawArraysInstanced(struct gl_context *ctx, GLenum mode, GLint first,
                                   GLsizei count, GLsizei numInstances)
{
   if (first < 0) {
      _mesa_error(ctx, GL_INVALID_VALUE,
                  "glDrawArraysInstanced(start=%d)", first);
      return GL_FALSE;
   }

   if (numInstances <= 0) {
      if (numInstances < 0)
         _mesa_error(ctx, GL_INVALID_VALUE,
                     "glDrawArraysInstanced(numInstances=%d)", numInstances);
      return GL_FALSE;
   }

   return validate_draw_arrays(ctx, "glDrawArraysInstanced", mode, count, 1);
}


/**
 * Called to error check the function parameters.
 *
 * Note that glMultiDrawArrays is not part of GLES, so there's limited scope
 * for sharing code with the validation of glDrawArrays.
 */
bool
_mesa_validate_MultiDrawArrays(struct gl_context *ctx, GLenum mode,
                               const GLsizei *count, GLsizei primcount)
{
   int i;

   FLUSH_CURRENT(ctx, 0);

   if (!_mesa_valid_prim_mode(ctx, mode, "glMultiDrawArrays"))
      return false;

   if (!check_valid_to_render(ctx, "glMultiDrawArrays"))
      return false;

   if (primcount < 0) {
      _mesa_error(ctx, GL_INVALID_VALUE, "glMultiDrawArrays(primcount=%d)",
                  primcount);
      return false;
   }

   for (i = 0; i < primcount; ++i) {
      if (count[i] < 0) {
         _mesa_error(ctx, GL_INVALID_VALUE, "glMultiDrawArrays(count[%d]=%d)",
                     i, count[i]);
         return false;
      }
   }

   if (need_xfb_remaining_prims_check(ctx)) {
      struct gl_transform_feedback_object *xfb_obj
         = ctx->TransformFeedback.CurrentObject;
      size_t xfb_prim_count = 0;

      for (i = 0; i < primcount; ++i)
         xfb_prim_count += vbo_count_tessellated_primitives(mode, count[i], 1);

      if (xfb_obj->GlesRemainingPrims < xfb_prim_count) {
         _mesa_error(ctx, GL_INVALID_OPERATION,
                     "glMultiDrawArrays(exceeds transform feedback size)");
         return false;
      }
      xfb_obj->GlesRemainingPrims -= xfb_prim_count;
   }

   return true;
}


GLboolean
_mesa_validate_DrawElementsInstanced(struct gl_context *ctx,
                                     GLenum mode, GLsizei count, GLenum type,
                                     const GLvoid *indices, GLsizei numInstances)
{
   FLUSH_CURRENT(ctx, 0);

   if (numInstances < 0) {
      _mesa_error(ctx, GL_INVALID_VALUE,
                  "glDrawElementsInstanced(numInstances=%d)", numInstances);
      return GL_FALSE;
   }

   return validate_DrawElements_common(ctx, mode, count, type, indices,
                                       "glDrawElementsInstanced")
      && (numInstances > 0);
}


GLboolean
_mesa_validate_DrawTransformFeedback(struct gl_context *ctx,
                                     GLenum mode,
                                     struct gl_transform_feedback_object *obj,
                                     GLuint stream,
                                     GLsizei numInstances)
{
   FLUSH_CURRENT(ctx, 0);

   if (!_mesa_valid_prim_mode(ctx, mode, "glDrawTransformFeedback*(mode)")) {
      return GL_FALSE;
   }

   if (!obj) {
      _mesa_error(ctx, GL_INVALID_VALUE, "glDrawTransformFeedback*(name)");
      return GL_FALSE;
   }

   /* From the GL 4.5 specification, page 429:
    * "An INVALID_VALUE error is generated if id is not the name of a
    *  transform feedback object."
    */
   if (!obj->EverBound) {
      _mesa_error(ctx, GL_INVALID_VALUE, "glDrawTransformFeedback*(name)");
      return GL_FALSE;
   }

   if (stream >= ctx->Const.MaxVertexStreams) {
      _mesa_error(ctx, GL_INVALID_VALUE,
                  "glDrawTransformFeedbackStream*(index>=MaxVertexStream)");
      return GL_FALSE;
   }

   if (!obj->EndedAnytime) {
      _mesa_error(ctx, GL_INVALID_OPERATION, "glDrawTransformFeedback*");
      return GL_FALSE;
   }

   if (numInstances <= 0) {
      if (numInstances < 0)
         _mesa_error(ctx, GL_INVALID_VALUE,
                     "glDrawTransformFeedback*Instanced(numInstances=%d)",
                     numInstances);
      return GL_FALSE;
   }

   if (!check_valid_to_render(ctx, "glDrawTransformFeedback*")) {
      return GL_FALSE;
   }

   return GL_TRUE;
}

static GLboolean
valid_draw_indirect(struct gl_context *ctx,
                    GLenum mode, const GLvoid *indirect,
                    GLsizei size, const char *name)
{
   const uint64_t end = (uint64_t) (uintptr_t) indirect + size;

   /* OpenGL ES 3.1 spec. section 10.5:
    *
    *      "DrawArraysIndirect requires that all data sourced for the
    *      command, including the DrawArraysIndirectCommand
    *      structure,  be in buffer objects,  and may not be called when
    *      the default vertex array object is bound."
    */
   if (ctx->Array.VAO == ctx->Array.DefaultVAO) {
      _mesa_error(ctx, GL_INVALID_OPERATION, "(no VAO bound)");
      return GL_FALSE;
   }

   /* From OpenGL ES 3.1 spec. section 10.5:
    *     "An INVALID_OPERATION error is generated if zero is bound to
    *     VERTEX_ARRAY_BINDING, DRAW_INDIRECT_BUFFER or to any enabled
    *     vertex array."
    *
    * Here we check that for each enabled vertex array we have a vertex
    * buffer bound.
    */
   if (_mesa_is_gles31(ctx) &&
       ctx->Array.VAO->_Enabled & ~ctx->Array.VAO->VertexAttribBufferMask) {
      _mesa_error(ctx, GL_INVALID_OPERATION, "%s(No VBO bound)", name);
      return GL_FALSE;
   }

   if (!_mesa_valid_prim_mode(ctx, mode, name))
      return GL_FALSE;

   /* OpenGL ES 3.1 specification, section 10.5:
    *
    *      "An INVALID_OPERATION error is generated if
    *      transform feedback is active and not paused."
    *
    * The OES_geometry_shader spec says:
    *
    *    On p. 250 in the errors section for the DrawArraysIndirect command,
    *    and on p. 254 in the errors section for the DrawElementsIndirect
    *    command, delete the errors which state:
    *
    *    "An INVALID_OPERATION error is generated if transform feedback is
    *    active and not paused."
    *
    *    (thus allowing transform feedback to work with indirect draw commands).
    */
   if (_mesa_is_gles31(ctx) && !ctx->Extensions.OES_geometry_shader &&
       _mesa_is_xfb_active_and_unpaused(ctx)) {
      _mesa_error(ctx, GL_INVALID_OPERATION,
                  "%s(TransformFeedback is active and not paused)", name);
   }

   /* From OpenGL version 4.4. section 10.5
    * and OpenGL ES 3.1, section 10.6:
    *
    *      "An INVALID_VALUE error is generated if indirect is not a
    *       multiple of the size, in basic machine units, of uint."
    */
   if ((GLsizeiptr)indirect & (sizeof(GLuint) - 1)) {
      _mesa_error(ctx, GL_INVALID_VALUE,
                  "%s(indirect is not aligned)", name);
      return GL_FALSE;
   }

   if (!_mesa_is_bufferobj(ctx->DrawIndirectBuffer)) {
      _mesa_error(ctx, GL_INVALID_OPERATION,
                  "%s: no buffer bound to DRAW_INDIRECT_BUFFER", name);
      return GL_FALSE;
   }

   if (_mesa_check_disallowed_mapping(ctx->DrawIndirectBuffer)) {
      _mesa_error(ctx, GL_INVALID_OPERATION,
                  "%s(DRAW_INDIRECT_BUFFER is mapped)", name);
      return GL_FALSE;
   }

   /* From the ARB_draw_indirect specification:
    * "An INVALID_OPERATION error is generated if the commands source data
    *  beyond the end of the buffer object [...]"
    */
   if (ctx->DrawIndirectBuffer->Size < end) {
      _mesa_error(ctx, GL_INVALID_OPERATION,
                  "%s(DRAW_INDIRECT_BUFFER too small)", name);
      return GL_FALSE;
   }

   if (!check_valid_to_render(ctx, name))
      return GL_FALSE;

   return GL_TRUE;
}

static inline GLboolean
valid_draw_indirect_elements(struct gl_context *ctx,
                             GLenum mode, GLenum type, const GLvoid *indirect,
                             GLsizeiptr size, const char *name)
{
   if (!valid_elements_type(ctx, type, name))
      return GL_FALSE;

   /*
    * Unlike regular DrawElementsInstancedBaseVertex commands, the indices
    * may not come from a client array and must come from an index buffer.
    * If no element array buffer is bound, an INVALID_OPERATION error is
    * generated.
    */
   if (!_mesa_is_bufferobj(ctx->Array.VAO->IndexBufferObj)) {
      _mesa_error(ctx, GL_INVALID_OPERATION,
                  "%s(no buffer bound to GL_ELEMENT_ARRAY_BUFFER)", name);
      return GL_FALSE;
   }

   return valid_draw_indirect(ctx, mode, indirect, size, name);
}

static inline GLboolean
valid_draw_indirect_multi(struct gl_context *ctx,
                          GLsizei primcount, GLsizei stride,
                          const char *name)
{

   /* From the ARB_multi_draw_indirect specification:
    * "INVALID_VALUE is generated by MultiDrawArraysIndirect or
    *  MultiDrawElementsIndirect if <primcount> is negative."
    *
    * "<primcount> must be positive, otherwise an INVALID_VALUE error will
    *  be generated."
    */
   if (primcount < 0) {
      _mesa_error(ctx, GL_INVALID_VALUE, "%s(primcount < 0)", name);
      return GL_FALSE;
   }


   /* From the ARB_multi_draw_indirect specification:
    * "<stride> must be a multiple of four, otherwise an INVALID_VALUE
    *  error is generated."
    */
   if (stride % 4) {
      _mesa_error(ctx, GL_INVALID_VALUE, "%s(stride %% 4)", name);
      return GL_FALSE;
   }

   return GL_TRUE;
}

GLboolean
_mesa_validate_DrawArraysIndirect(struct gl_context *ctx,
                                  GLenum mode,
                                  const GLvoid *indirect)
{
   const unsigned drawArraysNumParams = 4;

   FLUSH_CURRENT(ctx, 0);

   return valid_draw_indirect(ctx, mode,
                              indirect, drawArraysNumParams * sizeof(GLuint),
                              "glDrawArraysIndirect");
}

GLboolean
_mesa_validate_DrawElementsIndirect(struct gl_context *ctx,
                                    GLenum mode, GLenum type,
                                    const GLvoid *indirect)
{
   const unsigned drawElementsNumParams = 5;

   FLUSH_CURRENT(ctx, 0);

   return valid_draw_indirect_elements(ctx, mode, type,
                                       indirect, drawElementsNumParams * sizeof(GLuint),
                                       "glDrawElementsIndirect");
}

GLboolean
_mesa_validate_MultiDrawArraysIndirect(struct gl_context *ctx,
                                       GLenum mode,
                                       const GLvoid *indirect,
                                       GLsizei primcount, GLsizei stride)
{
   GLsizeiptr size = 0;
   const unsigned drawArraysNumParams = 4;

   FLUSH_CURRENT(ctx, 0);

   /* caller has converted stride==0 to drawArraysNumParams * sizeof(GLuint) */
   assert(stride != 0);

   if (!valid_draw_indirect_multi(ctx, primcount, stride,
                                  "glMultiDrawArraysIndirect"))
      return GL_FALSE;

   /* number of bytes of the indirect buffer which will be read */
   size = primcount
      ? (primcount - 1) * stride + drawArraysNumParams * sizeof(GLuint)
      : 0;

   if (!valid_draw_indirect(ctx, mode, indirect, size,
                            "glMultiDrawArraysIndirect"))
      return GL_FALSE;

   return GL_TRUE;
}

GLboolean
_mesa_validate_MultiDrawElementsIndirect(struct gl_context *ctx,
                                         GLenum mode, GLenum type,
                                         const GLvoid *indirect,
                                         GLsizei primcount, GLsizei stride)
{
   GLsizeiptr size = 0;
   const unsigned drawElementsNumParams = 5;

   FLUSH_CURRENT(ctx, 0);

   /* caller has converted stride==0 to drawElementsNumParams * sizeof(GLuint) */
   assert(stride != 0);

   if (!valid_draw_indirect_multi(ctx, primcount, stride,
                                  "glMultiDrawElementsIndirect"))
      return GL_FALSE;

   /* number of bytes of the indirect buffer which will be read */
   size = primcount
      ? (primcount - 1) * stride + drawElementsNumParams * sizeof(GLuint)
      : 0;

   if (!valid_draw_indirect_elements(ctx, mode, type,
                                     indirect, size,
                                     "glMultiDrawElementsIndirect"))
      return GL_FALSE;

   return GL_TRUE;
}

static GLboolean
valid_draw_indirect_parameters(struct gl_context *ctx,
                               const char *name,
                               GLintptr drawcount)
{
   /* From the ARB_indirect_parameters specification:
    * "INVALID_VALUE is generated by MultiDrawArraysIndirectCountARB or
    *  MultiDrawElementsIndirectCountARB if <drawcount> is not a multiple of
    *  four."
    */
   if (drawcount & 3) {
      _mesa_error(ctx, GL_INVALID_VALUE,
                  "%s(drawcount is not a multiple of 4)", name);
      return GL_FALSE;
   }

   /* From the ARB_indirect_parameters specification:
    * "INVALID_OPERATION is generated by MultiDrawArraysIndirectCountARB or
    *  MultiDrawElementsIndirectCountARB if no buffer is bound to the
    *  PARAMETER_BUFFER_ARB binding point."
    */
   if (!_mesa_is_bufferobj(ctx->ParameterBuffer)) {
      _mesa_error(ctx, GL_INVALID_OPERATION,
                  "%s: no buffer bound to PARAMETER_BUFFER", name);
      return GL_FALSE;
   }

   if (_mesa_check_disallowed_mapping(ctx->ParameterBuffer)) {
      _mesa_error(ctx, GL_INVALID_OPERATION,
                  "%s(PARAMETER_BUFFER is mapped)", name);
      return GL_FALSE;
   }

   /* From the ARB_indirect_parameters specification:
    * "INVALID_OPERATION is generated by MultiDrawArraysIndirectCountARB or
    *  MultiDrawElementsIndirectCountARB if reading a <sizei> typed value
    *  from the buffer bound to the PARAMETER_BUFFER_ARB target at the offset
    *  specified by <drawcount> would result in an out-of-bounds access."
    */
   if (ctx->ParameterBuffer->Size < drawcount + sizeof(GLsizei)) {
      _mesa_error(ctx, GL_INVALID_OPERATION,
                  "%s(PARAMETER_BUFFER too small)", name);
      return GL_FALSE;
   }

   return GL_TRUE;
}

GLboolean
_mesa_validate_MultiDrawArraysIndirectCount(struct gl_context *ctx,
                                            GLenum mode,
                                            GLintptr indirect,
                                            GLintptr drawcount,
                                            GLsizei maxdrawcount,
                                            GLsizei stride)
{
   GLsizeiptr size = 0;
   const unsigned drawArraysNumParams = 4;

   FLUSH_CURRENT(ctx, 0);

   /* caller has converted stride==0 to drawArraysNumParams * sizeof(GLuint) */
   assert(stride != 0);

   if (!valid_draw_indirect_multi(ctx, maxdrawcount, stride,
                                  "glMultiDrawArraysIndirectCountARB"))
      return GL_FALSE;

   /* number of bytes of the indirect buffer which will be read */
   size = maxdrawcount
      ? (maxdrawcount - 1) * stride + drawArraysNumParams * sizeof(GLuint)
      : 0;

   if (!valid_draw_indirect(ctx, mode, (void *)indirect, size,
                            "glMultiDrawArraysIndirectCountARB"))
      return GL_FALSE;

   return valid_draw_indirect_parameters(
         ctx, "glMultiDrawArraysIndirectCountARB", drawcount);
}

GLboolean
_mesa_validate_MultiDrawElementsIndirectCount(struct gl_context *ctx,
                                              GLenum mode, GLenum type,
                                              GLintptr indirect,
                                              GLintptr drawcount,
                                              GLsizei maxdrawcount,
                                              GLsizei stride)
{
   GLsizeiptr size = 0;
   const unsigned drawElementsNumParams = 5;

   FLUSH_CURRENT(ctx, 0);

   /* caller has converted stride==0 to drawElementsNumParams * sizeof(GLuint) */
   assert(stride != 0);

   if (!valid_draw_indirect_multi(ctx, maxdrawcount, stride,
                                  "glMultiDrawElementsIndirectCountARB"))
      return GL_FALSE;

   /* number of bytes of the indirect buffer which will be read */
   size = maxdrawcount
      ? (maxdrawcount - 1) * stride + drawElementsNumParams * sizeof(GLuint)
      : 0;

   if (!valid_draw_indirect_elements(ctx, mode, type,
                                     (void *)indirect, size,
                                     "glMultiDrawElementsIndirectCountARB"))
      return GL_FALSE;

   return valid_draw_indirect_parameters(
         ctx, "glMultiDrawElementsIndirectCountARB", drawcount);
}
