/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.io;

import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.DBTableModel;

import java.sql.Connection;
import java.sql.SQLException;
import java.util.Map;
import java.util.HashMap;
import java.util.Arrays;

import org.jfree.chart.LegendItemSource;
import org.jfree.data.jdbc.JDBCPieDataset;
import org.jfree.data.jdbc.JDBCXYDataset;
import org.jfree.data.general.Dataset;
import org.jfree.data.general.DefaultValueDataset;

import org.jfree.data.category.DefaultCategoryDataset;
import org.jfree.data.xy.CategoryTableXYDataset;
import org.jfree.data.general.DefaultPieDataset;
import org.jfree.data.DefaultKeyedValues;

/**
 * 引数タイプに応じたレンデラーやデータセットを管理します。
 *
 * タイプ、レンデラー、データセット の組み合わせで、構築するオブジェクトが異なります。
 *
 * @version  0.9.0	2007/06/21
 * @author	 Kazuhiko Hasegawa
 * @since	 JDK1.1,
 */
final class TypeRenderer {
	private static final String REND_CLASS = "org.jfree.chart.renderer." ;
	private static final String HYBS_CLASS = "org.opengion.hayabusa.io." ;		// 4.1.1.0 (2008/02/04)

	private final String type ;
	private final String rend ;		// org.jfree.chart.renderer 以降の文字列
	private final String dtset ;	// org.opengion.hayabusa.io 以降の文字列
	private final String plot ;		// 以降の文字列

	/**
	 * TypeRenderer オブジェクトを作成します。
	 *
	 * チャートタイプ は、外部からチャートを指定するのに便利なように、キー化
	 * されています。このキーに基づいて、ChartFactory クラスの
	 * チャートタイプ変換表に基づいて、レンデラーや、データセットを作成します。
	 * このクラスは、これらの変換表の個々の属性を管理しています。
	 *
	 * @og.rev 5.3.0.0 (2010/12/01) plot 追加
	 *
	 * @param	type    	チャートのタイプを区別する文字列
	 * @param	renderer	チャートのタイプに応じたレンデラーのキー文字列
	 * @param	dtset   	チャートのタイプに応じたデータセットのキー文字列
	 * @param	plot    	チャートのタイプに応じたプロットのキー文字列
	 */
//	public TypeRenderer( final String type,final String renderer,final String dtset ) {
	public TypeRenderer( final String type,final String renderer,final String dtset,final String plot ) {
		this.type  = type ;
		this.rend  = renderer ;
		this.dtset = dtset ;
		this.plot  = plot ;			// 5.3.0.0 (2010/12/01) plot 追加
	}

	/**
	 * チャートのタイプを区別する文字列を返します。
	 *
	 * @return	チャートのタイプを区別する文字列
	 */
	public String getType() { return type; }

	/**
	 * チャートのタイプに応じたレンデラーのキー文字列を返します。
	 *
	 * @return	チャートのタイプに応じたレンデラーのキー文字列
	 */
	public String getRendererType() { return rend; }

	/**
	 * チャートのタイプに応じたレンデラーオブジェクトを返します。
	 *
	 * org.jfree.chart.renderer パッケージのサブモジュールのレンデラークラスを
	 * 先に登録してある レンデラーのキー文字列 と合成して、クラスを動的に作成します。
	 *
	 * @og.rev 4.1.1.0 (2008/02/04) Barチャート追加
	 * @og.rev 5.3.0.0 (2010/12/01) レンデラーが null の場合の対応
	 *
	 * @return	LegendItemSource チャートのタイプに応じたレンデラーオブジェクト(存在しない場合は、null)
	 */
	public LegendItemSource getRenderer() {
		if( rend == null ) { return null; }		// 5.3.0.0 (2010/12/01)

		String key ;
		if( type.startsWith( "Hybs" ) ) {
			key = HYBS_CLASS + rend ;
		}
		else {
			key = REND_CLASS + rend ;
		}

		return (LegendItemSource)StringUtil.newInstance( key ) ;
	}

	/**
	 * チャートのタイプに応じたデータセットのキー文字列を返します。
	 *
	 * @return	チャートのタイプに応じたデータセットのキー文字列
	 */
	public String getDatasetType() { return dtset; }

	/**
	 * チャートのタイプに応じたプロットのキー文字列を返します。
	 *
	 * @og.rev 5.3.0.0 (2010/12/01) 新規追加
	 *
	 * @return	チャートのタイプに応じたプロットのキー文字列
	 */
	public String getPlotType() { return plot; }
}
