/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.db;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.resource.CodeData;

/**
 * データのコード情報を取り扱うクラスです。
 *
 * コードのキーとラベルの情報から、HTMLのメニューやリストを作成するための オプション
 * タグを作成したり、与えられたキーをもとに、チェック済みのオプションタグを作成したり
 * します。
 *
 * @og.group 選択データ制御
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
// public class Selection_RADIO implements Selection {
public class Selection_RADIO extends Selection_NULL {
	private final CodeData codeData ;

	/**
	 * コンストラクター
	 *
	 * @param	cdData	コードデータオブジェクト
	 *
	 */
	public Selection_RADIO( final CodeData cdData ) {
		if( cdData == null ) {
			String errMsg = "コードリソースが定義されていません。" + HybsSystem.CR ;
			throw new HybsSystemException( errMsg );
		}

		codeData = cdData ;
	}

	/**
	 * 初期値が選択済みの 選択肢(オプション)を返します。
	 * このオプションは、引数の値を初期値とするオプションタグを返します。
	 * ※ このクラスでは実装されていません。
	 *
	 * @og.rev 3.6.0.6 (2004/10/22) シーケンスアクセス機能を指定する seqFlag を導入
	 * @og.rev 5.7.7.1 (2014/06/13) Selection_NULL を 継承するため、削除
	 *
	 * @param	selectValue 	選択されている値
	 * @param	seqFlag 	シーケンスアクセス機能の指定
	 *
	 * @return  オプションタグ
	 */
//	public String getOption( final String selectValue,final boolean seqFlag ) {
//		String errMsg = "このクラスでは実装されていません。";
//		throw new UnsupportedOperationException( errMsg );
//	}

	/**
	 * 初期値が選択済みの 選択肢(オプション)を返します。
	 * このオプションは、引数の値を初期値とするオプションタグを返します。
	 * ※ このクラスでは実装されていません。
	 *
	 * @og.rev 5.1.3.0 (2010/02/01) 追加
	 *
	 * @param	selectValue	選択されている値
	 * @param	seqFlag	シーケンスアクセス機能の指定
	 * @param	useShortLabel	短ラベルの指定
	 *
	 * @return  オプションタグ
	 */
	@Override
	public String getOption( final String selectValue,final boolean seqFlag, final boolean useShortLabel ) {
		String errMsg = "このクラスでは実装されていません。";
		throw new UnsupportedOperationException( errMsg );
	}

	/**
	 * 初期値が選択済みの 選択肢(オプション)を返します。
	 * このオプションは、引数の値を初期値とするオプションタグを返します。
	 *
	 * @og.rev 2.1.0.1 (2002/10/17) 選択リストを、正方向にしか選べないようにする sequenceFlag を導入する
	 * @og.rev 3.5.6.3 (2004/07/12) キャッシュを利用せず毎回タグを作成します。
	 * @og.rev 3.8.6.0 (2006/09/29) useLabel 属性 追加
	 *
	 * @param   name         ラジオの name
	 * @param   selectValue  選択されている値
	 * @param   useLabel     ラベル表示の有無 [true:有/false:無]
	 *
	 * @return  オプションタグ
	 */
	@Override
	public String getRadio( final String name,final String selectValue,final boolean useLabel ) {
		String inputTag = "<input type=\"radio\" name=\"" + name + "\" value=\"" ;
		StringBuilder buf = new StringBuilder( HybsSystem.BUFFER_LARGE );
		int size = codeData.getSize();
		for( int i=0; i<size; i++ ) {
			String value = codeData.getCodeKey(i);
			if( useLabel ) { buf.append( "<label>" ); }
			buf.append( inputTag ).append( value ).append( "\"" );
			if( value.equals( selectValue ) ) {
				buf.append( " checked=\"checked\"" );
			}
			buf.append( "/>" );
			if( useLabel ) { buf.append( codeData.getShortLabel(i) ).append( "</label>" ); }
		}
		return buf.toString();
	}

	/**
	 * 選択肢(value)に対するラベルを返します。
	 * 選択肢(value)が、存在しなかった場合は、選択肢そのものを返します。
	 * getValueLabel( XX ) は、getValueLabel( XX,false ) と同じです。
	 *
	 * @og.rev 5.7.7.1 (2014/06/13) Selection_NULL を 継承するため、削除
	 *
	 * @param   selectValue 選択肢の値
	 *
	 * @return  選択肢のラベル
	 * @see     #getValueLabel( String,boolean )
	 */
//	public String getValueLabel( final String selectValue ) {
//		return getValueLabel( selectValue,false );
//	}

	/**
	 * 選択肢(value)に対するラベルを返します。
	 * 選択肢(value)が、存在しなかった場合は、選択肢そのものを返します。
	 * このメソッドでは、短縮ラベルを返すかどうかを指定するフラグを指定します。
	 * getValueLabel( XX,false ) は、getValueLabel( XX ) と同じです。
	 *
	 * @og.rev 4.0.0.0 (2005/11/30) を追加
	 *
	 * @param	selectValue	選択肢の値
	 * @param	flag	短縮ラベルを使用する [true:使用する/false:しない]
	 *
	 * @return  選択肢のラベル
	 * @see     #getValueLabel( String )
	 */
	@Override
	public String getValueLabel( final String selectValue,final boolean flag ) {
		// マッチするアドレスを探す。
		int selected = codeData.getAddress( selectValue );

		if( selected < 0 ) {
			// マッチしなければ、選択肢そのものを返す。
			return selectValue;
		}
		else {
			if( flag ) {
				return codeData.getShortLabel(selected);
			}
			else {
				return codeData.getLongLabel(selected);
			}
		}
	}

	/**
	 * 選択肢(value)に対するラベルを返します。
	 * 選択肢(value)が、存在しなかった場合は、選択肢そのものを返します。
	 * getValueLabel( XX ) は、getValueLabel( XX,false ) と同じです。
	 *
	 * @param   selectValue 選択肢の値
	 *
	 * @return  選択肢のラベル
	 * @see     #getValueLabel( String,boolean )
	 */
	@Override
	public String getRadioLabel( final String selectValue ) {

		StringBuilder rtn = new StringBuilder( HybsSystem.BUFFER_MIDDLE );
		int size = codeData.getSize();
		for( int i=0; i<size; i++ ) {
			String value = codeData.getCodeKey(i);
			if( value.equals( selectValue ) ) {
				rtn.append( "○" );			// (2003/10/08 修正分)
			}
			else {
				rtn.append( "　" );			// (2003/10/08 修正分)
			}
		}

		return rtn.toString();
	}

	/**
	 * マルチ・キーセレクトを使用するかどうかを返します。
	 * true：使用する。false:使用しない です。
	 * ただし、実際に使用するかどうかは、HTML出力時に決めることが出来ます。
	 * ここでは、USE_MULTI_KEY_SELECT が true で、USE_SIZE(=20)以上の場合に
	 * true を返します。
	 *
	 * @og.rev 3.5.5.7 (2004/05/10) 新規作成
	 * @og.rev 5.7.7.1 (2014/06/13) Selection_NULL を 継承するため、削除
	 *
	 * @return  選択リストで、マルチ・キーセレクトを使用するかどうか(true:使用する)
	 */
//	public boolean useMultiSelect() {
//		return false;
//	}

	/**
	 * オブジェクトのキャッシュが時間切れかどうかを返します。
	 * キャッシュが時間切れ(無効)であれば、true を、有効であれば、
	 * false を返します。
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) 新規作成
	 * @og.rev 5.7.7.1 (2014/06/13) Selection_NULL を 継承するため、削除
	 *
	 * @return  キャッシュが時間切れなら true
	 */
//	public boolean isTimeOver() {
//		return false;
//	}
}
