/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.table;

import java.io.File;
import java.io.PrintWriter;
import java.util.Locale;
import java.util.Map;

import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.util.FileUtil;
import org.opengion.fukurou.util.FixLengthData;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.AbstractTableFilter;
import org.opengion.hayabusa.db.DBTableModel;

/**
 * TableFilter_CMNT_TBL は、TableFilter インターフェースを継承した、DBTableModel 処理用の
 * 実装クラスです。
 *
 * ここでは、テーブル一覧の検索結果より、ORACLE のテーブルにコメントを作成します。
 * 構文は、「COMMENT ON TABLE テーブル名 IS 'コメント'」です。
 * このコメントを取り出す場合は、
 *「SELECT COMMENTS FROM USER_TAB_COMMENTS WHERE TABLE_NAME = 'テーブル名'」
 * とします。
 * 
 * SQLのツール類には、このコメントを使用して、テーブルの日本語名を表示させたりします。
 * 
 * 検索では、(TABLE_NAME,NAME_JA) の項目を取得する必要があります。
 *
 * パラメータは、tableFilterタグの keys, vals にそれぞれ記述するか、BODY 部にCSS形式で記述します。
 * 出力ファイル名は、通常、テーブル で１つ、カラムで１つにまとめて作成されます。
 * 【パラメータ】
 *  {
 *       DIR  : {&#064;BASE_DIR}/sql/install/08_CMNT ;    出力ファイルの基準フォルダ(必須)
 *       FILE : false ;                                   出力ファイル名(初期値:CMNT_TBL[.sql|.xml])
 *       XML  : false ;                                   XML出力を行うかどうか[true/false]を指定します(初期値:false)
 *  }
 *
 * @og.formSample
 * ●形式：
 *      select SYSTEM_ID,TABLE_NAME,NAME_JA from GF02
 * 
 *      ① &lt;og:tableFilter classId="CMNT_TBL" keys="DIR" vals='"{&#064;BASE_DIR}/sql/install/08_CMNT"' /&gt;
 *
 *      ② &lt;og:tableFilter classId="CMNT_TBL" &gt;
 *               {
 *                   DIR   : {&#064;BASE_DIR}/sql/install/08_CMNT ;
 *                   FILE  : CMNT_TBL ;
 *                   XML   : false ;
 *               }
 *         &lt;/og:tableFilter&gt;
 *
 * @og.rev 4.0.0.0 (2005/08/31) 新規作成
 *
 * @version  0.9.0  2000/10/17
 * @author   Kazuhiko Hasegawa
 * @since    JDK1.6,
 */
public class TableFilter_CMNT_TBL extends AbstractTableFilter {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "5.6.6.2 (2013/07/19)" ;

	/**
	 * keys の整合性チェックを行うための初期設定を行います。
	 *
	 * @og.rev 5.6.6.1 (2013/07/12) keys の整合性チェック対応
	 *
	 * @param	keysMap keys の整合性チェックを行うための Map
	 */
	@Override
	protected void init( final Map<String,String> keysMap ) {
		keysMap.put( "DIR"	, "出力ファイルの基準フォルダ(必須)"						);
		keysMap.put( "FILE"	, "出力ファイル名(初期値:CMNT_TBL[.sql|.xml])"				);
		keysMap.put( "XML"	, "XML出力を行うかどうか[true/false]を指定(初期値:false)"	);
	}

	private static final String[] DBKEY = {"TABLE_NAME","NAME_JA"};

	/** データのアクセス用の配列番号 {@value} */
	protected static final int TABLE_NAME		= 0;
	/** データのアクセス用の配列番号 {@value} */
	protected static final int NAME_JA			= 1;

 //	private static final String ENCODE = "Windows-31J" ;
	private static final String ENCODE = "UTF-8" ;

	private static final int X = FixLengthData.X ;		// type 定数
	private static final int K = FixLengthData.K ;		// type 定数

	/** 各種定数  */
	protected static final String XML_START_TAG	= "<?xml version='1.0' encoding='UTF-8'?>" + CR + "<ROWSET>";
	protected static final String XML_END_TAG	= "</ROWSET>";
	protected static final String EXEC_START_TAG= "<EXEC_SQL>";
	protected static final String EXEC_END_TAG	= "</EXEC_SQL>";

	/** XML形式かどうか */
	protected boolean		isXml				= false;

	/** ファイル名(拡張子なし) */
	private String		fileName				= "CMNT_TBL";

	/**
	 * DBTableModel処理を実行します。
	 *
	 * @return	実行結果のテーブルモデル
	 */
	public DBTableModel execute() {
		DBTableModel table = getDBTableModel();

		isXml   = StringUtil.nval( getValue( "XML"   ), isXml   );

		int[] clmNo = getTableColumnNo( DBKEY );
		int rowCnt = table.getRowCount();

		File dir = new File( getValue( "DIR" ) );

		if( ! dir.exists() && ! dir.mkdirs() ) {
			String errMsg = "所定のフォルダが作成できませんでした。[" + dir + "]" ;
			// 4.3.4.4 (2009/01/01)
			throw new HybsSystemException( errMsg );
		}

		fileName =  StringUtil.nval( getValue( "FILE" ), fileName );

		// COMMENT ON TABLE テーブル名 IS 'コメント'
		int[] addLen = new int[] { 0,0,0,0 };	// 各データ間のスペース
		int[] type   = new int[] { X,X,X,K };	// 各データの種別 X:半角 S:空白前埋め K:全角混在
		FixLengthData fixData = new FixLengthData( addLen,type );

		String[] data  = null;
		int row = 0;
		try {
			PrintWriter writer = FileUtil.getPrintWriter( new File( dir,fileName + ( isXml ? ".xml" : ".sql" ) ),ENCODE );

			if( isXml ) { writer.println( XML_START_TAG ); }

			// 全データを読み込んで、最大長の計算を行う。
			for( row=0; row<rowCnt; row++ ) {
				data      = table.getValues( row );
				String tableName = data[clmNo[TABLE_NAME]];
				String name_ja   = "'" + data[clmNo[NAME_JA]] + "'";

				String[] outData = new String[] { "COMMENT ON TABLE " , tableName , " IS " , name_ja } ;
				fixData.addListData( outData );
			}

			// データの出力
			for( row=0; row<rowCnt; row++ ) {
				if( isXml ) { writer.print( EXEC_START_TAG ); }
				writer.print( fixData.getFixData( row ) );
				if( isXml ) { writer.println( EXEC_END_TAG ); }
				else		{ writer.println( ";" ); }
			}

			if( isXml ) { writer.println( XML_END_TAG ); }
			writer.close();
		}
		catch( RuntimeException ex ) {
			ErrorMessage errMessage = makeErrorMessage( "TableFilter_CMNT_TBL Error",ErrorMessage.NG );
			data = table.getValues( row );
			errMessage.addMessage( row+1,ErrorMessage.NG,"TABLE",StringUtil.array2csv( data ) );
			// BAT から呼び出す場合があるため、標準エラー出力にも情報を出しておきます。
			System.out.println( errMessage );
		}

		return table;
	}
}
