/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.hayabusa.db.AbstractRenderer;
import org.opengion.hayabusa.db.CellRenderer;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.hayabusa.db.Selection;
import org.opengion.hayabusa.db.SelectionFactory;			// 6.0.4.0 (2014/11/28)

/**
 * MENU レンデラーは、カラムのデータをコードリソースに対応したラベルで
 * プルダウンメニュー表示する場合に使用するクラスです。
 *
 *  カラムの表示に必要な属性は, DBColumn オブジェクト より取り出します。
 * このクラスは、DBColumn オブジェクト毎に１つ作成されます。
 *
 * @og.group データ表示
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Renderer_MENU extends AbstractRenderer {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.0.4.0 (2014/11/28)" ;

	// 3.2.3.0 (2003/06/06) final を削除。サブクラスからアクセスできるように変更。
	private final Selection selection	;
	private final String useSLabel		;	// 5.5.1.0 (2012/04/03)

	private String errMsg	;				// 6.0.4.0 (2014/11/28)

	/**
	 * デフォルトコンストラクター。
	 * このコンストラクターで、基本オブジェクトを作成します。
	 *
	 * @og.rev 3.1.1.1 (2003/04/03) 各オブジェクトから自分のインスタンスを返すファクトリメソッドを追加。
	 * @og.rev 3.2.3.0 (2003/06/06) key 変数をローカル化。
	 * @og.rev 5.5.1.0 (2012/04/03) Slabel対応
	 *
	 */
	public Renderer_MENU() {
		selection  = null;
		useSLabel = "auto";		// 5.5.1.0 (2012/04/03)
	}

	/**
	 * デフォルトコンストラクター。
	 *
	 * @og.rev 3.1.1.1 (2003/04/03) 各オブジェクトから自分のインスタンスを返すファクトリメソッドを追加。
	 * @og.rev 3.2.3.0 (2003/06/06) key 変数をローカル化。
	 * @og.rev 3.3.1.1 (2003/07/03) CodeSelection の設定において、バグ修正。
	 * @og.rev 3.5.4.2 (2003/12/15) makeCodeSelection メソッドを CodeSelectionクラスに変更。
	 * @og.rev 3.5.5.7 (2004/05/10) SelectionFactory を使用して、オブジェクト作成
	 * @og.rev 4.0.0.0 (2005/01/31) Selection_CODE の作成の引数を CodeData に変更。
	 * @og.rev 4.0.0.0 (2007/11/07) SelectionオブジェクトをDBColumnから取得
	 * @og.rev 5.5.1.0 (2012/04/03) Slabel対応
	 * @og.rev 6.0.4.0 (2014/11/28) selection が null の場合、警告表示します。
	 * @og.rev 6.0.4.0 (2014/11/28) サブクラスで タイプを指定するための対応
	 *
	 * @param	clm		DBColumnオブジェクト
	 * @param	type	MENUのサブクラスのタイプ(KEYVAL)
	 */
//	private Renderer_MENU( final DBColumn clm ) {
	protected Renderer_MENU( final DBColumn clm , final String type ) {
		useSLabel = clm.getUseSLabel() ;	// 5.5.1.0 (2012/04/03)
		// 6.0.4.0 (2014/11/28) selection は、Column から取得するのではなく、Factory で作成する。
//		selection = clm.getSelection();										// 4.0.0.0 (2007/11/07)
		if( type != null ) {
			selection = SelectionFactory.newSelection( type,clm.getRendererParam() );
		}
		else {
			selection = SelectionFactory.newSelection( "MENU",clm.getCodeData() );
		}

		// 6.0.4.0 (2014/11/28) selection が null の場合、警告表示します。
		if( selection == null ) {
			errMsg = "codeData が未設定です。"
								+ " name="  + clm.getName()
								+ " label=" + clm.getLabel()
								+ " rendType="  + clm.getRenderer() ;
			System.out.println( errMsg );
		}
	}

	/**
	 * 各オブジェクトから自分のインスタンスを返します。
	 * 自分自身をキャッシュするのか、新たに作成するのかは、各サブクラスの実装に
	 * まかされます。
	 *
	 * @og.rev 3.1.1.1 (2003/04/03) 各オブジェクトから自分のインスタンスを返すファクトリメソッドを追加。
	 * @og.rev 3.1.2.1 (2003/04/10) synchronized を、削除します。
	 * @og.rev 6.0.4.0 (2014/11/28) サブクラスで タイプを指定するための対応
	 *
	 * @param	clm	DBColumnオブジェクト
	 *
	 * @return	CellRendererオブジェクト
	 * @og.rtnNotNull
	 */
	public CellRenderer newInstance( final DBColumn clm ) {
//		return new Renderer_MENU( clm );
		return new Renderer_MENU( clm , null );
	}

	/**
	 * データの表示用文字列を返します。
	 *
	 * @og.rev 5.5.1.0 (2012/04/03) Slabel対応
	 * @og.rev 6.0.4.0 (2014/11/28) selection が null の場合、警告表示します。
	 *
	 * @param   value 入力値
	 *
	 * @return  データの表示用文字列
	 */
	@Override
	public String getValue( final String value ) {
		// 6.0.4.0 (2014/11/28) selection が null の場合、警告表示します。
		if( selection == null ) {
			return "<span class=\"error\">" + errMsg + " value=" + value + "</span>";
		}

//		final boolean uslbl = !"true".equals(useSLabel) ? false : true; // 5.5.1.0 (2012/04/03)
		final boolean uslbl = "true".equals(useSLabel);					// 6.0.4.0 (2014/11/28)
		return selection.getValueLabel( value,uslbl );					// 5.5.1.0 (2012/04/03)
	}

	/**
	 * name属性を変えた、データ表示/編集用のHTML文字列を作成します。
	 * テーブル上の name に 行番号を付加して、名前_行番号 で登録するキーを作成し,
	 * リクエスト情報を１つ毎のフィールドで処理できます。
	 *
	 * @og.rev 4.0.0.0 (2005/11/30) 一覧表示では、短縮ラベルを使用します。
	 * @og.rev 5.5.1.0 (2012/04/03) Slabel対応
	 * @og.rev 6.0.4.0 (2014/11/28) selection が null の場合、警告表示します。
	 *
	 * @param   row   行番号
	 * @param   value 入力値
	 *
	 * @return  データ表示/編集用の文字列
	 */
	@Override
	public String getValue( final int row,final String value ) {
		// 6.0.4.0 (2014/11/28) selection が null の場合、警告表示します。
		if( selection == null ) {
			return "<span class=\"error\">" + errMsg + " value=" + value + " row=" + row + "</span>";
		}

		final boolean uslbl = "auto".equalsIgnoreCase( useSLabel ) || "true".equalsIgnoreCase( useSLabel );		// 5.5.1.0 (2012/04/03)
		return selection.getValueLabel( value,uslbl );
	}
}
