/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import java.util.StringTokenizer;

import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.TagBuffer;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.db.AbstractRenderer;
import org.opengion.hayabusa.db.CellRenderer;
import org.opengion.hayabusa.db.DBColumn;

/**
 * AUTOAREA レンデラは、カラムのデータをテキストエリアで表示する場合に
 * 使用するクラスです。
 * readonlyのテキストエリアでclass=renderer-textareaとして出力し、
 * name属性は付けません。(データは送信されません)
 * エリアの縦、横サイズはエディタの自動拡張テキストエリア(AUTOAREA)と同様にして算出されます。
 *
 * カラムの表示に必要な属性は, DBColumn オブジェクト より取り出します。
 * このクラスは、DBColumn オブジェクト毎に１つ作成されます。
 *
 * @og.rev 5.3.4.0 (2011/04/01) 新規作成
 * @og.group データ編集
 *
 * @version  4.0
 * @author   Hiroki Nakamura
 * @since    JDK5.0,
 */
public class Renderer_AUTOAREA extends AbstractRenderer {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.0.4.0 (2014/11/28)" ;

	private final int COLUMNS_MAXSIZE = HybsSystem.sysInt( "HTML_COLUMNS_MAXSIZE" );
	// viewタグで表示する場合のカラムの大きさ
	private final int VIEW_COLUMNS_MAXSIZE = HybsSystem.sysInt( "HTML_VIEW_COLUMNS_MAXSIZE" );

	private  String		size1;
	private  String		size2;

//	private static final int COL = 0 ;
//	private static final int ROW = 1 ;
	private static final String CODE = "Windows-31J";

	private	int	cols1	;
	private	int	cols2	;
	private	int	rows1	;
	private	int	rows2	;
	private int maxColSize = HybsSystem.sysInt( "HTML_AUTOAREA_MAX_COL_SIZE" );
	private int maxRowSize = HybsSystem.sysInt( "HTML_AUTOAREA_MAX_ROW_SIZE" );

	private final TagBuffer	tagBuffer = new TagBuffer();

	/**
	 * デフォルトコンストラクター。
	 * このコンストラクターで、基本オブジェクトを作成します。
	 *
	 */
	public Renderer_AUTOAREA() {
		// 何もありません。(super を呼び出しません)
	}

	/**
	 * コンストラクター
	 * textareaのサイズを決めるため、sizeとrowを決定する
	 * editorの計算を移植。
	 *
	 * @param	clm	DBColumnオブジェクト
	 */
	private Renderer_AUTOAREA( final DBColumn clm ) {
		String size = clm.getViewLength();
		final int maxlength = clm.getTotalSize();

		if( size != null ) {
			if( size.indexOf( ',' ) >= 0 ) {
				size = size.substring( 0, size.indexOf( ',' ) );
			}
			size1 = size;
			size2 = size;
		}
		else {
			size1 = String.valueOf( clm.getFieldSize( maxlength, COLUMNS_MAXSIZE ) );
			size2 = String.valueOf( clm.getFieldSize( maxlength, VIEW_COLUMNS_MAXSIZE ) );
		}

		final int r1 = clm.getTotalSize()/Integer.parseInt(size1) + 1;	// 4.0.0 (2005/01/31) メソッド名変更
		if( r1 > 5 ) { rows1 = 5; }
		else { rows1 = r1; }

		final int r2 = clm.getTotalSize()/Integer.parseInt(size2) + 1;	// 4.0.0 (2005/01/31) メソッド名変更
		if( r2 > 5 ) { rows2 = 5; }
		else { rows2 = r2; }

		// 3.8.0.2 (2005/07/11) size に、"rows-maxRow,cols-maxCols" を指定
		final String param = StringUtil.nval( clm.getRendererParam(),clm.getViewLength() );
		if( param != null && param.length() != 0 ) {
			final int prmAdrs = param.indexOf( ',' );
			if( prmAdrs > 0 ) {
				final String rowStr = param.substring( 0,prmAdrs );
				final String colStr = param.substring( prmAdrs+1 );

				final int rowAdrs = rowStr.indexOf( '-' );	// rows-maxRow 設定時 '-' がなければ、ただのrows
				if( rowAdrs > 0 ) {
					rows1 = Integer.parseInt( rowStr.substring( 0,rowAdrs ) );
					maxRowSize = Integer.parseInt( rowStr.substring( rowAdrs+1 ) );
				}
				else {
					rows1 = Integer.parseInt( rowStr );
				}
				rows2 = rows1 ;

				final int colAdrs = colStr.indexOf( '-' );	// cols-maxCols 設定時 '-' がなければ、ただのcols
				if( colAdrs > 0 ) {
					cols1 = Integer.parseInt( colStr.substring( 0,colAdrs ) );
					maxColSize = Integer.parseInt( colStr.substring( colAdrs+1 ) );
				}
				else {
					cols1 = Integer.parseInt( colStr );
				}
				cols2 = cols1;
			}
		}
	}

	/**
	 * 各オブジェクトから自分のインスタンスを返します。
	 * 自分自身をキャッシュするのか、新たに作成するのかは、各サブクラスの実装に
	 * まかされます。
	 *
	 * @param	clm	DBColumnオブジェクト
	 *
	 * @return	CellRendererオブジェクト
	 * @og.rtnNotNull
	 */
	public CellRenderer newInstance( final DBColumn clm ) {
		return new Renderer_AUTOAREA( clm );
	}

	/**
	 * データの表示用文字列を返します。
	 *
	 * @og.rev 6.0.4.0 (2014/11/28) ロジックの共通化
	 *
	 * @param   value 入力値
	 *
	 * @return  データの表示用文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String getValue( final String value ) {
		return getRowsColsValue( value==null ? "" : value,cols1,rows1 );

//		int[] rowcol = getRowsCols( value,cols1,rows1 );
//
//		TagBuffer tag = new TagBuffer( "textarea" );
//		tag.add( "cols"    , String.valueOf( rowcol[COL] ) );
//		tag.add( "rows"    , String.valueOf( rowcol[ROW] ) );
//		tag.add( "readonly", "readonly" );
//		tag.add( "class"   , "renderer-textarea" );
//		tag.add( tagBuffer.makeTag() );
//		tag.setBody( value );
//
//		return tag.makeTag();
	}

	/**
	 * データの表示用文字列を返します。
	 *
	 * @og.rev 6.0.4.0 (2014/11/28) ロジックの共通化
	 *
	 * @param   row   行番号
	 * @param   value 入力値
	 *
	 * @return  データ表示用の文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String getValue( final int row,final String value ) {
		return getRowsColsValue( value==null ? "" : value,cols2,rows2 );

//		int[] rowcol = getRowsCols( value,cols2,rows2 );
//
//		TagBuffer tag = new TagBuffer( "textarea" );
//		tag.add( "cols"    , String.valueOf( rowcol[COL] ) );
//		tag.add( "rows"    , String.valueOf( rowcol[ROW] ) );
//		tag.add( "readonly", "readonly" );
//		tag.add( "class"   , "renderer-textarea" );
//		tag.add( tagBuffer.makeTag() );
//		tag.setBody( value );
//
//		return tag.makeTag( row,value );		// ※ return tag.makeTag(); でよい。
	}

	/**
	 * データ出力用の文字列を作成します。
	 * ファイル等に出力する形式を想定しますので、HTMLタグを含まない
	 * データを返します。
	 * 基本は、#getValue( String ) をそのまま返します。
	 *
	 * @og.rev 6.0.4.0 (2014/11/28) データ出力用のレンデラー
	 *
	 * @param   value 入力値
	 *
	 * @return  データ出力用の文字列
	 * @og.rtnNotNull
	 * @see		#getValue( String )
	 */
	@Override
	public String getWriteValue( final String value ) {
		return value==null ? "" : value;
	}

	/**
	 * 自動表示する行列の数を求めて、値を返します。
	 * 行数は、引数の文字列中に含まれる 改行コードの個数を求めます。
	 * 列数は、各行数のなかの最大桁数より求めます。これには半角、全角が含まれる為、
	 * 半角換算での文字数ということになります。
	 * 行数と列数が、初期設定の行数と列数より小さい場合は、初期設定値が使用されます。
	 *
	 * @og.rev 6.0.4.0 (2014/11/28) ロジックの共通化
	 *
	 * @param   value 入力値 表示文字列
	 * @param	cols	最小カラム数
	 * @param	rows	最小行数
	 *
	 * @return  データ表示用の文字列
	 * @og.rtnNotNull
	 */
//	private int[] getRowsCols( final String value,final int cols, final int rows ) {
	private String getRowsColsValue( final String value,final int cols, final int rows ) {
		// 6.0.4.0 (2014/11/28) null は引数に来ない。
//		if( value == null ) {
//			return new int[] { rows,cols };
//		}

		final StringTokenizer token = new StringTokenizer( value, "\n", true );

		int cntRow = 1;
		int maxCol = 0;
		while ( token.hasMoreTokens() ) {
			final String val = token.nextToken();
			if( "\n".equals( val ) ) { cntRow++; }
			else {
				final byte[] byteValue = StringUtil.makeByte( val,CODE );	// 3.5.5.3 (2004/04/09)
				final int byteSize = byteValue.length;
				if( maxColSize > 0 && byteSize > maxColSize ) {		// 最大列数
					cntRow += byteSize / maxColSize;
					maxCol = maxColSize ;
				}
				else if( byteSize > maxCol ) { maxCol = byteSize; }
			}
			if( maxRowSize > 0 && cntRow >= maxRowSize ) {		// 最大行数
				cntRow = maxRowSize;
				break;
			}
		}

		maxCol += 2;	// マージン。フォントや画面サイズに影響する為、比率のほうがよい？

		// 6.0.4.0 (2014/11/28) 配列にいれずに、直接設定する。
//		int[] rtn = new int[2];
//		rtn[ROW] = (rows<cntRow) ? cntRow : rows ;
//		rtn[COL] = (cols<maxCol) ? maxCol : cols ;

//		return rtn ;

		// 6.0.4.0 (2014/11/28) タグを作成する処理も、ここで行います。
		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
		return new TagBuffer( "textarea" )
//					.add( "cols"    , String.valueOf( (cols<maxCol) ? maxCol : cols ) )
//					.add( "rows"    , String.valueOf( (rows<cntRow) ? cntRow : rows ) )
					.add( "cols"    , String.valueOf( Math.max( cols,maxCol ) ) )		// 6.1.1.0 (2015/01/17) ソースの見やすさ優先
					.add( "rows"    , String.valueOf( Math.max( rows,cntRow ) ) )
					.add( "readonly", "readonly" )
					.add( "class"   , "renderer-textarea" )
					.add( tagBuffer.makeTag() )
					.addBody( value )
					.makeTag();

//		final TagBuffer tag = new TagBuffer( "textarea" );
//		tag.add( "cols"    , String.valueOf( (cols<maxCol) ? maxCol : cols ) );
//		tag.add( "rows"    , String.valueOf( (rows<cntRow) ? cntRow : rows ) );
//		tag.add( "readonly", "readonly" );
//		tag.add( "class"   , "renderer-textarea" );
//		tag.add( tagBuffer.makeTag() );
//		tag.setBody( value );
//
//		return tag.makeTag();
	}
}
