/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;

import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.ToString;						// 6.1.1.0 (2015/01/17)
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.db.DBTableModelUtil;
import org.opengion.hayabusa.report2.QueueManager_DIRECT;

import static org.opengion.fukurou.util.StringUtil.nval;

/**
 * 検索結果の DBTableModelオブジェクトをレポート形式に変換するタグです。
 *
 * データ(DBTableModel)と、コントローラ(DBTableReport クラス)を与えて、
 * 外部からコントロールすることで、各種形式で データ(DBTableModel)を表示させることが
 * 可能です。
 * このタグを使用するには、OpenOffice.orgのモジュールがインストールされてている必要があります。
 * また、出力するために帳票システム関連のデータ設定やマスタ設定は一切必要ありません。
 *
 * @og.formSample
 * ●形式：&lt;og:report fileURL="[･･･]" listId="[･･･]" ･･･ /&gt;
 * ●body：なし
 *
 * ●Tag定義：
 *   &lt;og:report2
 *       fileURL            【TAG】雛型のHTMLファイルの保存してある ディレクトリを指定します
 *       listId           ○【TAG】帳票IDを指定します(必須)。
 *       outFileURL         【TAG】出力HTMLファイルの保存してあるディレクトリを指定します(初期値:FILE_URL[={@og.value org.opengion.hayabusa.common.SystemData#FILE_URL}])
 *       outFilename      ○【TAG】ファイルを作成するときの出力ファイル名をセットします(必須)。
 *       headerKeys         【TAG】固定部の{&#064;KEY} の KEY 部分をCSV形式で複数指定します
 *       headerVals         【TAG】固定部のKEY に対応する値をCSV形式で複数指定します
 *       footerKeys         【TAG】繰り返し部の終了後に表示する key 部分をCSV形式で複数指定します
 *       footerVals         【TAG】固定部のKEY に対応する値をCSV形式で複数指定します
 *       pageEndCut         【TAG】ボディー部(繰り返し部)がなくなったときに、それ以降を表示するかどうか[true/false]を指定します(初期値:true)
 *       useLocalResource   【TAG】各システムのリソース(ローカルリソース)を使用するか[true/false]を指定します(初期値:true)
 *       useSheetName       【TAG】PAGEBREAKカラムの値を、シート名として使うかどうか[true/false]を指定します(初期値:false)
 *       fgrun              【TAG】出力方法を指定します(初期値:P(PDF出力))
 *       printerName        【TAG】プリンター名を指定します
 *       language           【TAG】タグ内部で使用する言語コード[ja/en/zh/…]を指定します
 *       scope              【TAG】キャッシュする場合のスコープ[request/page/session/applicaton]を指定します(初期値:session)
 *       tableId            【TAG】(通常使いません)sessionから所得する DBTableModelオブジェクトの ID
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   /&gt;
 *
 * ●使用例
 *
 * @og.group その他出力
 *
 * @version  4.0
 * @author   Hiroki Nakamura
 * @since    JDK5.0,
 */
public class ReportTableTag2 extends CommonTagSupport {
	private static final String VERSION = "5.7.6.2 (2014/05/16)" ;

	private static final long serialVersionUID = 576220140516L ;

	private final String  BASE_URL	= HybsSystem.sys( "FILE_URL" );

	private String		fileURL			= BASE_URL;	// 雛形ファイルURL
	private String		listId			;			// 帳票ID
	private String		outFileURL		= BASE_URL;	// 出力ファイルURL
	private String		outFilename		;			// 出力ファイル名
	private String[]	headerKeys		;			// 固定部の{@KEY} の KEY 部分を指定する。カンマで複数指定できる。
	private String[]	headerVals		;			// 固定部のKEY に対応する値を指定する。 {@KEY} に置き換わる。
	private String[]	footerKeys		;			// 繰り返し部の終了後に表示する key 部分を指定する。カンマで複数指定できる。
	private String[]	footerVals		;			// 繰り返し部の終了後に表示する key に対する値を指定する。
	private boolean		pageEndCut		= true;		// ページエンドカットをするか
	private boolean		useLocalResource= true;		// ローカルリソースを使用するか
	private boolean		useSheetName	;			// 5.7.6.2 (2014/05/16) PAGEBREAKカラムの値を、シート名として使うかどうか。

	private String		fgrun			= "P";		// PDF出力
	private String		printerName		;			// プリンタ名

	private String		tableId			= HybsSystem.TBL_MDL_KEY ;

	private transient DBTableModel body		;
	private transient DBTableModel header	;
	private transient DBTableModel footer	;

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();

		final int rtnCode;

		body = (DBTableModel)getObject( tableId );
		if( body == null || body.getRowCount() == 0 ) {
			rtnCode = SKIP_PAGE ; // ページの残りの処理を行わない。
		}
		else {
			exec();
			rtnCode = EVAL_PAGE ;
		}

		return rtnCode ;
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 5.7.6.2 (2014/05/16) PAGEBREAKカラムの値を、シート名として使うかどうか。
	 */
	@Override
	protected void release2() {
		super.release2();
		fileURL			= BASE_URL;
		listId			= null;
		outFileURL		= BASE_URL;
		outFilename		= null;
		headerKeys		= null;
		headerVals		= null;
		footerKeys		= null;
		footerVals		= null;
		pageEndCut		= true;
		useLocalResource= true;
		useSheetName	= false;	// 5.7.6.2 (2014/05/16) PAGEBREAKカラムの値を、シート名として使うかどうか。
		fgrun			= "P";
		printerName		= null;
		tableId			= HybsSystem.TBL_MDL_KEY ;
		body			= null;
		header			= null;
		footer			= null;
	}

	/**
	 * 帳票処理を行います。
	 *
	 * @og.rev 4.3.3.4 (2008/11/01) ヘッダー、フッター値が設定されていない場合にNullPointerExceptionが出るバグを修正
	 * @og.rev 4.3.3.4 (2008/11/01) 雛形のパス及び、出力先のパスを実ディレクトリのパスに変換
	 * @og.rev 5.7.6.2 (2014/05/16) PAGEBREAKカラムの値を、シート名として使うかどうか。
	 *
	 */
	private void exec()  {
		final QueueManager_DIRECT manager = new QueueManager_DIRECT();
		manager.setListId( listId );
		manager.setLang( getLanguage() );
		manager.setOutputName( HybsSystem.url2dir( outFileURL ) + outFilename );
		manager.setOutputType( fgrun );
		manager.setTemplateName( HybsSystem.url2dir( fileURL ) + listId );
		manager.setPrinterName( printerName );
		manager.setFgcut( pageEndCut );
		manager.setFglocal( useLocalResource );
		manager.setUseSheetName( useSheetName );		// 5.7.6.2 (2014/05/16) PAGEBREAKカラムの値を、シート名として使うかどうか。

		manager.setBody( body );

		// 4.3.3.4 (2008/11/01)
		if( headerVals != null && headerVals.length > 0 ) {
			String[][] hvals = new String[headerVals.length][1];
			hvals[0] = headerVals;
			header = DBTableModelUtil.makeDBTable( headerKeys, hvals, getResource() );
			manager.setHeader( header );
		}

		// 4.3.3.4 (2008/11/01)
		if( footerVals != null && footerVals.length > 0 ) {
			String[][] fvals = new String[footerVals.length][1];
			fvals[0] = footerVals;
			footer = DBTableModelUtil.makeDBTable( footerKeys, fvals, getResource() );
			manager.setFooter( footer );
		}

		manager.create();
		manager.waitExec();
	}

	/**
	 * 【TAG】雛型のHTMLファイルの保存してある ディレクトリを指定します。
	 *
	 * @og.tag
	 * この属性で指定されるディレクトリのファイルを読み取ります。
	 * 指定方法は、通常の fileURL 属性と同様に、先頭が、'/' (UNIX) または、２文字目が、
	 * ":" (Windows)の場合は、指定のURLそのままのディレクトリに、そうでない場合は、
	 * システムパラメータ の FILE_URL 属性で指定のフォルダの下に、作成されます。
	 * fileURL = "{&#064;USER.ID}" と指定すると、FILE_URL 属性で指定のフォルダの下に、
	 * さらに、各個人ID別のフォルダを作成して、そこを操作します。
	 *
	 * @param	url 雛型のHTMLファイルのディレクトリ
	 */
	public void setFileURL( final String url ) {
		String furl = nval( getRequestParameter( url ),null );
		if( furl != null ) {
			final char ch = furl.charAt( furl.length()-1 );
			if( ch != '/' && ch != '\\' ) { furl = furl + "/"; }
			fileURL = StringUtil.urlAppend( fileURL,furl );
		}
	}

	/**
	 * 【TAG】帳票IDを指定します。
	 *
	 * @og.tag
	 * 帳票IDを指定します。
	 *
	 * @param	listId	帳票ID
	 */
	public void setListId( final String listId ) {
		this.listId = nval( getRequestParameter( listId ), this.listId );
	}

	/**
	 * 【TAG】出力HTMLファイルの保存してあるディレクトリを指定します
	 * 		(初期値:FILE_URL[={@og.value org.opengion.hayabusa.common.SystemData#FILE_URL}])。
	 *
	 * @og.tag
	 * この属性で指定されるディレクトリにファイルを出力します。
	 * 指定方法は、通常の fileURL 属性と同様に、先頭が、'/' (UNIX) または、２文字目が、
	 * ":" (Windows)の場合は、指定のURLそのままのディレクトリに、そうでない場合は、
	 * システムパラメータ の FILE_URL 属性で指定のフォルダの下に、作成されます。
	 * fileURL = "{&#064;USER.ID}" と指定すると、FILE_URL 属性で指定のフォルダの下に、
	 * さらに、各個人ID別のフォルダを作成して、そこに出力します。
	 * (初期値:FILE_URL[={@og.value org.opengion.hayabusa.common.SystemData#FILE_URL}])。
	 *
	 * @param	url 出力HTMLファイルのディレクトリ
	 */
	public void setOutFileURL( final String url ) {
		String furl = nval( getRequestParameter( url ),null );
		if( furl != null ) {
			final char ch = furl.charAt( furl.length()-1 );
			if( ch != '/' && ch != '\\' ) { furl = furl + "/"; }
			outFileURL = StringUtil.urlAppend( outFileURL,furl );
		}
	}

	/**
	 * 【TAG】ファイルを作成するときの出力ファイル名をセットします。
	 *
	 * @og.tag
	 * ファイルを作成するときの出力ファイル名をセットします。
	 * 紙に印字する場合などファイルに出力しない場合は不要です。
	 *
	 * @param   filename 出力ファイル名
	 */
	public void setOutFilename( final String filename ) {
		this.outFilename = nval( getRequestParameter( filename ),this.outFilename );
	}

	/**
	 * 【TAG】固定部の{&#064;KEY} の KEY 部分をCSV形式で複数指定します。
	 *
	 * @og.tag
	 * カンマで複数指定できます。
	 * 分解方法は、CSV変数を先に分解してから、getRequestParameter で値を取得します。
	 * こうしないとデータ自身にカンマを持っている場合に分解をミスる為です。
	 *
	 * @param   hKeys 固定部のkey
	 */
	public void setHeaderKeys( final String hKeys ) {
		headerKeys = getCSVParameter( hKeys );
	}

	/**
	 * 【TAG】固定部のKEY に対応する値をCSV形式で複数指定します。
	 *
	 * @og.tag
	 * カンマで複数指定で、リクエスト情報でも設定できます。
	 * 分解方法は、CSV変数を先に分解してから、getRequestParameter で値を取得します。
	 * こうしないとデータ自身にカンマを持っている場合に分解をミスる為です。
	 *
	 * @param   hVals 固定部の値
	 */
	public void setHeaderVals( final String hVals ) {
		headerVals = getCSVParameter( hVals );
	}

	/**
	 * 【TAG】繰り返し部の終了後に表示する key 部分をCSV形式で複数指定します。
	 *
	 * @og.tag
	 * カンマで複数指定できます。
	 * 分解方法は、CSV変数を先に分解してから、getRequestParameter で値を取得します。
	 * こうしないとデータ自身にカンマを持っている場合に分解をミスる為です。
	 *
	 * @param   ftKeys 繰り返し部の終了後に表示するkey
	 */
	public void setFooterKeys( final String ftKeys ) {
		footerKeys = getCSVParameter( ftKeys );
	}

	/**
	 * 【TAG】固定部のKEY に対応する値をCSV形式で複数指定します。
	 *
	 * @og.tag
	 * カンマで複数指定で、リクエスト情報でも設定できます。
	 * 分解方法は、CSV変数を先に分解してから、getRequestParameter で値を取得します。
	 * こうしないとデータ自身にカンマを持っている場合に分解をミスる為です。
	 *
	 * @param   ftVals 繰り返し部の終了後に表示する値
	 */
	public void setFooterVals( final String ftVals ) {
		footerVals = getCSVParameter( ftVals );
	}

	/**
	 * 【TAG】ページエンドカットするかどうか[true:する/false:しない]を指定します(初期値:true:カットする)。
	 *
	 * @og.tag
	 * ページエンドカットとは、ボディー部(繰り返し部)がなくなったときに、それ以降の表示を打ち切る（カットする）
	 * 機能で、(true:カットする) を指定すると、それ以降を出力しません。
	 * 初期値は "true" (なくなった時点で、出力しない)です。
	 *
	 * @param   peCut ページ終了カット [true:する/false:しない]
	 */
	public void setPageEndCut( final String peCut ) {
		pageEndCut = nval( getRequestParameter( peCut ),pageEndCut );
	}

	/**
	 * 【TAG】各システムのリソース(ローカルリソース)を使用するかどうか[true:する/false:しない]を指定します(初期値:true)。
	 *
	 * @og.tag
	 * true の場合、各システムに登録されたリソース情報を使用して帳票データが変換されます。
	 * false の場合は、帳票デーモンが起動しているシステム(通常は'GE')のリソースが適用されます。
	 * 初期値は "true" (ローカルリソースを使用する)です。
	 *
	 * @param   fgl  ローカルリソースの使用 [true:する/false:しない]
	 */
	public void setUseLocalResource( final String fgl ) {
		useLocalResource = nval( getRequestParameter( fgl ),useLocalResource );
	}

	/**
	 * 【TAG】PAGEBREAKカラムの値を、シート名として使うかどうかをセットします(初期値:false)。
	 *
	 * @og.tag
	 * PAGEBREAK で、シートチェンジを行う場合、シート名も指定したい場合があります。
	 * その場合、この、useSheetName="true" とすることで、PAGEBREAKカラムの値を、シート名として
	 * 使用します。
	 * useSheetName="false" の場合は、"Page"+ページ番号+"_Row"+現在行番号 がシート名になります。
	 *
	 * PAGEBREAK は、FIRSTシート雛形にも適用されます。
	 * ちなみに、FIRSTシート雛形は、特殊で、useSheetName="false" の場合でも、
	 * FIRST_**** などと記述した場合は、**** 文字列をシート名に使用します。
	 * FIRST だけの場合は、従来と同じシート名になります。
	 * 初期値は、互換性を考慮し、false:シート名として使用しない です。
	 *
	 * @og.rev 5.7.6.2 (2014/05/16) 新規追加
	 *
	 * @param useSName PAGEBREAKカラムのシート名使用可否 [true:使用/false:使用しない]
	 */
	public void setUseSheetName( final String useSName ) {
		useSheetName = nval( getRequestParameter( useSName ),useSheetName );
	}

	/**
	 * 【TAG】出力方法を指定します(初期値:P(PDF出力))。
	 *
	 * @og.tag
	 * 出力方法のコードは、FGRUNのコードリソースと同じものが指定できます。
	 * 初期値は "P" (PDF出力)です。
	 *
	 * @param   code 出力方法(FGRUNのコードリソースと同じもの)
	 */
	public void setFgrun( final String code ) {
		fgrun= nval( getRequestParameter( code ),fgrun );
	}

	/**
	 * 【TAG】プリンター名を指定します。
	 *
	 * @og.tag
	 * プリンター名を指定します。このプリンター名は帳票サーバー上でのプリンタ名です。
	 * ファイル出力等、紙に印刷しない場合は不要です。
	 *
	 * @param   ptnm プリンター名
	 */
	public void setPrinterName( final String ptnm ) {
		printerName = nval( getRequestParameter( ptnm ),printerName );
	}

	/**
	 * 【TAG】(通常は使いません)結果のDBTableModelを、sessionに登録するときのキーを指定します
	 *		(初期値:HybsSystem#TBL_MDL_KEY[={@og.value org.opengion.hayabusa.common.HybsSystem#TBL_MDL_KEY}])。
	 *
	 * @og.tag
	 * 検索結果より、DBTableModelオブジェクトを作成します。これを、下流のviewタグ等に
	 * 渡す場合に、通常は、session を利用します。その場合の登録キーです。
	 * query タグを同時に実行して、結果を求める場合、同一メモリに配置される為、
	 * この tableId 属性を利用して、メモリ空間を分けます。
	 *		(初期値:HybsSystem#TBL_MDL_KEY[={@og.value org.opengion.hayabusa.common.HybsSystem#TBL_MDL_KEY}])。
	 *
	 * @param	id sessionに登録する時のID
	 */
	public void setTableId( final String id ) {
		tableId = nval( getRequestParameter( id ), tableId );
	}

	/**
	 * タグの名称を、返します。
	 * 自分自身のクラス名より、自動的に取り出せないため、このメソッドをオーバーライドします。
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) 新規追加
	 *
	 * @return  タグの名称
	 * @og.rtnNotNull
	 */
	@Override
	protected String getTagName() {
		return "report2" ;
	}

	/**
	 * シリアライズ用のカスタムシリアライズ書き込みメソッド。
	 *
	 * @og.rev 4.0.0.0 (2006/09/31) 新規追加
	 * @serialData 一部のオブジェクトは、シリアライズされません。
	 *
	 * @param	strm	ObjectOutputStreamオブジェクト
	 * @throws IOException	入出力エラーが発生した場合
	 */
	private void writeObject( final ObjectOutputStream strm ) throws IOException {
		strm.defaultWriteObject();
	}

	/**
	 * シリアライズ用のカスタムシリアライズ読み込みメソッド
	 *
	 * ここでは、transient 宣言された内部変数の内、初期化が必要なフィールドのみ設定します。
	 *
	 * @og.rev 4.0.0.0 (2006/09/31) 新規追加
	 * @serialData 一部のオブジェクトは、シリアライズされません。
	 *
	 * @param	strm	ObjectInputStreamオブジェクト
	 * @see #release2()
	 * @throws IOException	シリアライズに関する入出力エラーが発生した場合
	 * @throws ClassNotFoundException	クラスを見つけることができなかった場合
	 */
	private void readObject( final ObjectInputStream strm ) throws IOException , ClassNotFoundException {
		strm.defaultReadObject();
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION	)
				.println( "fileURL"			,fileURL			)
				.println( "listId"			,listId				)
				.println( "outFileURL"		,outFileURL			)
				.println( "outFilename"		,outFilename		)
				.println( "headerKeys"		,headerKeys			)
				.println( "headerVals"		,headerVals			)
				.println( "footerKeys"		,footerKeys			)
				.println( "footerVals"		,footerVals			)
				.println( "pageEndCut"		,pageEndCut			)
				.println( "useLocalResource",useLocalResource	)
				.println( "fgrun"			,fgrun				)
				.println( "printerName"		,printerName		)
				.println( "tableId"			,tableId			)
				.println( "BASE_URL"		,BASE_URL			)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
