/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.fukurou.util.Attributes;					// 6.1.1.0 (2015/01/17) TagBuffer でよい。
import org.opengion.fukurou.util.TagBuffer;						// 6.1.1.0 (2015/01/17)
import org.opengion.fukurou.util.XHTMLTag;
import org.opengion.fukurou.util.ToString;						// 6.1.1.0 (2015/01/17)
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.hayabusa.db.DBEditConfig;

import static org.opengion.fukurou.util.StringUtil.nval;

/**
 * query.jsp での検索ボタンを表示します。
 * 検索ボタン以外に、pageSize、maxRowCount、prevNext、timeView などの
 * 項目も表示可能です。
 * また、BODY部に登録した値は、table 要素として、td 部に追記されます。
 *
 * @og.formSample
 * ●形式：&lt;og:queryButton /&gt;
 * ●body：あり(EVAL_BODY_BUFFERED:BODYを評価し、{&#064;XXXX} を解析します)
 *
 * ●Tag定義：
 *   &lt;og:queryButton
 *       command            【TAG】リクエストとして送信するコマンドを指定します(初期値:NEW)
 *       pageSize           【TAG】pageSize プルダウンメニューの初期値を指定します (初期値:HTML_PAGESIZE[=100])
 *       maxRowCount        【TAG】maxRowCount プルダウンメニューの初期値を指定します (初期値:DB_MAX_ROW_COUNT[=1000])
 *       useCrear           【TAG】JavaScriptによる初期化ボタンを使用するかどうか[true/false]を指定します(初期値:true)
 *       type               【TAG】典型的なボタンの形状に設定します(初期値:null)
 *       scope              【TAG】リクエストとして送信するスコープを指定します(初期値:null)
 *       lbl                【TAG】ラベルリソースのラベルIDを指定します
 *       accesskey          【TAG】検索ボタンに使用するショートカットキー文字を指定します(初期値:F)
 *       usePrevNext        【TAG】進む／戻るのprev_next リンクを表示するかどうかを指定します(初期値:true)
 *       usePageSize        【TAG】pageSize 引数を使用するかどうかを指定します(初期値:true)
 *       useRowCount        【TAG】maxRowCount 引数を使用するかどうかを指定します(初期値:true)
 *       useTimeView        【TAG】処理時間を表示する TimeView を表示するかどうかを指定します(初期値:true)
 *       clearAccesskey     【TAG】クリアボタンに使用するショートカットキー文字を指定します(初期値:C)
 *       useDummy           【TAG】IE用ダミーinputを出力するかどうか
 *       useEditMenu        【TAG】編集プルダウンを出すかどうか(初期値:true)
 *       title              【HTML】要素に対する補足的情報(title)を設定します
 *       style              【HTML】この要素に対して適用させるスタイルシート(style)を設定します
 *       disabled           【HTML】その部品に対して、選択や変更が出来ないように(disabled)指定します(サーバーに送信されない)
 *       tabindex           【HTML】タブの移動順(tabindex)を指定します(0 ～ 32767)
 *       clazz              【HTML】要素に対して class 属性を設定します
 *       onClick            【HTML】JavaScriptのイベント onClick を設定します(例:onClick="renew('query.jsp','QUERY');")
 *       onBlur             【HTML】JavaScriptのイベント onBlur を設定します(例:onBlur="this.value=value.toUpperCase();")
 *       onFocus            【HTML】JavaScriptのイベント onFocus を設定します
 *       ondblClick         【HTML】JavaScriptのイベント ondblClick を設定します
 *       onMouseDown        【HTML】JavaScriptのイベント onMouseDown を設定します
 *       onMouseUp          【HTML】JavaScriptのイベント onMouseUp を設定します
 *       onMouseMove        【HTML】JavaScriptのイベント onMouseMove を設定します
 *       onMouseOut         【HTML】JavaScriptのイベント onMouseOut を設定します
 *       onMouseOver        【HTML】JavaScriptのイベント onMouseOver を設定します
 *       autofocus          【HTML5】指定した入力欄にカーソルが当たって自動的にフォーカスされます。
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   &gt;   ... Body ...
 *   &lt;/og:queryButton&gt;
 *
 * ●使用例
 *    &lt;og:queryButton
 *        command     = "NEW"      NEW:通常の検索(初期値)／ENTRY:エントリ登録時
 *        type        = "ENTRY"    null:標準 , SELECT:検索ボタンのみ , ENTRY:登録ボタンのみ
 *        pageSize    = "100"      pageSize の初期値設定(初期値:HTML_PAGESIZE)
 *        maxRowCount = "100"      maxRowCount の初期値設定(初期値:DB_MAX_ROW_COUNT)
 *        useCrear    = "true"     true:JavaScriptによる初期化／false:通常のresetボタン
 *        scope       = "request"  null(初期値)／session:セッション登録／request:リクエスト登録
 *        lbl         = "MSG0010"  検索ボタンに表示するメッセージID(初期値:MSG0010)
 *        accesskey   = "F"        ショートカットキー文字(初期値:F)
 *        usePrevNext = "true"     Prev-Nextリンクを使用するかどうか(初期値:true)
 *        usePageSize = "true"     pageSize 引数を使用するかどうか(初期値:true)
 *        useRowCount = "true"     maxRowCount 引数を使用するかどうか(初期値:true)
 *        useTimeView = "true"     timeView表示(実行時間表示)を使用するかどうか(初期値:true)
 *        useEditMenu = "true"     編集プルダウンを出すかどうか(初期値:true)
 *    /&gt;
 *
 *  標準的な検索ボタン
 *    &lt;og:queryButton /&gt;
 *
 *  標準的な登録ボタン(command="NEW" で、query.jsp で使用するボタン)
 *    &lt;og:queryButton type="ENTRY" /&gt;
 *
 *  BODY 部に追加要素を表示
 *    &lt;og:queryButton &gt;
 *        &lt;td&gt;&lt;jsp:directive.include file="/jsp/common/timeViewImage.jsp" /&gt;&lt;/td&gt;
 *    &lt;/og:queryButton &gt;
 *
 * @og.rev 4.0.0.0 (2005/01/31) 新規作成
 * @og.rev 5.7.1.0 (2013/12/06) HTMLTagSupport を継承し、ボタンに付ける属性を増やします。
 * @og.group 画面制御
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class QueryButtonTag extends HTMLTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "5.7.6.2 (2014/05/16)" ;

	private static final long serialVersionUID = 576220140516L ;

	private static final String COMMAND	=
		"<input type=\"hidden\" name=\"command\" value=\"NEW\" id=\"h_cmd\" />";

	private static final String JSP = HybsSystem.sys( "JSP" );

	// 4.2.1.0 (2008/04/07) TDにIDを追加
	// 4.3.1.0 (2008/09/03) 戻る・進むボタンをsubmitにより実行する。
	// 6.0.2.4 (2014/10/17) img タグに、title 属性追記
	private static final String PREV_NEXT =
		"<td id=\"prev-next\"><a href=\"forward.jsp?command=FIRST\" target=\"RESULT\" >"		+ CR +
		"  <img src=\"" + JSP + "/image/FIRST.gif\" alt=\"FIRST\" title=\"FIRST\" border=\"0px\" " + CR +
		"    onclick=\"return submitPrevNextForm('FIRST');\"/>"	+ CR +
		"</a>"																+ CR +
		"<a href=\"forward.jsp?command=PREV\" target=\"RESULT\" >"			+ CR +
		"  <img src=\"" + JSP + "/image/PREV.gif\" alt=\"PREV\" title=\"PREV\" border=\"0px\" "	+ CR +
		"    onclick=\"return submitPrevNextForm('PREV');\"/>"	+ CR +
		"</a>"																+ CR +
		"<a href=\"forward.jsp?command=NEXT\" target=\"RESULT\" >"			+ CR +
		"  <img src=\"" + JSP + "/image/NEXT.gif\" alt=\"NEXT\" title=\"NEXT\" border=\"0px\" "	+ CR +
		"    onclick=\"return submitPrevNextForm('NEXT');\"/>"	+ CR +
		"</a>"																+ CR +
		"<a href=\"forward.jsp?command=LAST\" target=\"RESULT\" >"			+ CR +
		"  <img src=\"" + JSP + "/image/LAST.gif\" alt=\"LAST\" title=\"LAST\" border=\"0px\" "	+ CR +
		"    onclick=\"return submitPrevNextForm('LAST');\"/>"	+ CR +
		"</a></td>" + CR ;

	private static final String TIME_VIEW =		// 4.2.1.0 (2008/04/07) 直接色指定をやめる。
		"<td><table cellspacing=\"0\" cellpadding=\"0\">"	+ CR +
		"<tr>"												+ CR +
		"<td width=\"20px\"> </td>"							+ CR +
		"<td width=\"0px\" id=\"queryTbl\" title=\"\" ></td>" + CR +	// #ff0000
		"<td width=\"0px\" id=\"viewTbl\"  title=\"\" ></td>" + CR +	// #0000ff
		"<td width=\"0px\" id=\"pageTbl\"  title=\"\" ></td>" + CR +	// #00ff00
		"<td width=\"10px\"></td>"						+ CR +
		"<td style=\"background-color:transparent;\" id=\"totalTbl\"></td>" + CR +
		"</tr>"											+ CR +
		"</table></td>" + CR ;

	// 4.3.6.0 (2009/04/01) 件数をボタンで開閉させる
	// 6.0.2.4 (2014/10/17) img タグに、title 属性追記
	private static final String COUNT_SWITCH=
		"<td>" +
		"  <img id=\"queryHide\" src=\"" + JSP + "/image/expand_r.png\" alt=\"pageSize\" title=\"pageSize\" border=\"0px\" "+	// 5.5.3.2 (2012/06/08)
				" onClick=\"hideClass( event, this, 'hide', 'expand_r', 'shrink_r' )\"> "	+ CR +
		"</td>";

	// 6.1.1.0 (2015/01/17) DUMMY_INPUT_TAG は、システム共通で固定値でよい。
	// 5.6.1.0 (2013/02/01) display:noneのダミー出力(IEのEnterボタンのための、隠しinput出力)
//	private static final String DUMMY_INPUT	= "HYBS_DUMMY_NPUT";
	private static final String DUMMY_INPUT_TAG =
		new TagBuffer( "input" )
			.add( "name"	, "HYBS_DUMMY_NPUT" )
			.add( "style"	, "display:none;" )
			.makeTag() ;

	// 5.7.6.2 (2014/05/16) IEのHTML5機能が有効か無効かの判定値を飛ばす、hidden タグを作成します。
	// 判定方法は、HTML5で新規追加された canvas タグが、有効かどうか(内部メソッドの戻り値があるかどうか)で判定します。
	private static final String IE_HTML5_CHECK = 
			"<script type=\"text/javascript\">" + CR
		+	"  (function($){" + CR
		+	"    var test = document.createElement('canvas');" + CR
		+	"    $.support.canvas = !(!test || !test.getContext);" + CR
		+	"  })(jQuery);" + CR
		+	"  $('<input/>').attr('type','hidden').attr('name','"
		+			HybsSystem.IE_HTML5_KEY
		+		"').val( $.support.canvas ).appendTo('form');" + CR
		+	"</script>" + CR ;

	private String	command		= "NEW";
	private String	pageSize	;
	private String	maxRowCount	;
	private String	accesskey	= "F";
	private boolean	useCrear	= true;
	private String	type		;			// null:標準 , SELECT:検索ボタンのみ , ENTRY:登録ボタンのみ
	private String	scope		;
	private boolean	usePrevNext	= true;
	private boolean	usePageSize	= true;
	private boolean	useRowCount	= true;
	private boolean	useTimeView	= true;
	private String	body		;
	private String	clearAcskey	= "C";
	private boolean useDummy	= HybsSystem.sysBool( "USE_DUMMY_INPUT" ); // 5.6.1.0 (2013/02/01)
	private boolean useEditMenu	= true;		// 5.7.5.0 (2014/04/04) 編集プルダウンを出すかどうか(初期値:true)。

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @return	後続処理の指示( EVAL_BODY_BUFFERED )
	 */
	@Override
	public int doStartTag() {
		return EVAL_BODY_BUFFERED ;	// Body を評価する。( extends BodyTagSupport 時)
	}

	/**
	 * Taglibのタグ本体を処理する doAfterBody() を オーバーライドします。
	 *
	 * @og.rev 2.2.0.0 (2002/12/17) 中国語(国際化)対応 エンコードの取得方法変更
	 * @og.rev 3.0.0.0 (2002/12/25) StringUtil#changeString 廃止
	 * @og.rev 3.1.1.0 (2003/03/28) ボディの内容を取得する処理を、CommonTagSupport で行う。
	 *
	 * @return	後続処理の指示(SKIP_BODY)
	 */
	@Override
	public int doAfterBody() {
		body = nval( getBodyString(),body );

		return SKIP_BODY ;
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 5.3.6.0 (2011/06/01) 集計、合計などのEdit機能に対応します。
	 * @og.rev 5.6.5.1 (2013/06/14) debugプルダウンの追加。ついでに、PageSize と MaxRowCount の 表示条件を、整備します。
	 * @og.rev 5.6.9.4 (2013/10/31) editNameの初期値を外部から与えられるようにする
	 * @og.rev 5.7.1.0 (2013/12/06) HTMLTagSupportを継承する為、メソッドを、getSubmit() ⇒ makeTag() に変更
	 * @og.rev 5.7.1.2 (2013/12/20) USE_DEFAULT_EDITNAMEでeditNameの初期値変更
	 * @og.rev 5.7.5.0 (2014/04/04) useEditMenu 追加
	 * @og.rev 5.7.6.2 (2014/05/16) IEのHTML5機能が有効か無効かの判定値を飛ばす、hidden タグを作成します。
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)
		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE )
			.append( getHiddenScope() ).append( CR )
			.append( "<table border=\"0\" frame=\"border\" summary=\"submit\" ><tr>" )
			.append( CR )
			.append( "<td>" ).append( makeTag()		).append( "</td>" ).append( CR )		// 5.7.1.0 (2013/12/06)
			.append( "<td>" ).append( getReset()	).append( "</td>" ).append( CR )
			.append( "<td>" ).append( getCommand()	).append( "</td>" ).append( CR )
			.append( "<td>" ).append( getGamenId()	).append( "</td>" ).append( CR );
		if( usePrevNext ) { buf.append( PREV_NEXT	).append( CR ); }
		buf.append( "<td><span id=\"query-button-append\"></span></td>").append( CR );		//4 3.6.0 (2009/04/01)
		// if( usePageSize ) { buf.append( getPageSize()	).append( CR ); }
		// if( useRowCount ) { buf.append( getMaxRowCount()).append( CR ); }
		if( body != null ) {
			buf.append( "<td>" ).append( body ).append( "</td>" ).append( CR );
		}

		// 5.7.5.0 (2014/04/04) useEditMenu 追加。trueで、編集プルダウンを出力します。
		if( useEditMenu ) {
			// 5.3.6.0 (2011/06/01) 集計、合計などのEdit機能に対応します。
			final String guikey = getGUIInfoAttri( "KEY" );
			// 6.1.0.0 (2014/12/26) findBugs: null ではなく長さが0の配列を返す。
			final DBEditConfig[] configs = getUser().getEditConfigs( guikey );
//			if( configs != null && configs.length > 0 ) {
			if( configs.length > 0 ) {
	//			String selEditName = getUser().getSelectedEdit( getGUIInfoAttri( "KEY" ) );
	//			String selEditName = nval( getRequestValue( "editName" ) , getUser().getSelectedEdit( getGUIInfoAttri( "KEY" ) )); // 5.6.9.4 (2013/10/31)
				String selEditName ; // 5.7.1.2 (2013/12/20)
				if( "FALSE".equalsIgnoreCase( sys( "USE_DEFAULT_EDITNAME" ) ) ){ 
					selEditName = getRequestValue( "editName" );
				}
				else{
					selEditName = nval( getRequestValue( "editName" ) , getUser().getSelectedEdit( getGUIInfoAttri( "KEY" ) )); // 5.6.9.4 (2013/10/31)

				}
				buf.append( getEditSelect( configs, selEditName ) ).append( CR );
			}
		}

		// 5.6.5.1 (2013/06/14) debugプルダウンの追加。ついでに、PageSize と MaxRowCount の 表示条件を、整備します。
		if( usePageSize || useRowCount) {
			buf.append( COUNT_SWITCH ).append( CR );
			if( usePageSize ) { buf.append( getPageSize()	).append( CR ); }
			if( useRowCount ) { buf.append( getMaxRowCount()).append( CR ); }
			// デバッグは、ユーザーのロールが、root の場合のみ表示するようにします。
			if( "root".equalsIgnoreCase( getUser().getRoles() ) ) {
				buf.append( getDebugClm() ).append( CR );						// 5.6.5.1 (2013/06/14) debugプルダウンの追加
			}
		}

		if( useTimeView ) { buf.append( TIME_VIEW ).append( CR ); }

		buf.append( "</tr></table>" ).append( CR );

		// 5.7.6.2 (2014/05/16) IEのHTML5機能が有効か無効かの判定値を飛ばす、hidden タグを作成します。
		// IE_HTML5_KEY をキーとする設定値がなければ、まだ未設定なので、hidden タグを作成します。
		if( getSessionAttribute( HybsSystem.IE_HTML5_KEY ) == null ) {
			buf.append( IE_HTML5_CHECK );
		}

		jspPrint( buf.toString() );
//		if( useDummy ){ jspPrint( getDummyInput() ); }	// 5.6.1.0 (2013/02/01)
		if( useDummy ){ jspPrint( DUMMY_INPUT_TAG ); }	// 6.1.1.0 (2015/01/17) 固定値でよい

		return EVAL_PAGE ;		// ページの残りを評価する。
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 5.6.1.0 (2013/02/01) useDummy 追加
	 * @og.rev 5.7.5.0 (2014/04/04) useEditMenu 追加
	 */
	@Override
	protected void release2() {
		super.release2();
		command		= "NEW";
		pageSize	= null;
		maxRowCount	= null;
		accesskey	= "F";
		useCrear	= true;
		type		= null;
		scope		= null;
		usePrevNext	= true;
		usePageSize	= true;
		useRowCount	= true;
		useTimeView	= true;
		clearAcskey	= "C";										// 5.3.5.0 (2011/05/01) クリアボタンアクセスキー対応
		useDummy	= HybsSystem.sysBool( "USE_DUMMY_INPUT" );	// 5.6.1.0 (2013/02/01)
		useEditMenu	= true;										// 5.7.5.0 (2014/04/04) 編集プルダウンを出すかどうか(初期値:true)。
	}

	/**
	 * サブミットボタンを作成します。
	 *
	 * @og.rev 5.3.6.0 (2011/06/01) accessKeyを削除できるように対応
	 * @og.rev 5.5.7.1 (2012/10/05) 不要な括弧出力を削除
	 * @og.rev 5.6.9.3 (2013/10/25) IDを出力する
	 * @og.rev 5.7.1.0 (2013/12/06) HTMLTagSupportを継承する為、メソッドを、getSubmit() ⇒ makeTag() に変更
	 *
	 * @return	サブミットボタン
	 * @og.rtnNotNull
	 */
	@Override
	protected String makeTag() {
		String msglbl = getLongLabel();		// 4.0.0 (2005/01/31)
		if( msglbl == null ) {
			if( "ENTRY".equalsIgnoreCase( type ) ) {
				msglbl = getLabel( "ENTRY" );
			}
			else {
				msglbl = getLabel( "QUERY" );
			}
		}

		// 5.7.1.0 (2013/12/06) XHTMLTag を使用することで、各種属性を追加できるようにします。
		if( accesskey != null ) {
			set( "accesskey"	, accesskey );
			msglbl = msglbl + "(" + accesskey + ")" ;
		}

		set( "type"	, "submit" );
		set( "id"	, "queryButtonSubmit" );
		set( "body"	, msglbl );

		return XHTMLTag.button( getAttributes() );
	}

	/**
	 * リセットボタンを作成します。
	 *
	 * @og.rev 4.0.0.0 (2007/10/18) メッセージリソース統合( getResource().getMessage ⇒ getResource().getLabel )
	 * @og.rev 5.0.2.0 (2009/11/01) 戻るボタンが出力されている状態で、クリアボタンを押すと、戻るボタンが消える不具合を修正
	 * @og.rev 5.3.5.0 (2011/05/01) クリアボタンにもアクセスキーが付加できるように対応
	 * @og.rev 5.3.6.0 (2011/06/01) accessKeyを削除できるように対応
	 *
	 * @return	リセットボタン
	 * @og.rtnNotNull
	 */
	private String getReset() {
		final String rtn ;

		if( useCrear ) {
			rtn = "<button onClick=\"window.open('" + JSP + "/" + getGUIInfoAttri("ADDRESS") +
				"/query.jsp?GAMENID=" + getGUIInfoAttri("KEY") + "&command=NEW','QUERY')\" " +
				( clearAcskey == null ? "" : "accesskey=\"" + clearAcskey + "\"" ) +
				">" +
				getResource().getLabel( "MSG0011" ) +
				( clearAcskey == null ? "" : "(" + clearAcskey + ")" ) +
				"</button>" ;
		}
		else {
			rtn = "<button type=\"reset\" " +
				( clearAcskey == null ? "" : "accesskey=\"" + clearAcskey + "\"" ) +
				">" +
				getResource().getLabel( "MSG0011" ) +
				( clearAcskey == null ? "" : "(" + clearAcskey + ")" ) +
				"</button>" ;
		}

		return rtn;
	}

	/**
	 * コマンドを出力する hidden を作成します。
	 *
	 * @return	コマンドを出力する hidden
	 * @og.rtnNotNull
	 */
	private String getCommand() {
		// 6.1.1.0 (2015/01/17) refactoring. ロジックを見直します。
//		final String rtn ;
//
//		if( "NEW".equals( command ) ) {
//			rtn = COMMAND;
//		}
//		else {
//			rtn = "<input type=\"hidden\" name=\"command\" value=\"" +
//						command + "\" id=\"h_cmd\" />" ;
//		}
//
//		return rtn;

		return "NEW".equals( command )
					? COMMAND
					: "<input type=\"hidden\" name=\"command\" value=\"" +
						command + "\" id=\"h_cmd\" />" ;
	}

	/**
	 * スコープを出力する hidden を作成します。
	 *
	 * @return	スコープを出力する hidden
	 * @og.rtnNotNull
	 */
	private String getHiddenScope() {
		// 6.1.1.0 (2015/01/17) refactoring. ロジックを見直します。
//		String rtn = "";
//
//		if( scope != null ) {
//			rtn = "<input type=\"hidden\" name=\"scope\" value=\"" +
//						scope + "\" />" ;
//		}
//
//		return rtn;

		return scope == null
				? ""
				: "<input type=\"hidden\" name=\"scope\" value=\"" +
						scope + "\" />" ;
	}

	/**
	 * 画面IDを出力する hidden を作成します。
	 *
	 * @return	画面IDを出力する hidden
	 * @og.rtnNotNull
	 */
	private String getGamenId() {
		// 6.1.1.0 (2015/01/17) refactoring. ロジックを見直します。
//		final String rtn = "<input type=\"hidden\" name=\"GAMENID\" value=\"" +
//						getGUIInfoAttri("KEY") + "\" />";
//
//		return rtn;
		return "<input type=\"hidden\" name=\"GAMENID\" value=\"" +
						getGUIInfoAttri("KEY") + "\" />";
	}

	/**
	 * ページサイズプルダウンメニューを作成します。
	 *
	 * @og.rev 4.3.6.0 (2008/04/01) idとclassの変更
	 *
	 * @return	ページサイズプルダウンメニュー
	 * @og.rtnNotNull
	 */
	private String getPageSize() {
		if( pageSize == null ) {
			final String PAGESIZE  = sys( "HTML_PAGESIZE" );
			pageSize = nval( getRequestValue( "pageSize" ),PAGESIZE ) ;
		}

		final DBColumn column = getDBColumn( "pageSize" );
		final String tag = column.getEditorValue( pageSize );

		// return ( "<td id=\"label\">" +
		return "<td class=\"label pageSize hide\">"					// 4.3.6.0 (2008/04/01) class出力
				+	column.getLongLabel()
				+	":</td><td class=\"pageSize hide\">" + tag + "</td>" ;
	}

	/**
	 * 検索時の最大検索数のプルダウンメニューを作成します。
	 *
	 * @og.rev 4.3.6.0 (2008/04/01) idとclassの変更
	 *
	 * @return	最大検索数のプルダウンメニュー
	 * @og.rtnNotNull
	 */
	private String getMaxRowCount() {
		if( maxRowCount == null ) {
			final String ROW_COUNT = sys( "DB_MAX_ROW_COUNT" );
			maxRowCount = nval( getRequestValue( "maxRowCount" ), ROW_COUNT ) ;
		}

		final DBColumn column = getDBColumn( "maxRowCount" );
		final String tag = column.getEditorValue( maxRowCount );

		// return ( "<td id=\"label\">" +
		return "<td class=\"label maxRowCount hide\">"			// 4.3.6.0 (2008/04/01) class出力
				+	column.getLongLabel()
				+	":</td><td class=\"maxRowCount hide\">" + tag + "</td>" ;
	}

	/**
	 *  debugプルダウンメニューを作成します。
	 *
	 * @og.rev 5.6.5.1 (2013/06/14) 新規追加
	 *
	 * @return	debugプルダウンメニュー
	 * @og.rtnNotNull
	 */
	private String getDebugClm() {
		final DBColumn column = getDBColumn( "debug" );
		final String tag = column.getEditorValue( "false" );

		return "<td class=\"label debug hide\">"
				+	column.getLongLabel()
				+	":</td><td class=\"debug hide\">" + tag + "</td>" ;
	}

	/**
	 * 編集名一覧のプルダウンメニューを作成します。
	 *
	 * @og.rev 5.3.6.0 (2011/06/01) 新規追加
	 *
	 * @param configs 編集名一覧(配列)
	 * @param selEditName 選択中の編集名
	 *
	 * @return	編集名一覧のプルダウン
	 * @og.rtnNotNull
	 */
	private String getEditSelect( final DBEditConfig[] configs, final String selEditName ) {
		final DBColumn column = getDBColumn( "editName" );

		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE )
			.append( "<td class=\"label editName\">" )
			.append( column.getLongLabel() )
			.append( ":</td><td class=\"editName\">" )
			.append( "<select name=\"editName\">" )
			.append( "<option />" );
		for( final DBEditConfig config : configs ) {
			final String name = config.getEditName();
			if( name != null && name.length() > 0 ) {
				buf.append( "<option value=\"" ).append( name ).append( '"' );		// 6.0.2.5 (2014/10/31) char を append する。
				if( config.isCommon() ) {
					buf.append( " class=\"commonEdit\"" );
				}
				if( name.equals( selEditName ) ) {
					buf.append( " selected" );
				}
				buf.append( '>' ).append( name ).append( "</option>" );				// 6.0.2.5 (2014/10/31) char を append する。
			}
		}
		buf.append( "</select></td>" );
		return buf.toString();
	}

//	/**
//	 * IEのEnterボタンのための、隠しinput出力。
//	 *
//	 * ※内部コール用
//	 *
//	 * @og.rev 5.6.1.0 (2013/02/01) 新規作成
//	 * @og.rev 6.1.1.0 (2015/01/17) static final String DUMMY_INPUT_TAG で定義するので、廃止
//	 *
//	 * @return	隠し属性タグ
//	 * @og.rtnNotNull
//	 */
//	private final String getDummyInput() {
//
//		final StringBuilder strRet = new StringBuilder( BUFFER_MIDDLE );
//		final Attributes attributes = new Attributes();
//		attributes.set("name",DUMMY_INPUT);
//		attributes.set("value",null);
//		attributes.set("optionAttributes","style='display:none;'");
//		strRet.append( XHTMLTag.input( attributes ) );
//
//		return strRet.toString();
//	}

	/**
	 * 【TAG】リクエストとして送信するコマンドを指定します(初期値:NEW)。
	 *
	 * @og.tag
	 *  NEW:通常の検索(初期値)／ENTRY:エントリ登録時
	 *
	 * @param	cmd	送信するコマンド
	 */
	public void setCommand( final String cmd ) {
		command = nval( getRequestParameter( cmd ),command );
	}

	/**
	 * 【TAG】JavaScriptによる初期化ボタンを使用するかどうか[true/false]を指定します(初期値:true)。
	 *
	 * @og.tag
	 * 通常の初期化ボタンでは、HTMLが作成された状態に戻すだけです。例えば、
	 * 引数付きで設定された場合、その引数が設定されている状態へ戻します。
	 * このJavaScriptによる初期化ボタンの場合(trueの場合)は、JavaScriptにより
	 * INIT状態でもう一度QUERYフレームを呼び出しますので、指定された引数も
	 * クリアされます。
	 * 初期値は、true:JavaScriptによる初期化です。
	 *
	 * @param   flag JavaScriptによる初期化ボタン使用可否 [true:JavaScript初期化ボタン/false:通常のresetボタン]
	 */
	public void setUseCrear( final String flag ) {
		useCrear = nval( getRequestParameter( flag ),useCrear );
	}

	/**
	 * 【TAG】典型的なボタンの形状に設定します(初期値:null)。
	 *
	 * @og.tag
	 * クエリーボタンでは、検索ボタン以外に、pageSize、maxRowCount、
	 * prevNext、timeView などの項目も表示可能です。
	 * これらを、個別に設定可能ですが、通常使用するタイプとして、３タイプを
	 * 用意します。これは、設定状態の初期値といえます。
	 * null  :通常の検索ボタンで、フル装備の指定です。
	 * SELECT:１件のみ検索するようなケースで使用します。
	 *        検索ボタンとクリアボタンのみを表示します。
	 * ENTRY :いきなり登録するようなケースで使用します。
	 *        登録ボタンとクリアボタンのみを表示します。(command="NEW" です。)
	 *
	 * @param	tp	タイプ指定 [null/SELECT/ENTRY]
	 */
	public void setType( final String tp ) {
		type = nval( getRequestParameter( tp ),type );

		if( "SELECT".equalsIgnoreCase( type ) ) {
			accesskey	= "F";
			usePrevNext	= false;
			usePageSize	= false;
			useRowCount	= false;
			useTimeView	= false;
		}
		else if( "ENTRY".equalsIgnoreCase( type ) ) {
			accesskey	= "W";
			usePrevNext	= false;
			usePageSize	= false;
			useRowCount	= false;
			useTimeView	= false;
		}
	}

	/**
	 * 【TAG】リクエストとして送信するスコープを指定します(初期値:null)。
	 *
	 * @og.tag
	 * リクエスト時に、付与する引数情報のうちの、スコープを指定します。
	 * これは、検索専用画面の場合は、scope="request" にする事で、メモリの
	 * 無駄な確保を行わずに、検索結果を表示できます。
	 * ただし、その結果に対して、NEXT-PREV や、ヘッダーソートリンクなどの
	 * 機能が使えなくなります。
	 * 通常は、何も指定しない状態(null)で、RESULT画面の設定に任せるのが良いでしょう。
	 * 初期値は、null = session(セッション登録) です。
	 *
	 * @param   scp スコープ [session:セッション登録/request:リクエスト登録]
	 */
	@Override
	public void setScope( final String scp ) {
		scope = nval( getRequestParameter( scp ),scope );
	}

	/**
	 * 【TAG】検索ボタンに使用するショートカットキー文字を指定します(初期値:F)。
	 *
	 * @og.tag
	 * ボタンを、Alt-XX で実行できるようにするショートカットキーを設定します。
	 * 標準では、検索ボタンは、"F" に設定していますので、Alt-F となります。
	 * ゼロ文字列を指定すると、ショートカットキーなしに設定できます。
	 * 初期値は、Find の "F" です。
	 *
	 * @og.rev 5.3.6.0 (2011/06/01) accessKeyを削除できるように対応
	 * @og.rev 5.6.0.3 (2012/01/24) getRequestParameter を何度も呼んでいるので修正。
	 *
	 * @param	accsKey	ショートカットキー文字(初期値:F)
	 */
	@Override
	public void setAccesskey( final String accsKey ) {
		final String val = getRequestParameter( accsKey );
		if( val == null || val.isEmpty() ) {
			accesskey = null;
		}
		else {
			accesskey = val;								// 5.6.0.3 (2012/01/24)
		}
	}

	/**
	 * 【TAG】クリアボタンに使用するショートカットキー文字を指定します(初期値:C)。
	 *
	 * @og.tag
	 * ボタンを、Alt-XX で実行できるようにするショートカットキーを設定します。
	 * 標準では、検索ボタンは、"C" に設定していますので、Alt-C となります。
	 * ゼロ文字列を指定すると、ショートカットキーなしに設定できます。
	 * 初期値は、Clear の "C" です。
	 *
	 * @og.rev 5.3.6.0 (2011/06/01) accessKeyを削除できるように対応
	 * @og.rev 5.6.0.3 (2012/01/24) getRequestParameter を何度も呼んでいるので修正。
	 *
	 * @param	accsKey	クリアボタンのショートカットキー文字(初期値:C)
	 */
	public void setClearAccesskey( final String accsKey ) {
		final String val = getRequestParameter( accsKey );
		if( val == null || val.isEmpty() ) {
			clearAcskey = null;
		}
		else {
			clearAcskey = val;								// 5.6.0.3 (2012/01/24)
		}
	}

	/**
	 * 【TAG】進む／戻るのprev_next リンクを表示するかどうかを指定します(初期値:true)。
	 *
	 * @og.tag
	 * 進む／戻るリンクは、RESULT画面にコマンド (FIRST,PREV,NEXT,LAST) という
	 * コマンドを送る事で実現している画像リンクです。
	 * PREV_NEXT が存在しない画面では、消去しておく必要があります。
	 * 初期値は、true(表示する)です。
	 *
	 * @param	flag	prev_nextリンク表示 [true:する/false:しない]
	 */
	public void setUsePrevNext( final String flag ) {
		usePrevNext = nval( getRequestParameter( flag ),usePrevNext );
	}

	/**
	 * 【TAG】pageSize 引数を使用するかどうかを指定します(初期値:true)。
	 *
	 * @og.tag
	 * pageSize 引数は、RESULT画面のviewタグで使用する 一回の表示に
	 * 何件のデータを表示させるかを指定する引数です。
	 * この件数が多いと、多くのデータを見ることが出来ますが、その分表示が
	 * 遅くなります。逆に、少ないと、PREV_NEXT を多用することになり、
	 * 目的の行を探し出しにくくなります。
	 * pageSize の値は、pageSizeパラメータで画面単位に指定するか、システム
	 * パラメータの HTML_PAGESIZE でシステム単位に指定できます。
	 * 初期値は、true(使用する)です。
	 *
	 * @param	flag	pageSize引数使用 [true:する/false:しない]
	 * @see     #setPageSize( String )
	 */
	public void setUsePageSize( final String flag ) {
		usePageSize = nval( getRequestParameter( flag ),usePageSize );
	}

	/**
	 * 【TAG】maxRowCount 引数を使用するかどうかを指定します(初期値:true)。
	 *
	 * @og.tag
	 * maxRowCount 引数は、RESULT画面のqueryタグで使用する 一回の検索時に
	 * 何件のデータを取得させるかを指定する引数です。
	 * この件数が多いと、多くのデータを検索することが出来ますが、その分表示が
	 * 遅くなります。逆に、少ないと、目的の行まで探し出せないかもしれません。
	 * maxRowCount の値は、rowCountパラメータで画面単位に指定するか、システム
	 * パラメータの DB_MAX_ROW_COUNT でシステム単位に指定できます。
	 *
	 * 初期値は、true(使用する)です。
	 *
	 * @param	flag	maxRowCount引数使用 [true:する/false:しない]
	 * @see     #setMaxRowCount( String )
	 */
	public void setUseRowCount( final String flag ) {
		useRowCount = nval( getRequestParameter( flag ),useRowCount );
	}

	/**
	 * 【TAG】処理時間を表示する TimeView を表示するかどうか[true:する/false:しない]を指定します(初期値:true)。
	 *
	 * @og.tag
	 * true に設定すると、処理時間を表示するバーイメージが表示されます。
	 * これは、DB検索、APサーバー処理、画面表示の各処理時間をバーイメージで
	 * 表示させる機能です。処理時間の目安になります。
	 * 初期値は、true です。
	 *
	 * @param	flag	処理時間を表示 [true:する/false:しない]
	 */
	public void setUseTimeView( final String flag ) {
		useTimeView = nval( getRequestParameter( flag ),useTimeView );
	}

	/**
	 * 【TAG】pageSize プルダウンメニューの初期表示件数を指定します
	 *		(初期値:HTML_PAGESIZE[={@og.value org.opengion.hayabusa.common.SystemData#HTML_PAGESIZE}])。
	 *
	 * @og.tag
	 * プルダウンメニューを表示する場合の初期値を指定します。
	 * (初期値:ユーザー定数のHTML_PAGESIZE[={@og.value org.opengion.hayabusa.common.SystemData#HTML_PAGESIZE}])。
	 *
	 * @param	pSize	表示件数
	 * @see     #setUsePageSize( String )
	 * @see		org.opengion.hayabusa.common.SystemData#HTML_PAGESIZE
	 */
	public void setPageSize( final String pSize ) {
		pageSize = nval( getRequestParameter( pSize ),pageSize );
	}

	/**
	 * 【TAG】maxRowCount プルダウンメニューの初期値を指定します
	 *		(初期値:DB_MAX_ROW_COUNT[={@og.value org.opengion.hayabusa.common.SystemData#DB_MAX_ROW_COUNT}])。
	 *
	 * @og.tag
	 * プルダウンメニューを表示する場合のmaxRowCountの初期値を指定します。
	 * (初期値:ユーザー定数のDB_MAX_ROW_COUNT[={@og.value org.opengion.hayabusa.common.SystemData#DB_MAX_ROW_COUNT}])。
	 *
	 * @param	maxSize	最大件数
	 * @see     #setUseRowCount( String )
	 * @see		org.opengion.hayabusa.common.SystemData#DB_MAX_ROW_COUNT
	 */
	public void setMaxRowCount( final String maxSize ) {
		maxRowCount = nval( getRequestParameter( maxSize ),maxRowCount );
	}

	/**
	 * 【TAG】ダミーのinputタグ(display:none)を出力するか設定します
	 *		(初期値:USE_DUMMY_INPUT[={@og.value org.opengion.hayabusa.common.SystemData#USE_DUMMY_INPUT}])。
	 *
	 * @og.tag
	 * ダミーのinputタグ(display:none)を出力するか設定します。
	 * IEではinputが１つだけの場合にエンターでのsubmitが上手くいかないため、ダミーを配置する必要があります。
	 * trueに設定するとWriteCheckタグがダミーを（常に）出力します。
	 * (初期値:USE_DUMMY_INPUT[={@og.value org.opengion.hayabusa.common.SystemData#USE_DUMMY_INPUT}])。
	 *
	 * @og.rev 5.6.1.0 (2013/02/01) 新規追加
	 *
	 * @param   flag 出力可否 [true:する/false:しない]
	 * @see		org.opengion.hayabusa.common.SystemData#USE_DUMMY_INPUT
	 */
	public void setUseDummy( final String flag ) {
		useDummy = nval( getRequestParameter( flag ), useDummy );
	}

	/**
	 * 【TAG】query画面等で編集プルダウンを出力するかどうか[true:する/false:しない]を指定します(初期値:true)。
	 *
	 * @og.tag
	 * falseを指定すると編集プルダウンを出さない。
	 * popup画面等、画面ID単位では編集機能は使うが、部分的に使いたくない場合に
	 * タグのパラメータとして個別に指定できるようにします。
	 * 初期値は、互換性の関係から、編集プルダウンを出力する(=true)です。
	 *
	 * @og.rev 5.7.5.0 (2014/04/04) 新規追加
	 *
	 * @param	flag	編集メニュー出力 [true:する/false:しない]
	 * @see		org.opengion.hayabusa.taglib.FileUDTag#setUseEditLink(String)
	 */
	public void setUseEditMenu( final String flag ) {
		useEditMenu = nval( getRequestParameter( flag ),useEditMenu );
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION		)
				.println( "command"		,command		)
				.println( "pageSize"	,pageSize		)
				.println( "maxRowCount"	,maxRowCount	)
				.println( "accesskey"	,accesskey		)
				.println( "scope"		,scope			)
				.println( "useCrear"	,useCrear		)
				.println( "usePrevNext"	,usePrevNext	)
				.println( "usePageSize"	,usePageSize	)
				.println( "useRowCount"	,useRowCount	)
				.println( "useTimeView"	,useTimeView	)
				.println( "body"		,body			)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
