/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.view;

import java.util.List;

import org.opengion.fukurou.util.StringUtil;
// import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.html.TableFormatter;

/**
 * ヘッダー部分のフォーマットに応じたテーブルを自動作成する、フォーマットテーブル作成クラスです。
 *
 * AbstractViewForm により、setter/getterメソッドのデフォルト実装を提供しています。
 * 各HTMLのタグに必要な setter/getterメソッドのみ，追加定義しています。
 * [XXXX]は、カラムを指定します。ラベル＋入力フィールドをそれぞれtdで囲います。
 * [#XXXX]は、対応するカラムのラベルを出力します。
 * [$XXXX]は、対応するカラムのレンデラーを出力します。
 * [!XXXX]は、対応するカラムの値を出力します。
 * 特殊記号の解釈は、HTMLFormatTextField系とHTMLFormatTable系で異なりますので
 * ご注意ください。
 *
 * AbstractViewForm を継承している為,ロケールに応じたラベルを出力させる事が出来ます。
 *
 * @og.group 画面表示
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class ViewForm_HTMLFormatTable extends ViewForm_HTMLTable  {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "5.1.6.0 (2010/05/01)" ;

	// 3.5.4.0 (2003/11/25) TableFormatter クラス追加
	private TableFormatter format	;

	/**
	 * DBTableModel から HTML文字列を作成して返します。
	 * startNo(表示開始位置)から、pageSize(表示件数)までのView文字列を作成します。
	 * 表示残りデータが pageSize 以下の場合は,残りのデータをすべて出力します。
	 *
	 * @og.rev 3.5.0.0 (2003/09/17) BODY要素の noClass 属性を追加。
	 * @og.rev 3.5.0.0 (2003/09/17) &lt;tr&gt;属性は、元のフォーマットのまま使用します。
	 * @og.rev 3.5.2.0 (2003/10/20) ヘッダー繰り返し属性( headerSkipCount )を採用
	 * @og.rev 3.5.3.1 (2003/10/31) skip属性を採用。headerLine のキャッシュクリア
	 * @og.rev 3.5.4.0 (2003/11/25) TableFormatter クラスを使用するように変更。
	 * @og.rev 3.5.5.0 (2004/03/12) systemFormat(例：[KEY.カラム名]形式等)の対応
	 * @og.rev 3.5.5.0 (2004/03/12) No 欄そのものの作成判断ロジックを追加
	 * @og.rev 3.5.5.7 (2004/05/10) [#カラム名] , [$カラム名] に対応
	 * @og.rev 3.5.6.0 (2004/06/18) '!' 値のみ 追加 既存の '$' は、レンデラー
	 * @og.rev 3.5.6.4 (2004/07/16) ヘッダーとボディー部をJavaScriptで分離
	 * @og.rev 3.7.0.3 (2005/03/01) getBgColorCycleClass に、選択行マーカーを採用
	 * @og.rev 4.0.0.0 (2005/01/31) 新規作成(getColumnClassName ⇒ getColumnDbType)
	 * @og.rev 4.3.1.0 (2008/09/08) フォーマットが設定されていない場合のエラー追加・編集行のみを表示する属性(isSkipNoEdit)追加
	 * @og.rev 4.3.3.0 (2008/10/01) noTransition属性対応
	 * @og.rev 4.3.7.4 (2009/07/01) tbodyタグの入れ子を解消(FireFox対応)
	 *
	 * @param  startNo    表示開始位置
	 * @param  pageSize   表示件数
	 *
	 * @return  DBTableModelから作成された HTML文字列
	 */
	@Override
	public String create( final int startNo, final int pageSize )  {
		if( getRowCount() == 0 ) { return ""; }	// 暫定処置

		// 4.3.1.0 (2008/09/08)
		if( format == null ) {
			final String errMsg = "ViewTagで canUseFormat() = true の場合、Formatter は必須です。";
			throw new HybsSystemException( errMsg );
		}

		headerLine	 = null;		// 3.5.3.1 (2003/10/31) キャッシュクリア
		final int lastNo = getLastNo( startNo, pageSize );
		final int blc = getBackLinkCount();
		final int hsc = getHeaderSkipCount();		// 3.5.2.0 (2003/10/20)
		int hscCnt = 1;						// 3.5.2.0 (2003/10/20)

		final StringBuilder out = new StringBuilder( BUFFER_LARGE )
			.append( getCountForm( startNo,pageSize ) )
			.append( getHeader() );

		final String ckboxTD = "<td " + format.getRowspan() + ">";
		int bgClrCnt = 0;
		for( int row=startNo; row<lastNo; row++ ) {
			if( isSkip( row ) || isSkipNoEdit( row ) ) { continue; } // 4.3.1.0 (2008/09/08)
			if( ! format.isUse( row,getDBTableModel() ) ) { continue; }		// 3.5.4.0 (2003/11/25)
			out.append("<tbody").append( getBgColorCycleClass( bgClrCnt++,row ) );
			if( isNoTransition() ) {	// 4.3.3.0 (2008/10/01)
				out.append( getHiddenRowValue( row ) );
			}
			out.append('>')			// 6.0.2.5 (2014/10/31) char を append する。
				.append( format.getTrTag() );

			// 3.5.5.0 (2004/03/12) No 欄そのものの作成判断追加
			if( isNumberDisplay() ) {
				out.append( makeCheckbox( ckboxTD,row,blc ) );
			}

			int cl = 0;
			for( ; cl < format.getLocationSize(); cl++ ) {
				String fmt = format.getFormat(cl);				// 3.5.0.0
				final int loc = format.getLocation(cl);			// 3.5.5.0
				if( ! format.isNoClass() && loc >= 0 ) {		// 3.5.5.7 (2004/05/10)
//					final StringBuilder newtg = new StringBuilder( BUFFER_LARGE )
//						.append("<td class=\"")
//						.append( getColumnDbType(loc) )			// 4.0.0 (2005/01/31)
//						.append("\" ");
////					final String tdclass = newtg.toString();
////					fmt = StringUtil.replace( format.getFormat(cl) ,"<td", tdclass );
//					fmt = StringUtil.replace( format.getFormat(cl) ,"<td", newtg.toString() );
					final String tdclass = "<td class=\"" + getColumnDbType(loc) + "\" " ;
					fmt = StringUtil.replace( format.getFormat(cl) ,"<td", tdclass );
				}
				out.append( fmt );			// 3.5.0.0
				// 3.5.5.7 (2004/05/10) #,$ 対応
				if( loc >= 0 ) {
					switch( format.getType(cl) ) {
						case '#' : out.append( getColumnLabel(loc) );		break;
						case '$' : out.append( getRendererValue(row,loc) );	break;
						case '!' : out.append( getValue(row,loc) );			break;
						default  : out.append( getValueLabel(row,loc) );	break;
					}
				}
				else {
					out.append( format.getSystemFormat(row,loc) );
				}
			}
			out.append( format.getFormat(cl) )
				.append("</tbody>").append( CR );

		// 3.5.2.0 (2003/10/20) ヘッダー繰り返し属性( headerSkipCount )を採用
			if( hsc > 0 && hscCnt % hsc == 0 ) {
				out.append("<tbody class=\"row_h\" >")
					.append( getHeadLine() )
					.append("</tbody>");
				hscCnt = 1;
			}
			else {
				hscCnt ++ ;
			}
		}
		out.append("</table>").append( CR )
			.append( getScrollBarEndDiv() );	// 3.8.0.3 (2005/07/15)

		return out.toString();
	}

	/**
	 * 内容をクリア(初期化)します。
	 *
	 * @og.rev 3.1.1.0 (2003/03/28) 同期メソッド(synchronized付き)を非同期に変更する。
	 * @og.rev 3.5.0.0 (2003/09/17) Noカラムに、表示を全て消せるように、class 属性を追加。
	 * @og.rev 3.5.4.0 (2003/11/25) TableFormatter クラスを使用するように変更。
	 *
	 */
	@Override
	public void clear() {
		super.clear();
		format          = null;
	}

	/**
	 * DBTableModel から テーブルのタグ文字列を作成して返します。
	 *
	 * @og.rev 3.2.4.0 (2003/06/12) makeFormat() する位置を移動。
	 * @og.rev 3.5.0.0 (2003/09/17) &lt;tr&gt;属性は、元のフォーマットのまま使用します。
	 * @og.rev 3.5.1.0 (2003/10/03) Noカラムに、numberType 属性を追加
	 * @og.rev 3.5.2.0 (2003/10/20) ヘッダー繰り返し部をgetHeadLine()へ移動
	 * @og.rev 3.5.3.1 (2003/10/31) VERCHAR2 を VARCHAR2 に修正。
	 * @og.rev 3.5.5.0 (2004/03/12) No 欄そのものの作成判断ロジックを追加
	 * @og.rev 3.5.6.5 (2004/08/09) thead に、id="header" を追加
	 * @og.rev 4.0.0.0 (2005/01/31) DBColumn の 属性(CLS_NM)から、DBTYPEに変更
	 *
	 * @return  テーブルのタグ文字列
	 */
	@Override
	protected String getTableHead() {

		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );
		// 3.5.5.0 (2004/03/12) No 欄そのものの作成判断追加
		if( isNumberDisplay() ) {
			buf.append("<colgroup class=\"X\" /><colgroup class=\"BIT\" /><colgroup class=\"S9\" />")
				.append(CR);
		}

	// 3.5.2.0 (2003/10/20) ヘッダー繰り返し部をgetHeadLine()へ移動
		buf.append("<thead id=\"header\">").append( CR )	// 3.5.6.5 (2004/08/09)
			.append( getHeadLine() )
			.append("</thead>").append( CR );

		return buf.toString();
	}

	/**
	 * ヘッダー繰り返し部を、getTableHead()メソッドから分離。
	 *
	 * @og.rev 3.5.2.0 (2003/10/20) 新規作成
	 * @og.rev 3.5.4.0 (2003/11/25) TableFormatter クラスを使用するように変更。
	 * @og.rev 3.5.4.3 (2004/01/05) useCheckControl 属性の機能を追加
	 * @og.rev 3.5.4.6 (2004/01/30) numberType="none" 時の処理を追加(Noラベルを出さない)
	 * @og.rev 3.5.4.7 (2004/02/06) ヘッダーにソート機能用のリンクを追加します。
	 * @og.rev 3.5.5.0 (2004/03/12) systemFormat(例：[KEY.カラム名]形式等)の対応
	 * @og.rev 3.5.5.0 (2004/03/12) No 欄そのものの作成判断ロジックを追加
	 * @og.rev 3.7.0.1 (2005/01/31) 全件チェックコントロール処理変更
	 *
	 * @return	テーブルのタグ文字列
	 */
	@Override
	protected String getHeadLine() {
		if( headerLine != null ) { return headerLine; }		// キャッシュを返す。

		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE )
			.append( format.getTrTag() ).append( CR );

		// 3.5.5.0 (2004/03/12) No 欄そのものの作成判断追加
		if( isNumberDisplay() ) {
			// 3.5.4.3 (2004/01/05) 追加分
			if( isUseCheckControl() && "checkbox".equals( getSelectedType() ) ) {
				buf.append("<th" ).append( format.getRowspan() )
					.append("></th><th" ).append( format.getRowspan() )
					.append('>').append( getAllCheckControl() )
					.append("</th><th" ).append( format.getRowspan() )
					.append('>').append( getNumberHeader() ).append("</th>");
			}
			else {
				buf.append("<th colspan=\"3\"")
					.append( format.getRowspan() )
					.append('>').append( getNumberHeader() ).append("</th>");	// 6.0.2.5 (2014/10/31) char を append する。
			}
		}

		int cl = 0;
		for( ; cl < format.getLocationSize(); cl++ ) {
			buf.append( StringUtil.replace( format.getFormat(cl) ,"td","th" ));
			final int loc = format.getLocation(cl);
			if( loc >= 0 ) { buf.append( getSortedColumnLabel(loc) ); }
		}
		buf.append( StringUtil.replace( format.getFormat(cl) ,"td","th" ) ).append( CR );

		headerLine = buf.toString();
		return headerLine;
	}

	/**
	 * フォーマットを設定します。
	 *
	 * @og.rev 3.5.4.0 (2003/11/25) 新規作成
	 * @param	list	TableFormatterのリスト
	 */
	@Override
	public void setFormatterList( final List<TableFormatter> list ) {		// 4.3.3.6 (2008/11/15) Generics警告対応
		format = list.get(0);		// 4.3.3.6 (2008/11/15) Generics警告対応
		format.makeFormat( getDBTableModel() );
	}

	/**
	 * フォーマットメソッドを使用できるかどうかを問い合わせます。
	 *
	 * @return  使用可能(true)/ 使用不可能 (false)
	 */
	@Override
	public boolean canUseFormat() {
		return true;
	}

	/**
	 * ビューで表示したカラムの一覧をカンマ区切りで返します。
	 *
	 * @og.rev 5.1.6.0 (2010/05/01) 新規追加
	 *
	 * @return	ビューで表示したカラムの一覧
	 */
	@Override
	public String getViewClms() {
		final DBTableModel table = getDBTableModel();
		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );
		for( int i=0; i<format.getLocationSize(); i++ ) {
			if( buf.length() > 0 ) { buf.append( ',' ); }
			buf.append( table.getColumnName( format.getLocation( i ) ) );
		}
		return buf.toString();
	}
}
