/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.db.AbstractDBType;
import org.opengion.hayabusa.db.DBTypeCheckUtil;

/**
 * 半角数字の NUMBER を扱う為の、カラム属性を定義します。
 *
 * '0' ～ '9' ,'-' ,'.',',' でのみ構成されている文字型カラム属性を定義します。
 * S9 と異なり、カンマ','が含まれていても OK とし、データからも取り除きません。
 *
 * タイプチェックとして、以下の条件を判定します。
 * ・０～９、マイナス(-)、小数点(.)およびカンマ(,)を許可
 * ・文字列の長さチェック
 * ・小数点の位置チェック
 * ・符号の位置チェック
 * ・文字パラメータの 正規表現チェック
 *
 * @og.group データ属性
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class DBType_X9 extends AbstractDBType {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "5.2.2.0 (2010/11/01)" ;

	/**
	 * 半角０文字の固定長でFILL埋めされた文字列を返します。
	 * なお、エラーチェックは行われません。
	 * 実行前に、必ず valueCheck( String value ,int len ) が行われる必要があります。
	 *
	 * @og.rev 3.5.4.5 (2004/01/23) エンコード指定に変更します。
	 *
	 * @param	value	 ＦＩＬＬ埋めする文字列
	 * @param	sizeX	 整数部分の文字列の長さ
	 * @param	sizeY	 少数部分の文字列の長さ
	 * @param	encode	 エンコード指定
	 *
	 * @return	ＦＩＬＬ埋めした新しい文字列
	 */
	@Override
	public String valueFill( final String value ,final int sizeX ,final int sizeY,final String encode ) {
		final int len = (sizeY == 0) ? sizeX : sizeX + sizeY + 1;

		// 注意 マイナス記号の処理がまだです。
		return StringUtil.intFill( value,len );
	}

	/**
	 * String引数の文字列を＋１した文字列を返します。
	 * これは、文字型数字タイプの場合は, マイナス、カンマ、ピリオドを避けて、
	 * +1 します。(桁上がりもあり)
	 * データの値が、マイナスの場合は、結果的にマイナスしていきます。
	 * カンマ編集(12,345,678 など)がある場合に、桁上がりしても、カンマは追加されません。
	 * マイナス時の桁上がりには、対応しています。
	 * 引数が null の場合と、ゼロ文字列("")の場合は,物理的初期設定値(String getDefault())
	 * の値を返します。
	 *
	 * @param	value	String引数の文字列
	 *
	 * @return	String引数の文字列を＋１した文字列
	 */
	@Override
	public String valueAdd( final String value ) {
		if( value == null || value.isEmpty() ) { return getDefault(); }
		char[] chs = value.toCharArray() ;
		boolean over = false;
		for( int i=chs.length-1; i>=0; i-- ) {
			if( chs[i] == '-' || chs[i] == '.' || chs[i] == ',' ) { continue; }
			if( chs[i] == '9' ) { chs[i] = '0'; over = true; }
			else { chs[i]++; over = false; break; }
		}
		if( over ) {
			char[] chs2 = new char[ chs.length + 1 ];
			System.arraycopy( chs,0,chs2,1,chs.length );
			if( chs[0] == '-' ) {
				chs2[0] = '-';
				chs2[1] = '1';
			}
			else {
				chs2[0] = '1';
			}
			chs = chs2;
		}
		return new String( chs );
	}

	/**
	 * エディターで編集されたデータを登録する場合に、データそのものを
	 * 変換して、実登録データを作成します。
	 * 例えば,大文字のみのフィールドなら、大文字化します。
	 * 実登録データの作成は、DBType オブジェクトを利用しますので,
	 * これと Editor とがアンマッチの場合は、うまくデータ変換
	 * されない可能性がありますので、注意願います。
	 *
	 * @og.rev 3.3.3.0 (2003/07/09) 前後のスペースを取り除いておく。
	 * @og.rev 3.3.3.1 (2003/07/18) 後ろスペースを取り除く。(StringUtil#rTrim)
	 *
	 * @param	value	(一般に編集データとして登録されたデータ)
	 *
	 * @return	修正後の文字列(一般にデータベースに登録するデータ)
	 */
	@Override
	public String valueSet( final String value ) {
		return StringUtil.rTrim( value ) ;
	}

	/**
	 * データが登録可能かどうかをチェックします。
	 * データがエラーの場合は、そのエラー内容を返します。
	 * 許可する文字は、0-9 の数字、マイナス(-) 、小数点(.)です。
	 * 小数点の位置チェックは行いません。
	 *
	 * @og.rev 2.3.1.4 (2003/02/18) 属性チェックを強化した。
	 * @og.rev 3.6.0.0 (2004/09/22) チェック項目は、数字を許可する文字列とする。
	 * @og.rev 3.6.0.0 (2004/09/22) dbType パラメータを引数に追加
	 * @og.rev 5.2.2.0 (2010/11/01) 厳密にチェック(isStrict=true)するフラグを追加
	 *
	 * @param   key		キー
	 * @param   value	値
	 * @param   sizeX	整数部分の文字列の長さ
	 * @param   sizeY	少数部分の文字列の長さ
	 * @param   typeParam	dbType パラメータ
	 * @param   isStrict	厳密にチェックするかどうか[true:する/false:標準的]
	 *
	 * @return  エラー内容
	 */
	@Override
	public ErrorMessage valueCheck( final String key ,final String value ,
									final int sizeX ,final int sizeY ,final String typeParam ,final boolean isStrict) {

		final ErrorMessage msg = new ErrorMessage();
		if( value == null || value.isEmpty() ) { return msg; }

		// チェック用のvalue を用意します。カンマ(,)を取り除いておく為です。
		final String tmpValue = StringUtil.deleteChar(value,',' );
		String check;

		// 3.6.0.0 (2004/09/22) チェック内容変更
		check = DBTypeCheckUtil.decimalFormatCheck( tmpValue );
		if( check != null ) {
			// 指定の文字以外の文字が使われています。
			msg.addMessage( 0,ErrorMessage.NG,"ERR0009", key,check );
		}

		check = DBTypeCheckUtil.decimalCodeCheck( tmpValue );
		if( check != null ) {
			// 符号の位置が不正です。
			msg.addMessage( 0, ErrorMessage.NG, "ERR0023", key, check );
		}

		// 3.6.0.0 (2004/09/22) 追加
		check = DBTypeCheckUtil.decimalPointCheck( tmpValue );
		if( check != null ) {
			// 小数点の位置が不正です。
			msg.addMessage( 0, ErrorMessage.NG, "ERR0024", key, check );
		}

		// 3.6.0.0 (2004/09/22) dbType パラメータを使用したマッチチェック
		check = DBTypeCheckUtil.matcheCheck( tmpValue,typeParam );
		if( check != null ) {
			// 指定の文字以外の文字が使われています。
			msg.addMessage( 0,ErrorMessage.NG,"ERR0009", key,check );
		}

		return msg;
	}
}
