/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import static org.opengion.fukurou.util.StringUtil.nval ;

/**
 * switch タグは、指定された条件を、case タグに伝えます。
 *
 * 各属性は、{&#064;XXXX} 変数が使用できます。
 * これは、ServletRequest から、XXXX をキーに値を取り出し,この変数に
 * 割り当てます。つまり、このXXXXをキーにリクエストすれば、
 * この変数に値をセットすることができます。
 *
 * @og.formSample
 * ●形式：&lt;og:switch key="･･･" /&gt;
 *            &lt;og:case match="A" /&gt; ･･･ &lt;/og:case&gt;
 *            &lt;og:case match="B" /&gt; ･･･ &lt;/og:case&gt;
 *            &lt;og:case match="C" /&gt; ･･･ &lt;/og:case&gt;
 *            &lt;og:case isDefault="true" /&gt; ･･･ &lt;/og:case&gt;
 *         &lt;/og:switch&gt;
 * ●body：あり(EVAL_BODY_INCLUDE:BODYをインクルードし、{&#064;XXXX} は解析しません)
 *
 * ●Tag定義：
 *   &lt;og:switch
 *       key              ○【TAG】switch のマッチ判定用のキーを設定します(必須)。
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   &gt;   ... Body ...
 *   &lt;/og:switch&gt;
 *
 * ●使用例
 *         &lt;og:switch key="{&#064;PARAM}" /&gt;
 *            &lt;og:case match="A" /&gt; 処理A &lt;/og:case&gt;
 *            &lt;og:case match="B" /&gt; 処理B &lt;/og:case&gt;
 *            &lt;og:case match="C" /&gt; 処理C &lt;/og:case&gt;
 *            &lt;og:case isDefault="true" /&gt; 処理X &lt;/og:case&gt;
 *         &lt;/og:switch&gt;
 *
 *          ・switch の key に対して、case の match に指定された値が、マッチ(switch_key.match( case_match ))
 *            した場合に、case の BODY 部分が処理されます。
 *            マッチしなければ、BODY部は、スキップされます。
 *          ・isDefault="true" の場合は、どれとも マッチしなかった場合に、実行されます。
 *          ・Javaの switch-case 文は、最初に処理された case 以降を処理します。通常は、break を入れて
 *            後続処理を実行されないようにしています。
 *            この、switch-case タグは、caseタグの isBreak 属性で制御します。初期値が isBreak="true" に、
 *            なっているため、通常は、どれかの case が実行された段階で、switchの処理は、終了されます。
 *            isBreak="false" にすると、switchから抜けずに、継続して case との match を実行します。
 *            この場合、Java等と異なるのは、直後のcase文が実行されるのではなく、あくまで match 作業が
 *            継続されるということです。つまり、複数の case で処理を行いたい場合は、isBreak="false" に
 *            すると同時に、match 条件もそれぞれで、マッチするように設定する必要があります。
 *
 *         &lt;og:switch key="{&#064;PARAM}" /&gt;
 *            &lt;og:case match="[1]"   isBreak="false" /&gt; 処理A &lt;/og:case&gt;
 *            &lt;og:case match="[12]"  isBreak="false" /&gt; 処理B &lt;/og:case&gt;
 *            &lt;og:case match="[123]" isBreak="false" /&gt; 処理C &lt;/og:case&gt;
 *            &lt;og:case isNull="true" /&gt; 処理X &lt;/og:case&gt;
 *            &lt;og:case isDefault="true" /&gt; 処理Y &lt;/og:case&gt;
 *         &lt;/og:switch&gt;
 *
 *          ・上記指定では、isBreak="false" が指定されているため、マッチした後も継続して判定処理が実施されます。
 *          ・上記例で言うと、PARAM が "1" の場合、上記３つともにマッチします。
 *          ・isNull="true" は、switch の key が null の場合に成立します。(null とは、ゼロ文字列も含む)
 *
 * @og.group 画面制御
 * @og.rev 5.2.3.0 (2010/12/01) 新規追加
 *
 * @version  5.2.3.0 (2010/12/01)
 * @author	 Kazuhiko Hasegawa
 * @since    JDK1.6,
 */
public class SwitchTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "5.2.3.0 (2010/12/01)" ;

	private static final long serialVersionUID = 523020101201L ;

	private String	switchKey	;
	private boolean	useMatch	= true;		// マッチ処理を継続して行う場合、true	(5.3.0.0 (2010/12/01) 変数名変更)

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @return	後続処理の指示( EVAL_BODY_INCLUDE )
	 */
	@Override
	public int doStartTag() {
		useMatch	= true;	// 初期化

		return EVAL_BODY_INCLUDE ;	// Body インクルード( extends TagSupport 時)
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();

		return EVAL_PAGE ;
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
		switchKey	= null;
		useMatch	= true;
	}

	/**
	 * 【TAG】switch のマッチ判定用のキーを設定します。
	 *
	 * @og.tag switch のマッチ判定用のキーを設定します。
	 *
	 * @param	key マッチ判定用のキー
	 * @see		#getKey()
	 */
	public void setKey( final String key ) {
		switchKey = nval( getRequestParameter( key ),switchKey );
	}

	/**
	 * switch のマッチ判定用のキーを取得します。
	 *
	 * case タグで、この値を取り出して、マッチ判定を行います。
	 *
	 * @return	マッチ判定用のキー
	 * @see		#setKey( String )
	 */
	protected String getKey() {
		return switchKey;
	}

	/**
	 * case タグが、ブレイクした場合に、このメソッドを呼び出します。
	 *
	 * これは、 case タグが isBreak="true" でマッチした場合、このメソッドを
	 * 呼び出し、isMatch フラグを false に設定します。
	 * 他の case は、このフラグを参照して、false であれば、スルーします。
	 *
	 * @see		#isMatch()
	 */
	protected void setBreak() {
		useMatch= false ;
	}

	/**
	 * すでにマッチしたかどうかを返します。
	 *
	 * これは、 case タグが 処理を継続するかどうかの判定に利用します。
	 * case タグが isBreak="true" でマッチした場合、isMatch フラグは、
	 * false が返りますので、継続処理しません。
	 *
	 * @return	マッチしたかどうか[true:継続判定/false:スルー]
	 * @see		#setBreak()
	 */
	protected boolean isMatch() {
		return useMatch ;
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	@Override
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
				.println( "VERSION"			,VERSION	)
				.println( "switchKey"		,switchKey	)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
